<?php
namespace Smply\SGUtilities\ViewHelpers\Google;

/**
 * ViewHelper that prints an image with a static google map
 * with a specified center and optional markers
 *
 * == Example ==
 *
 * Add Google marker icon to constants, if you wish to have a custom icon.
 * Add Google API key to constants or pass API key with $key param.
 *
 * {namespace sg=Smply\SGUtilities\ViewHelpers}
 *
 * <sg:google.staticmap
 *    class="img-responsive"
 *    center="{realty.objektBreitengrad},{realty.objektLaengengrad}"
 *    markers="{site:google.realtytomarker(realties:{0:realty})}"
 *    zoom="15"
 *    height="200"
 *    width="350"
 *    alt="Umgebungskarte für die Mietwohnung {realty.objektStrasse} {realty.objektHausnummer} in {realty.objektOrt.bezeichner} {realty.objektStadtteil.bezeichner}">
 * </site:google.staticmap>
 *
 * @package Smply\SGUtilities\ViewHelpers\Google
 */
class StaticmapViewHelper  extends \TYPO3\CMS\Fluid\Core\ViewHelper\AbstractTagBasedViewHelper
{

  /**
   * @var string
   */
  protected $tagName = 'img';

  /**
   * Initialize arguments
   *
   * @return void
   * @api
   */
  public function initializeArguments()
  {
    $this->registerUniversalTagAttributes();
  }

  /**
   * Renders image tag with google static maps image url
   *
   * @param string $center      Specifies the static map center
   * @param int    $zoom        Specifies the zoom level of the static map
   * @param array  $markers     Specifies the markers
   * @param string $markerIcon  Specifies the path to marker icon
   *                            Expects array w/ fields:
   *                            location (required), color, label, size, icon (if not present default icon)
   *                            array('location' => '', ... )
   * @param string $alt         Specifies the alt-attribute of the static map image
   * @param int    $height      Specifies the height of the image
   * @param int    $width       Specifies the width of the image
   * @param int    $scale       Specifies the resolution factor
   * @param string $key         Specifies the API key. Overrides extension settings key value if present
   *
   * @return string
   * @throws \TYPO3\CMS\Fluid\Core\ViewHelper\Exception
   * @throws \TYPO3\CMS\Fluid\Core\ViewHelper\Exception\InvalidVariableException
   */
  public function render(
    $center = '51.38063,6.9347213',
    $zoom = 13,
    $markers = array(),
    $markerIcon = '',
    $alt = '',
    $height = 480,
    $width = 640,
    $scale = 1,
    $key = '' )
  {
    // retrieve settings and get google api key and marker icon.
    // var_dump( $markers );
    $settings = $this->templateVariableContainer->get( 'settings' );
    $apiKey = false;
    $staticmap = 'https://maps.googleapis.com/maps/api/staticmap?';

    if( $settings  && is_array( $settings ) )
    {
      // retrieve google api key
      if( isset( $settings['keys'] ) &&
          isset( $settings['keys']['googleapi'] ) &&
          $settings['keys']['googleapi'] !== false &&
          $settings['keys']['googleapi'] != '' )
      {
        $apiKey = $settings['keys']['googleapi'];
      }
      // take inline marker icon, if specified. if not take constants marker icon.
      if( $markerIcon === '' )
      {
        if( isset( $settings['statics'] ) &&
            isset( $settings['statics']['googleMapsMarkerIcon'] ) &&
            $settings['statics']['googleMapsMarkerIcon'] !== false &&
            $settings['statics']['googleMapsMarkerIcon'] != '' )
        {
          $markerIcon = $settings['statics']['googleMapsMarkerIcon'];
        }
      }
    }
    // override settings api key if key param var is present
    if( $key !== '' )
    {
      $apiKey = $key;
    }
    if( $apiKey === false )
    {
      throw new \TYPO3\CMS\Fluid\Core\ViewHelper\Exception('Smply\SGUtilities\ViewHelpers\Google\Staticmap You must specify a Google API key.');
    }

    // add center
    $staticmap .= "center={$center}";
    // add zoom
    $staticmap .= "&zoom={$zoom}";
    // add height and width
    $staticmap .= "&size={$width}x{$height}";
    $staticmap .= "&scale={$scale}";
    // add markers and icon
    if( $markers && is_array( $markers ) && count( $markers ) > 0 )
    {
      $markerGlue = '%7C';
      foreach( $markers as $marker )
      {
        // location must be available
        if( isset( $marker['location'] ) && $marker['location'] && $marker['location'] != '' )
        {
          $markerText = '&markers=';
          if( isset( $marker['color'] ) && $marker['color'] && $marker['color'] != '' )
          {
            $markerText .= "color:{$marker['color']}{$markerGlue}";
          }
          if( isset( $marker['label'] ) && $marker['label'] && $marker['label'] != '' )
          {
            $markerText .= "label:".rawurldecode( $marker['label'] )."{$markerGlue}";
          }
          if( isset( $marker['size'] ) && $marker['size'] && $marker['size'] != '' )
          {
            $markerText .= "size:{$marker['size']}{$markerGlue}";
          }
          // use specified icon or default
          if( isset( $marker['icon'] ) && $marker['icon'] && $marker['icon'] != '' )
          {
            $markerText .= "icon:{$marker['size']}{$markerGlue}";
          }
          else if ( $markerIcon != '' )
          {
            $markerText .= "icon:".rawurldecode( $markerIcon )."{$markerGlue}";
          }
          $markerText .= $marker['location'];
          $staticmap .= $markerText;
        }
      }
    }
    // add api key
    $staticmap .= "&key={$apiKey}";


    $this->tag->addAttribute('src', $staticmap );
    $this->tag->addAttribute('alt', $alt );
    $this->tag->forceClosingTag(false);
    return $this->tag->render();

  }

}