<?php
namespace Deployer;

$composerHome = getenv("COMPOSER_HOME") ?: (getenv("HOME") . '/.composer/');
if (is_dir($composerHome)) {
    include $composerHome . '/vendor/autoload.php';
} else {
    $xdgComposerHome = getenv('XDG_CONFIG_HOME') ?: (getenv("HOME") . '/.config');
    $xdgComposerHome = $xdgComposerHome . '/composer';
    if (is_dir($xdgComposerHome)) {
        include $xdgComposerHome . '/vendor/autoload.php';
    }
}

require 'recipe/typo3.php';
require 'recipe/rsync.php';

/**
 * Create symlinks for shared directories and files.
 *
 * This is a copy of the default `deploy:shared` task, but modified to create relative symlinks. See
 * `deploy:symlink` for details.
 */
task( 'deploy:shared', function() {
    $sharedPath = "{{deploy_path}}/shared";

    foreach (get( 'shared_dirs' ) as $dir) {
        // Remove from source.
        run( "if [ -d $(echo {{release_path}}/$dir) ]; then rm -rf {{release_path}}/$dir; fi" );

        // Create shared dir if it does not exist.
        run( "mkdir -p $sharedPath/$dir" );

        // Create path to shared dir in release dir if it does not exist.
        // (symlink will not create the path and will fail otherwise)
        run( "mkdir -p `dirname {{release_path}}/$dir`" );

        // Symlink shared dir to release dir
        $change_directory = 'cd {{release_path}}/' . dirname( $dir );
        $parent_levels    = str_repeat( '../', substr_count( $dir, '/' ) + 2 );
        $create_symlink   = "ln -nfs {$parent_levels}shared/$dir " . basename( $dir );

        run( "$change_directory && $create_symlink" );
    }

    foreach (get( 'shared_files' ) as $file) {
        $dirname = dirname( $file );
        // Remove from source.
        run( "if [ -f $(echo {{release_path}}/$file) ]; then rm -rf {{release_path}}/$file; fi" );
        // Ensure dir is available in release
        run( "if [ ! -d $(echo {{release_path}}/$dirname) ]; then mkdir -p {{release_path}}/$dirname;fi" );

        // Create dir of shared file
        run( "mkdir -p $sharedPath/" . $dirname );

        // Touch shared
        run( "touch $sharedPath/$file" );

        // Symlink shared dir to release dir
        $change_directory = 'cd {{release_path}}/' . dirname( $file );
        $parent_levels    = str_repeat( '../', substr_count( $file, '/' ) + 2 );
        $create_symlink   = "ln -sfn {$parent_levels}shared/$file " . basename( $file );

        run( "$change_directory && $create_symlink" );
    }
} )->desc( 'Creating symlinks for shared files' );

/**
 * Create symlink to last release.
 *
 * This is a copy of the default `deploy:symlink` task, but modified to create relative symlinks.
 *
 * See https://github.com/deployphp/deployer/issues/503
 *
 * On many shared hosting platforms, the user is chroot'd into a child directory, so absolute symlinks
 * fail because they require permission to execute on each folder in the path, all the way up to /
 *  - https://unix.stackexchange.com/a/21339/18886
 *  - https://gist.github.com/mkrisher/74721/revisions#diff-01cf30e04b9c4489ed9e927b99e3b66dR45
 *
 * Also, the symlinks need to be set relative to the symlink's target directory.
 *  - https://unix.stackexchange.com/a/15285/18886
 *  - https://askubuntu.com/a/524635
 */
task( 'deploy:symlink', function() {
    $relative_release_path = '.' . substr(
            get( 'release_path' ),
            strlen( get( 'deploy_path' ) )
        );
    set( 'relative_release_path', $relative_release_path );

    run( "cd {{deploy_path}} && ln -sfn {{relative_release_path}} current" ); // Atomic override symlink.
    run( "cd {{deploy_path}} && rm release" ); // Remove release link.
} )->desc( 'Creating symlink to release' );

/**
 * Rollback to previous release.
 *
 * This is a copy of the default `rollback` task, but modified to create relative symlinks. See
 * `deploy:symlink` for details.
 */
task( 'rollback', function() {
    $releases = get( 'releases_list' );

    if (isset($releases[1])) {
        $releaseDir = "./releases/{$releases[1]}";

        // Symlink to old release.
        run( "cd {{deploy_path}} && ln -nfs $releaseDir current" );

        // Remove release
        run( "rm -rf ./releases/{$releases[0]}" );

        writeln( "Rollback to `{$releases[1]}` release was successful." );
    } else {
        writeln( "<comment>No more releases you can revert to.</comment>" );
    }
} )->desc( 'Rollback to previous release' );

/**
 * Load Hosts from Inventory
 */
inventory('hosts.yml');

/**
 * DocumentRoot / WebRoot for the TYPO3 installation
 */
set('typo3_webroot', 'public');

/**
 * Shared directories
 */
set('shared_dirs', [
    '{{typo3_webroot}}/fileadmin',
    '{{typo3_webroot}}/typo3temp',
    '{{typo3_webroot}}/uploads',
    'var',
    'config'
]);
/**
 * Shared files
 */
set('shared_files', [
    '.env',
    '{{typo3_webroot}}/.htaccess'
]);

/**
 * rsync Settings
 */
set('rsync',[
    'exclude'      => [
        '.git',
        '.npm',
        '.composer',
        '.ddev',
        '_env',
        'deploy.php',
        '.gitignore',
        '.gitlab-ci.yml',
        'README.md',
        'package.json',
        'package-lock.json',
        'hosts.yml',
        '_hosts.yml',
        'public/typo3conf/ext/sg_sitepackage/Resources/Public/Css',
        'public/typo3conf/ext/sg_sitepackage/Resources/Public/JavaScript',
        'public/typo3conf/ext/sg_sitepackage/Resources/Public/node_modules',
        'public/typo3conf/ext/sg_sitepackage/Resources/Public/gulpfile.js',
        'public/typo3conf/ext/sg_sitepackage/Resources/Public/.npmrc',
    ],
    'exclude-file' => false,
    'include'      => [],
    'include-file' => false,
    'filter'       => [],
    'filter-file'  => false,
    'filter-perdir'=> false,
    'flags'        => 'rltz', // Recursive, with compress
    'options'      => ['delete'],
    'timeout'      => 60,
]);
set('rsync_src', __DIR__. "/");
set('rsync_dest','{{release_path}}');

task('upload', function () {
    upload(__DIR__ . "/", '{{release_path}}');
});

task('typo3cms:generatepackagestate', function (){
    run("cd {{release_path}} && vendor/bin/typo3cms install:generatepackagestates");
});
task('typo3cms:extensionsetupifpossible', function (){
    run("cd {{release_path}} && vendor/bin/typo3cms install:extensionsetupifpossible");
});

// Tasks
task('deploy', [
    'deploy:info',
    'deploy:prepare',
    'deploy:lock',
    'deploy:release',
    'rsync:warmup',
    'rsync',
    'deploy:shared',
    'deploy:vendors',
    'typo3cms:generatepackagestate',
    'typo3cms:extensionsetupifpossible',
    'deploy:symlink',
    'deploy:unlock',
    'cleanup',
    'success'
]);

// [Optional] if deploy fails automatically unlock.
after('deploy:failed', 'deploy:unlock');
