<?php
namespace SG\SgImmoservice\Tasks;

use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Scheduler\Task\AbstractTask;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Database\Connection;
use TYPO3\CMS\Extbase\Object\ObjectManager;
use SG\SgImmoservice\Domain\Repository\CompetenceTypeRepository;
use SG\SgImmoservice\Domain\Repository\ResidentialUnitRepository;
use SG\SgImmoservice\Domain\Repository\ContactPersonRepository;
use SG\SgImmoservice\Domain\Repository\CompetentAuthorityRepository;
use RuntimeException;

class Import extends AbstractTask
{
    const RESPONSE_ERROR = 'error';
    const RESPONSE_ALERT = 'alert';
    const RESPONSE_INFORMATION = 'information';

    const FILENAME_RESIDENTIALUNITS = 'residentialUnit.csv';
    const FILENAME_COMPETENTAUTHORITY = 'competentAuthority.csv';
    const FILENAME_CONTACTPERSON = 'contactPerson.csv';
    const FILENAME_CONNECTIONSCOMPETENTAUTHORITY = 'residentialUnitcompetentAuthority.csv';
    const FILENAME_LOGFILE = 'log.txt';

    /**
     * @var int
     */
    protected $storagePid = 0;

    /**
     * @var bool
     */
    protected $debugMode = false;

    /**
     * @var string
     */
    protected $importFile = '';

    /**
     * @var string
     */
    private $pathTemp = '';

    /**
     * @var string
     */
    private $pathArchive = '';

    /**
     * @var string
     */
    private $pathUpload = '';

    /**
     * @var array
     */
    private $runtimeErrors = array();

    /**
     * @var ObjectManager
     */
    protected $objectManager;

    public function execute(){
        if ($this->storagePid == 0){
            throw new RuntimeException('No StoragePid set!', 4711);
        }

        $this->initialize();

        if ($this->preProcess()){
            $cache_manager = GeneralUtility::makeInstance(CacheManager::class);
            $cache_manager->flushCaches();
            $this->import();
        }

        $this->postProcess();

        if (count($this->runtimeErrors) > 0){
            throw new \RuntimeException(implode(', ', $this->runtimeErrors), 4711);
        }else{
            return true;
        }
    }

    /**
     * Initializes Standardvalues
     * e.g. Path & Filenames
     */
    private function initialize(){
        $this->objectManager = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Object\ObjectManager');

        $this->pathTemp         = PATH_site . 'uploads/tx_sgimmoservice/temp/';
        $this->pathArchive      = PATH_site . 'uploads/tx_sgimmoservice/archive/';
        $this->pathUpload       = PATH_site . 'uploads/tx_sgimmoservice/import/';
    }

    /**
     * Pre Import Processes
     * - Search for uploaded File
     * - Extract Files in Temp Folder
     * - Move uploaded File to Archive Folder
     */
    private function preProcess()
    {

        if ($this->getImportFile() == 'all'){
            $returnArray = array();
            if (is_file($this->pathUpload.self::FILENAME_RESIDENTIALUNITS)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_RESIDENTIALUNITS, $this->pathTemp.self::FILENAME_RESIDENTIALUNITS);
                $returnArray[] = true;
            }else{
                $this->runtimeErrors[] = self::FILENAME_RESIDENTIALUNITS . ' is missing';
                $returnArray[] = false;
            }

            if (is_file($this->pathUpload.self::FILENAME_COMPETENTAUTHORITY)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_COMPETENTAUTHORITY, $this->pathTemp.self::FILENAME_COMPETENTAUTHORITY);
                $returnArray[] = true;
            }else{
                $this->runtimeErrors[] = self::FILENAME_COMPETENTAUTHORITY . ' is missing';
                $returnArray[] = false;
            }

            if (is_file($this->pathUpload.self::FILENAME_CONTACTPERSON)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_CONTACTPERSON, $this->pathTemp.self::FILENAME_CONTACTPERSON);
                $returnArray[] = true;
            }else{
                $this->runtimeErrors[] = self::FILENAME_CONTACTPERSON . ' is missing';
                $returnArray[] = false;
            }

            if (is_file($this->pathUpload.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY, $this->pathTemp.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY);
                $returnArray[] = true;
            }else{
                $this->runtimeErrors[] = self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY . ' is missing';
                $returnArray[] = false;
            }
            if (in_array(false, $returnArray)){
                return false;
            }else{
                return true;
            }
        }

        $return = false;
        if ($this->getImportFile() == 'competentAuthority'){
            if (is_file($this->pathUpload.self::FILENAME_COMPETENTAUTHORITY)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_COMPETENTAUTHORITY, $this->pathTemp.self::FILENAME_COMPETENTAUTHORITY);
                $return = true;
            }
        }

        if ($this->getImportFile() == 'contactPerson'){
            if (is_file($this->pathUpload.self::FILENAME_CONTACTPERSON)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_CONTACTPERSON, $this->pathTemp.self::FILENAME_CONTACTPERSON);
                $return = true;
            }
        }

        if ($this->getImportFile() == 'residentialUnit'){
            if (is_file($this->pathUpload.self::FILENAME_RESIDENTIALUNITS)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_RESIDENTIALUNITS, $this->pathTemp.self::FILENAME_RESIDENTIALUNITS);
                $return = true;
            }
        }

        if ($this->getImportFile() == 'residentialUnitcompetentAuthority'){
            if (is_file($this->pathUpload.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY)){
                $this->moveImportFile($this->pathUpload.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY, $this->pathTemp.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY);
                $return = true;
            }
        }
        return $return;
    }

    private function import(){
        $this->importResidentialUnits();
        $this->importCompetentAuthority();
        $this->importContactsAndCompetenceType();
        $this->importConnectionsCompetentAuthority();
    }

    /**
     * Imports Residential Units from CSV File
     */
    private function importResidentialUnits(){
        if (is_file($this->pathTemp.self::FILENAME_RESIDENTIALUNITS)){
            $logfile = fopen($this->pathArchive.self::FILENAME_LOGFILE,'a+');
            fwrite($logfile, 'ResidentialUnits START: '. date("Y-m-d H:i:s") . "\r\n");
            if (($handle = fopen($this->pathTemp.self::FILENAME_RESIDENTIALUNITS, "r")) !== FALSE) {
                /**
                 * @var $connection \TYPO3\CMS\Core\Database\Connection
                 */

                $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_residentialunit');

                $serviceUnitRepository = $this->objectManager->get('SG\SgImmoservice\Domain\Repository\ServiceUnitRepository');
                $serviceUnits = $serviceUnitRepository->getImportMatrix();

                $row = 1;

                // Remove all Residential Units
                $connection->truncate('tx_sgimmoservice_domain_model_residentialunit');

                $insertedExternalIds = array();
                $arrayToImport = array();
                while (($data = fgetcsv($handle, 0, ";")) !== FALSE) {
                    if ($row != 1){
                        $externalId = utf8_encode($data[1].'.'.$data[2]);
                        if ($externalId != '.' && !in_array($externalId, $insertedExternalIds) && isset($serviceUnits[utf8_encode($data[8])])){
                            $arrayToImport[] = array((int)$this->storagePid, (string)$externalId, (string)utf8_encode($data[4]), (string)utf8_encode($data[5]), (string)utf8_encode($data[6]), (string)utf8_encode($data[7]), (string)$serviceUnits[utf8_encode($data[8])]);
                            array_push($insertedExternalIds, $externalId);
                        }
                    }else{
                        $row++;
                    }
                }
                fclose($handle);
                if (count($arrayToImport) > 0){
                    $connection->bulkInsert(
                        'tx_sgimmoservice_domain_model_residentialunit',
                        $arrayToImport,
                        [
                            'pid',
                            'external_id',
                            'street',
                            'zip',
                            'city',
                            'district',
                            'service_unit'
                        ],
                        [
                            Connection::PARAM_INT,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                        ]
                    );
                }
            }
            fwrite($logfile, 'ResidentialUnits END: '. date("Y-m-d H:i:s") . "\r\n");
            fclose($logfile);
        }
    }

    /**
     * Imports Competent Authorities from CSV File
     */
    private function importCompetentAuthority(){
        if (is_file($this->pathTemp.self::FILENAME_COMPETENTAUTHORITY)){
            $logfile = fopen($this->pathArchive.self::FILENAME_LOGFILE,'a+');
            fwrite($logfile, 'CompetentAuthority START: '. date("Y-m-d H:i:s") . "\r\n");
            if (($handle = fopen($this->pathTemp.self::FILENAME_COMPETENTAUTHORITY, "r")) !== FALSE) {
                /**
                 * @var $connection \TYPO3\CMS\Core\Database\Connection
                 */
                $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_competentauthority');

                $row = 1;

                // Remove all Residential Units
                $connection->truncate('tx_sgimmoservice_domain_model_competentauthority');

                $insertedExternalIds = array();
                $arrayToImport = array();
                while (($data = fgetcsv($handle, 0, ";")) !== FALSE) {
                    if ($row != 1){
                        $externalId = utf8_encode($data[0]);
                        if (!in_array($externalId, $insertedExternalIds)){
                            $arrayToImport[] = array(
                                (int)$this->storagePid, // pid
                                (string)$externalId, // external_id
                                (string)utf8_encode($data[2]), // title
                                (string)utf8_encode($data[3]), // street
                                (string)utf8_encode($data[4]), // zip
                                (string)utf8_encode($data[5]), // city
                                (string)utf8_encode($data[6]), // phone
                                (string)utf8_encode($data[7]), // fax
                                (string)utf8_encode($data[8]), // email
                                (string)utf8_encode($data[9]), // url
                                (string)utf8_encode($data[10]), // additional_information
                                (string)utf8_encode($data[11]) // business_hours
                            );
                            array_push($insertedExternalIds, $externalId);
                        }
                    }else{
                        $row++;
                    }

                }
                fclose($handle);
                if (count($arrayToImport) > 0){
                    $connection->bulkInsert(
                        'tx_sgimmoservice_domain_model_competentauthority',
                        $arrayToImport,
                        [
                            'pid',
                            'external_id',
                            'title',
                            'street',
                            'zip',
                            'city',
                            'phone',
                            'fax',
                            'email',
                            'url',
                            'additional_information',
                            'business_hours'
                        ],
                        [
                            Connection::PARAM_INT,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR,
                            Connection::PARAM_STR
                        ]
                    );
                }
            }
            fwrite($logfile, 'CompetentAuthority END: '. date("Y-m-d H:i:s") . "\r\n");
            fclose($logfile);
        }
    }

    /**
     * Imports Competence Types and Contact Persons from CSV File
     */
    private function importContactsAndCompetenceType(){
        if (is_file($this->pathTemp.self::FILENAME_CONTACTPERSON)){
            $logfile = fopen($this->pathArchive.self::FILENAME_LOGFILE,'a+');
            fwrite($logfile, 'CompetenceContactsResponsibilityType START: '. date("Y-m-d H:i:s") . "\r\n");

            if (($handle = fopen($this->pathTemp.self::FILENAME_CONTACTPERSON, "r")) !== FALSE) {
                /**
                 * @var $connection \TYPO3\CMS\Core\Database\Connection
                 * @var $competenceTypeRepository CompetenceTypeRepository
                 */

                $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_competencetype');
                $connection2 = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_contactperson');
                $connection3 = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_responsibilitycontactperson');

                // Remove all Contact Person
                $connection2->truncate('tx_sgimmoservice_domain_model_contactperson');
                $connection3->truncate('tx_sgimmoservice_domain_model_responsibilitycontactperson');

                $competenceTypeRepository = $this->objectManager->get('SG\SgImmoservice\Domain\Repository\CompetenceTypeRepository');
                $residentialUnitRepository = $this->objectManager->get('SG\SgImmoservice\Domain\Repository\ResidentialUnitRepository');

                $competenceTypesImportMatrix = $competenceTypeRepository->getImportMatrix();
                $residentialUnitsImportMatrix = $residentialUnitRepository->getImportMatrix();

                $row = 1;

                $contactPersonsImportMatrix = [];
                $arrayToImportResponsibilityContactPerson = array();

                while (($data = fgetcsv($handle, 0, ";")) !== FALSE) {
                    if ($row != 1){
                        //get new Competence Types once
                        $externalCompetenceId = utf8_encode($data[7]);

                        if (!isset($competenceTypesImportMatrix[$externalCompetenceId])){
                            $connection->insert(
                                'tx_sgimmoservice_domain_model_competencetype',
                                [
                                    'pid' => (int)$this->storagePid,
                                    'external_id' => (string)$externalCompetenceId,
                                    'title' => (string)utf8_encode($data[7])
                                ],
                                [
                                    Connection::PARAM_INT,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR,
                                ]
                            );
                            $competenceTypesImportMatrix[$externalCompetenceId] = $connection->lastInsertId();
                        }

                        //get unique Contacts

                        $externalContactId = utf8_encode($data[8]);
                        if (!isset($contactPersonsImportMatrix[$externalContactId])){
                            $connection2->insert(
                                'tx_sgimmoservice_domain_model_contactperson',
                                [
                                    'pid' => (int)$this->storagePid,
                                    'external_id' => (string)$externalContactId,
                                    'lastname' => (string)utf8_encode($data[9]),
                                    'firstname' => (string)utf8_encode($data[10]),
                                    'phone' => (string)utf8_encode($data[11]),
                                    'fax' => (string)utf8_encode($data[12]),
                                    'email' => (string)utf8_encode($data[13]),
                                ],
                                [
                                    Connection::PARAM_INT,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR,
                                    Connection::PARAM_STR
                                ]
                            );
                            $contactPersonsImportMatrix[$externalContactId] = $connection2->lastInsertId();
                        }

                        //get responsibilities
                        if (
                        isset($residentialUnitsImportMatrix[utf8_encode($data[1].'.'.$data[2])])
                        ){
                            $arrayToImportResponsibilityContactPerson[] = array(
                                'pid' => (int)$this->storagePid,
                                'residential_unit' => (int)$residentialUnitsImportMatrix[utf8_encode($data[1].'.'.$data[2])],
                                'competence_type' => (int)$competenceTypesImportMatrix[utf8_encode($data[7])],
                                'contact_person' => (int)$contactPersonsImportMatrix[utf8_encode($data[8])]
                            );
                        }
                    }else{
                        $row++;
                    }

                }

                fclose($handle);

                if (count($arrayToImportResponsibilityContactPerson) > 0){
                    foreach ($arrayToImportResponsibilityContactPerson as $row){
                        $connection3->insert(
                            'tx_sgimmoservice_domain_model_responsibilitycontactperson',
                            $row,
                            [
                                Connection::PARAM_INT,
                                Connection::PARAM_INT,
                                Connection::PARAM_INT,
                                Connection::PARAM_INT
                            ]
                        );
                    }
                }
            }
            fwrite($logfile, 'CompetenceContactsResponsibilityType END: '. date("Y-m-d H:i:s") . "\r\n");
            fclose($logfile);
        }
    }

    /**
     * Import Connections from Competent Authority to Residential Units from CSV File
     */
    private function importConnectionsCompetentAuthority(){
        if (is_file($this->pathTemp.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY)){
            $logfile = fopen($this->pathArchive.self::FILENAME_LOGFILE,'a+');
            fwrite($logfile, 'ConnectionsCompetentAuthority START: '. date("Y-m-d H:i:s") . "\r\n");
            if (($handle = fopen($this->pathTemp.self::FILENAME_CONNECTIONSCOMPETENTAUTHORITY, "r")) !== FALSE) {
                /**
                 * @var $connection \TYPO3\CMS\Core\Database\Connection
                 * @var $residentialUnitRepository ResidentialUnitRepository
                 * @var $competentAuthorityRepository CompetentAuthorityRepository
                 */

                $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionForTable('tx_sgimmoservice_domain_model_responsibilitycontactperson');
                $residentialUnitRepository = $this->objectManager->get('SG\SgImmoservice\Domain\Repository\ResidentialUnitRepository');
                $competentAuthorityRepository = $this->objectManager->get('SG\SgImmoservice\Domain\Repository\CompetentAuthorityRepository');

                $row = 1;

                // Delete all Connections
                $connection->truncate('tx_sgimmoservice_domain_model_responsibilitycompetentauthority');

                $residentialUnitsImportMatrix = $residentialUnitRepository->getImportMatrix();
                $competentAuthoritiesImportMatrix = $competentAuthorityRepository->getImportMatrix();

                $arrayToImport = array();
                while (($data = fgetcsv($handle, 0, ";")) !== FALSE) {
                    if ($row != 1){
                        // Get Residential Unit

                        if (isset($residentialUnitsImportMatrix[utf8_encode($data[1].'.'.$data[2])])){
                            $baseArray = array(
                                'pid'                   => (int)$this->storagePid,
                                'residential_unit'      => (int)$residentialUnitsImportMatrix[utf8_encode($data[1].'.'.$data[2])]
                            );

                            // "Regie-Sanitär" (10)
                            if ($data[6] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[6])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 10,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[6])]
                                    ));
                                }
                            }

                            // "Sanitär" (11)
                            if ($data[7] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[7])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 11,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[7])]
                                    ));
                                }
                            }

                            // Elektroninstallation (21)
                            if ($data[8] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[8])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 21,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[8])]
                                    ));
                                }
                            }

                            // Regie Tischler (30)
                            if ($data[9] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[9])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 30,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[9])]
                                    ));
                                }
                            }

                            // Tischler (31)
                            if ($data[10] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[10])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 31,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[10])]
                                    ));
                                }
                            }

                            // Regie Rollladen (40)
                            if ($data[11] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[11])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 40,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[11])]
                                    ));
                                }
                            }

                            // Rollladen (41)
                            if ($data[12] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[12])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 41,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[12])]
                                    ));
                                }
                            }

                            // Schlosser (51)
                            if ($data[13] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[13])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 51,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[13])]
                                    ));
                                }
                            }

                            // Heizung (61)
                            if ($data[14] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[14])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 61,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[14])]
                                    ));
                                }
                            }

                            // Thermen (71)
                            if ($data[15] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[15])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 71,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[15])]
                                    ));
                                }
                            }

                            // Rohrreinigung (81)
                            if ($data[16] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[16])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 81,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[16])]
                                    ));
                                }
                            }

                            // Regenwasseraufbereitung (91)
                            if ($data[17] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[17])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 91,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[17])]
                                    ));
                                }
                            }

                            // Aufzug (101)
                            if ($data[18] != ''){
                                if (isset($competentAuthoritiesImportMatrix[utf8_encode($data[18])])){
                                    $arrayToImport[] = array_merge($baseArray, array(
                                        'type'                  => 101,
                                        'competent_authority'   => (int)$competentAuthoritiesImportMatrix[utf8_encode($data[18])]
                                    ));
                                }
                            }
                        }
                    }else{
                        $row++;
                    }
                }
                fclose($handle);
                if (count($arrayToImport) > 0){
                    foreach ($arrayToImport as $row){
                        $connection->insert(
                            'tx_sgimmoservice_domain_model_responsibilitycompetentauthority',
                            $row,
                            [
                                Connection::PARAM_INT,
                                Connection::PARAM_INT,
                                Connection::PARAM_INT,
                                Connection::PARAM_INT
                            ]
                        );
                    }
                }
            }
            fwrite($logfile, 'ConnectionsCompetentAuthority END: '. date("Y-m-d H:i:s") . "\r\n");
            fclose($logfile);
        }
    }


    /**
     * Post Import Processes
     * - Clean Up Temp Path
     */
    private function postProcess(){
        $handle=opendir($this->pathTemp);
        while ($datei = readdir ($handle)) {
            if (is_file($this->pathTemp.$datei)){
                rename($this->pathTemp.$datei, $this->pathArchive.date('YmdHis').'_'.$datei);
            }
        }
        closedir($handle);
    }

    /**
     * @return int
     */
    public function getStoragePid()
    {
        return $this->storagePid;
    }

    /**
     * @param int $storagePid
     * @return $this
     */
    public function setStoragePid($storagePid)
    {
        $this->storagePid = $storagePid;
        return $this;
    }

    /**
     * @return string
     */
    public function getImportFile()
    {
        return $this->importFile;
    }

    /**
     * @param string $importFile
     * @return $this
     */
    public function setImportFile($importFile)
    {
        $this->importFile = $importFile;
        return $this;
    }

    /**
     * @return bool
     */
    public function isDebugMode(): bool
    {
        return $this->debugMode;
    }

    /**
     * @param bool $debugMode
     */
    public function setDebugMode(bool $debugMode): void
    {
        $this->debugMode = $debugMode;
    }

    /**
     * This method returns the destination mail address as additional information
     *
     * @return string Information to display
     */
    public function getAdditionalInformation() {
        $return =
            'StoragePid:'.$this->getStoragePid().
            ' File:'. $this->getImportFile();
        if ($this->isDebugMode()){
            $return .= ' DEBUGMODE active';
        }

        return $return;
    }

    /**
     * @param $source
     * @param $target
     */
    private function moveImportFile($source, $target){
        if ($this->isDebugMode()){
            copy($source, $target);
        }else{
            rename($source, $target);
        }
    }

}