<?php
namespace SG\SgHeadless\Util;

use Doctrine\DBAL\FetchMode;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Configuration\ExtensionConfiguration;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\Mailer;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;

class Services
{

    /**
     * @var ConnectionPool
     */
    private $connectionPool;

    /**
     * @var ConfigurationManager
     */
    protected $configurationManager;

    /**
     * @param ConfigurationManager $configurationManager
     */
    public function injectConfigurationManager(ConfigurationManager $configurationManager)
    {
        $this->configurationManager = $configurationManager;
    }

    public function __construct()
    {
        $this->connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
    }

    private function geocodeAddress($country, $city, $zip, $street, $housenumber, $objectNumber, &$log)
    {
        $logDate = date('Y-m-d H:i:s', time()) . ' ';

        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $configuration = $configurationManager->getConfiguration(
            \TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        );
        $backendConfiguration = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance(ExtensionConfiguration::class)->get('sg_estate_core');
        $settings = $configuration['module.']['tx_sgestatecore.']['settings.'];
        if ($country == '') {
            $country = $this->standardCountry;
        }
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_geocode_cache');
        $result = $queryBuilder
            ->select('longitude', 'latitude')
            ->from('tx_sgestatecore_geocode_cache')
            ->where($queryBuilder->expr()->eq('zip', $queryBuilder->createNamedParameter($zip)))
            ->andWhere($queryBuilder->expr()->eq('street', $queryBuilder->createNamedParameter($street)))
            ->andWhere($queryBuilder->expr()->eq('housenumber', $queryBuilder->createNamedParameter($housenumber)))
            ->andWhere($queryBuilder->expr()->eq('country', $queryBuilder->createNamedParameter($country)))
            ->execute();
        if ($result->rowCount() > 0) {
            $resultItem = $result->fetch(FetchMode::ASSOCIATIVE);
            $coordinates['longitude'] = $resultItem['longitude'];
            $coordinates['latitude'] = $resultItem['latitude'];

            $log[] = $logDate . 'object "' . $objectNumber . '": found coordinates for ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country . ' in cache: ' . implode(', ', $coordinates);

            return $coordinates;
        }

        $coordinates = null;
        $address =
            $this->convertMutation($country) . ', ' . $zip . ' ' .
            $this->convertMutation($city) . ', ' .
            $this->convertMutation($street) . ' ' .
            $housenumber;

        $delay = 0;
        $base_url = 'https://maps.googleapis.com/maps/api/geocode/xml';

        $geocode_pending = true;

        $apikey = $backendConfiguration['geocodingApiKey'] != '' ? $backendConfiguration['geocodingApiKey'] : $settings['keys.']['googleapi'];

        while ($geocode_pending) {
            $request_url = $base_url . '?address=' . urlencode($address) . '&key=' . $apikey;
            $xml = simplexml_load_file($request_url) or die('url not loading'); // @todo: Throw Error ??
            $status = $xml->status;

            if (strcmp($status, 'OK') == 0) {
                // Successful geocode
                $geocode_pending = false;
                $coordinates = [
                    'latitude' => floatval($xml->result->geometry->location->lat),
                    'longitude' => floatval($xml->result->geometry->location->lng)
                ];
                $insertData = [
                    'zip'           => $zip,
                    'street'        => $street,
                    'housenumber'   => $housenumber,
                    'country'       => $country,
                    'longitude'     => floatval($xml->result->geometry->location->lng),
                    'latitude'      => floatval($xml->result->geometry->location->lat),
                ];

                $log[] = $logDate . 'object "' . $objectNumber . '": geocoded: ' . implode(', ', $insertData);

                $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_geocode_cache');
                $connection->insert(
                    'tx_sgestatecore_geocode_cache',
                    $insertData
                );
            } elseif (strcmp($status, 'OVER_QUERY_LIMIT') == 0) {
                // sent geocodes too fast
                $delay += 100000;

                $log[] = $logDate . '[WARNING] received over query limit... wait for ' . $delay . ' microseconds...';
            } else {

                // failure to geocode
                $geocode_pending = false;
                $log[] = $logDate . '[ERROR] object "' . $objectNumber . '": could not geocode: ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country;
            }
            usleep($delay);
        }
        return $coordinates;
    }

    /**
     * @param array $recipients
     * @param $sender
     * @param $subject
     * @param $templateName
     * @param array $bcc
     * @param array $variables
     * @param array $attachments
     *  [
     *      0 => [
     *          'type' => 'fromPath',
     *          'path' => '/path/to/attachment'
     *          'filename' => 'filenameInEmail.txt'
     *          ],
     *      1 => [
     *          'type' => 'content',
     *          'content' => 'iamthecontent'
     *          'filename' => 'filenameInEmail.txt'
     *          ]
     *  ]
     *
     * @param string $format
     * @return bool
     * @throws \Symfony\Component\Mailer\Exception\TransportExceptionInterface
     */
    public function sendTemplateEmail(array $recipients, $sender, $subject, $templateName, array $bcc = [], array $variables = [], array $attachments = [], $format = 'html')
    {
        $message = GeneralUtility::makeInstance(FluidEmail::class);

        if (\TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($sender)) {
            $message->from($sender);
        } else {
            return false;
        }

        // Add Recipients
        $recipients = array_filter($recipients);
        if (count($recipients) > 0) {
            foreach ($recipients as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addTo($email);
                }
            }
        } else {
            return false;
        }

        if (count($bcc) > 0) {
            foreach ($bcc as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addBcc($email);
                }
            }
        }

        if ($subject != '') {
            $message->subject($subject);
        } else {
            return false;
        }

        // text/plain and text/html is legacy format from
        // constant editor sg_estate_base pre 10.11.2
        // may be removed in 11.0.0
        if (in_array($format, ['text', 'text/plain'])) {
            $message->format('plain');
        } elseif (in_array($format, ['html', 'text/html'])) {
            $message->format('html');
        }

        $message->assignMultiple($variables);
        $message->setTemplate($templateName . '.' . $format);

        foreach ($attachments as $attachment) {
            switch ($attachment['type']) {
                case 'fromPath':
                    $message->attachFromPath($attachment['path'], $attachment['filename']);
                    break;
                case 'content':
                    $message->attach($attachment['content'], $attachment['filename']);
                    break;
            }
        }

        GeneralUtility::makeInstance(Mailer::class)->send($message);
        return true;
    }


    /**
     * @param $value
     * @return mixed
     */
    private function convertMutation($value)
    {
        $value = str_replace('ü', 'ue', $value);
        $value = str_replace('Ü', 'Ue', $value);
        $value = str_replace('ö', 'oe', $value);
        $value = str_replace('Ö', 'Oe', $value);
        $value = str_replace('ä', 'ae', $value);
        $value = str_replace('Ä', 'Ae', $value);
        $value = str_replace('ß', 'ss', $value);
        return $value;
    }
}
