<?php
namespace SG\SgEstateImportSwoppen\Command;

use Doctrine\DBAL\FetchMode;
use http\Exception\RuntimeException;
use SG\SgEstateCore\Util\Services;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;

class ImportCommand extends Command
{
    protected static $defaultName = 'sg.estate.import.swoppen';

    /**
     * @var integer $storagePid
     */
    protected $storagePid;

    /**
     * @var array
     */
    private $reportData = [];

    /**
     * @var string
     */
    private $tempPath = '';

    /**
     * @var string
     */
    private $archivePath = '';

    /**
     * @var string
     */
    private $uploadPath = '';

    /**
     * @var string
     */
    private $workPath = '';

    /**
     * @var string
     */
    private $mediaPath = '';

    /**
     * @var bool
     */
    private $importFilePresent = false;

    /**
     * @var bool
     */
    private $debugMode = false;

    /**
     * @var Services
     */
    private $sgEstateCoreServices;

    /**
     * @var int
     */
    private $importIdentifier;

    /**
     * @var ConnectionPool
     */
    private $connectionPool;
    /**
     * @var array
     */
    private $alerts = array();

    /**
     * @var string
     */
    private $dataFileName = 'data.csv';

    private $debugCounter = 0;

    public function configure()
    {
        $this->setName('sg.estate.import.swoppen');
        $this->setDescription('Import von Immobilien aus einer CSV-Datei (Abruf vom Anbieter Swoppen)');
        $this->addArgument('storagePid', InputArgument::REQUIRED, 'Storage Pid for imported realties');
        $this->addArgument('importIdentifier', InputArgument::REQUIRED, 'Unique Identifier for this Import (must be numeric)');
        $this->addArgument('reportSenderEmail', InputArgument::REQUIRED, 'E-Mail Address for sender of report');
        $this->addArgument('reportSubject', InputArgument::REQUIRED, 'Subject of Report E-Mail');
        $this->addArgument('reportReceiverEmails', InputArgument::REQUIRED, 'E-Mail Addresses for receivers of report (commaseparated)');
        $this->addArgument('urlForCsv', InputArgument::REQUIRED, 'URL zum Abruf der CSV Daten');
        $this->addArgument('basicAuthUsername', InputArgument::OPTIONAL, 'Benutzername für Basic Authentication');
        $this->addArgument('basicAuthPassword', InputArgument::OPTIONAL, 'Passwort für Basic Authentication');
        $this->addArgument('daysToKeepArchiveFiles', InputArgument::OPTIONAL, 'Days to keep archives Importfiles', '7');
        $this->addArgument('debugMode', InputArgument::OPTIONAL, 'Run in Debug Mode ? ');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $this->reportData['importStart'] = date('d.m.Y H:i:s');
        $this->checkAndSetRequirements($input, $output);
        if ($this->preprocess($input, $output)){
            if (is_file($this->workPath.$this->dataFileName)){
                $row = 1;
                if (($handle = fopen($this->workPath.$this->dataFileName, "r")) !== FALSE) {
                    $this->sgEstateCoreServices->deleteAllUpdatableRealties($this->importIdentifier);
                    while (($data = fgetcsv($handle, 0, ";")) !== FALSE) {
                        if ($row > 1){
                            $this->processRow($data);
                        }
                        $row++;
                        // For Testing only, imports 10 Properties only
                        //if ($row > 10){break;}
                    }
                    fclose($handle);
                    $this->cleanUpTempDirectory();
                    rename($this->workPath.$this->dataFileName, $this->archivePath.'data-'.uniqid().'.csv');
                }
            }
        }
        $this->reportData['importEnde'] = date('d.m.Y H:i:s');
        $this->postprocess($input, $output);
        return 0;
    }

    private function processRow($rowData){
        // UID der Stadt suchen, wenn nicht vorhanden anlegen
        $cityName = $rowData[$this->getCol('city',0)];
        if ($cityName != ''){

            $cityUid = $this->addCity($cityName, true);

            $districtUid = null;

            $propertyData = array(
                'pid' => $this->storagePid,
                'updatable' => 1,
                'import_number' => $this->importIdentifier
            );
            $kindOfUseStorage = NULL;
            $marketingMethodStorage = NULL;

            if ($cityUid != null){
                $propertyData['objekt_ort'] = $cityUid;
            }
            if ($districtUid != null){
                $propertyData['objekt_stadtteil'] = $districtUid;
            }
            if ($this->getCol('objectNumberInternal',0) !== null){
                $propertyData['objektnr_intern'] = $rowData[$this->getCol('objectNumberInternal',0)];
            }
            if ($this->getCol('objectNumberExternal',0) !== null){
                $propertyData['objektnr_extern'] = strtolower($rowData[$this->getCol('objectNumberExternal',0)]);
            }
            if ($this->getCol('openimmoObjectId',0) !== null){
                $propertyData['openimmo_objid'] = $rowData[$this->getCol('openimmoObjectId',0)];
            }
            if ($this->getCol('title',0) !== null){
                $propertyData['objekttitel'] = $rowData[$this->getCol('title',0)];
            }
            if ($this->getCol('description',0) != null){
                $propertyData['objektbeschreibung'] = $rowData[$this->getCol('description',0)];
            }
            if ($this->getCol('zip',0) !== null){
                $propertyData['objekt_plz'] = $rowData[$this->getCol('zip',0)];
            }
            if ($this->getCol('street',0) !== null){
                $propertyData['objekt_strasse'] = $rowData[$this->getCol('street',0)];
            }
            if ($this->getCol('housenumber',0) !== null){
                $propertyData['objekt_hausnummer'] = $rowData[$this->getCol('housenumber',0)];
            }
            if ($this->getCol('latitude',0) !== null){
                $propertyData['objekt_breitengrad'] = (double)$rowData[$this->getCol('latitude',0)];
            }
            if ($this->getCol('longitude',0) !== null){
                $propertyData['objekt_laengengrad'] = (double)$rowData[$this->getCol('longitude',0)];
            }
            if ($this->getCol('livingSpace',0) !== null){
                $propertyData['wohnflaeche'] = $rowData[$this->getCol('livingSpace',0)] != '' ? str_replace(',','.',$rowData[$this->getCol('livingSpace',0)]) : 0;
            }
            if ($this->getCol('roomCount',0) != null){
                $propertyData['anzahl_zimmer'] = $rowData[$this->getCol('roomCount',0)] != '' ? str_replace(',','.',$rowData[$this->getCol('roomCount',0)]) : 0;
            }
            if ($this->getCol('rentExclusiveOfHeating',0) != null){
                $propertyData['kaltmiete'] = $rowData[$this->getCol('rentExclusiveOfHeating',0)] != '' ? str_replace(',','.',$rowData[$this->getCol('rentExclusiveOfHeating',0)]) : 0;
            }
            if ($this->getCol('rent',0) != null){
                $propertyData['warmmiete'] = $rowData[$this->getCol('rent',0)] != '' ? str_replace(',','.',$rowData[$this->getCol('rent',0)]) : 0;
            }
            if ($this->getCol('minimumPersons',0) != null){
                $propertyData['minimale_personenanzahl'] = (int)$rowData[$this->getCol('minimumPersons',0)];
            }
            if ($this->getCol('maximumPersons',0) != null){
                $propertyData['maximale_personenanzahl'] = (int)$rowData[$this->getCol('maximumPersons',0)];
            }
            if ($this->getCol('beds',0) != null){
                $propertyData['anzahl_betten'] = (int)$rowData[$this->getCol('beds',0)];
            }
            if ($this->getCol('bedsSingle',0) != null){
                $propertyData['anzahl_betten_einzel'] = (int)$rowData[$this->getCol('bedsSingle',0)];
            }
            if ($this->getCol('bedsDouble',0) != null){
                $propertyData['anzahl_betten_doppel'] = (int)$rowData[$this->getCol('bedsDouble',0)];
            }
            if ($this->getCol('bedsCouch',0) != null){
                $propertyData['anzahl_betten_schlafsofa'] = (int)$rowData[$this->getCol('bedsCouch',0)];
            }

            // Country set Standard to Deutschland
            $queryBuilder__tx_sgestatecore_domain_model_land = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_land');
            $result = $queryBuilder__tx_sgestatecore_domain_model_land
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_land')
                ->where($queryBuilder__tx_sgestatecore_domain_model_land->expr()->eq('iso_kennzeichen', $queryBuilder__tx_sgestatecore_domain_model_land->createNamedParameter('DEU')))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))){
                $propertyData['objekt_land'] = $resultItemUid;
            }else{
                throw new RuntimeException('Standard Country not found');
            }

            // Kind Of Object set Standard to WOHNUNG
            $queryBuilder__tx_sgestatecore_domain_model_objektart = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektart');
            $result = $queryBuilder__tx_sgestatecore_domain_model_objektart
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_objektart')
                ->where($queryBuilder__tx_sgestatecore_domain_model_objektart->expr()->eq('kuerzel', $queryBuilder__tx_sgestatecore_domain_model_objektart->createNamedParameter('WOHNUNG')))
                ->execute();

            if ($result->rowCount() > 0){
                $propertyData['objektart'] = $result->fetch(FetchMode::ASSOCIATIVE)['uid'];
            }else{
                throw new RuntimeException('Standard Kind of Object not found');
            }

            // Currency set Standard to EURO
            $queryBuilder__tx_sgestatecore_domain_model_waehrung = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_waehrung');
            $result = $queryBuilder__tx_sgestatecore_domain_model_waehrung
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_waehrung')
                ->where($queryBuilder__tx_sgestatecore_domain_model_waehrung->expr()->eq('iso_kennzeichen', $queryBuilder__tx_sgestatecore_domain_model_waehrung->createNamedParameter('EUR')))
                ->execute();

            if ($result->rowCount() > 0){
                $propertyData['waehrung'] = $result->fetch(FetchMode::ASSOCIATIVE)['uid'];
            }else{
                throw new RuntimeException('Standard Currency not found');
            }

            // Equipment
            if ($this->getCol('equipmentDescription',0) != null){
                $unformattedEquipmentDescription = $rowData[$this->getCol('equipmentDescription',0)];
                $level1Items = explode(',', $unformattedEquipmentDescription);
                $equipmentArray = null;
                foreach ($level1Items as $level1Item){
                    $level2Items = explode('>', $level1Item);

                    if (count($level2Items) > 0){
                        if ($level2Items[0] != ''){
                            if (!isset($equipmentArray[$level2Items[0]])){
                                $equipmentArray[$level2Items[0]] = [];
                            }
                            if (isset($level2Items[1])){
                                array_push($equipmentArray[$level2Items[0]], $level2Items[1]);
                            }
                        }
                    }
                }

                if (is_array($equipmentArray) && count($equipmentArray) > 0){
                    $propertyData['austattungsbeschreibung'] = '<ul>';
                    foreach ($equipmentArray as $headline => $items){
                        $propertyData['austattungsbeschreibung'] .= '<li>'.$headline;
                        if (count($items) > 0){
                            $propertyData['austattungsbeschreibung'] .= '<ul>';
                            foreach ($items as $item){
                                $propertyData['austattungsbeschreibung'] .= '<li>'.$item.'</li>';
                            }
                            $propertyData['austattungsbeschreibung'] .= '</ul>';
                        }
                        $propertyData['austattungsbeschreibung'] .= '</li>';
                    }
                    $propertyData['austattungsbeschreibung'] .= '</ul>';
                }
            }

            // Set sys_language to -1
            $propertyData['sys_language_uid'] = -1;

            // Insert Data for Property
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
            $result = $queryBuilder->select('uid')
                                   ->from('tx_sgestatecore_domain_model_immobilie')
                                   ->where($queryBuilder->expr()->eq('objektnr_intern', $queryBuilder->createNamedParameter($rowData[$this->getCol('objectNumberInternal',0)])))
                                   ->andWhere($queryBuilder->expr()->eq('import_number', $queryBuilder->createNamedParameter($this->importIdentifier)))
                                   ->execute();
            if ($result->rowCount() >=1){
                // Update Property
                $uidProperty = $result->fetch()['uid'];
                $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
                $connectionPoolForProperties->update(
                    'tx_sgestatecore_domain_model_immobilie',
                    $propertyData,
                    ['uid' => $uidProperty]
                );
                $this->reportData['updated'][] = strval($rowData[$this->getCol('objectNumberInternal',0)]);
            }else{
                // Add new Property
                $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
                $connectionPoolForProperties->insert(
                    'tx_sgestatecore_domain_model_immobilie',
                    $propertyData
                );
                $uidProperty = (int)$connectionPoolForProperties->lastInsertId('tx_sgestatecore_domain_model_immobilie');
                $this->reportData['inserted'][] = strval($rowData[$this->getCol('objectNumberInternal',0)]);
            }

            //Ausstattung richtig
            if (is_array($equipmentArray) && count($equipmentArray) > 0){
                foreach ($equipmentArray as $headline => $items){
                    foreach ($items as $item){
                        $this->addAusstattung($headline, $item, $uidProperty);
                    }
                }
            }


            // Marketing Methods
            // Reset all Connections for this uidProperty
            $queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_vermarktungsart_mm');
            $queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm
                ->delete('tx_sgestatecore_immobilie_vermarktungsart_mm')
                ->where($queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm->expr()->eq('uid_local', $queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm->createNamedParameter($uidProperty)))
                ->execute();

            // Set Standard Mieten
            $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_vermarktungsart');
            $result = $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_vermarktungsart')
                ->where($queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->expr()->eq('kuerzel', $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->createNamedParameter('MIETE_PACHT')))
                ->execute();

            if ($result->rowCount() > 0){
                $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_vermarktungsart');
                $connection->insert(
                    'tx_sgestatecore_immobilie_vermarktungsart_mm',
                    [
                        'uid_local' => $uidProperty,
                        'uid_foreign' => $result->fetch(FetchMode::ASSOCIATIVE)['uid']
                    ]
                );
            }
            // Add from Field Zielgruppen
            if ($this->getCol('targetGroups',0) != null){
                if ($rowData[$this->getCol('targetGroups',0)] != ''){
                    $targetGroups = explode(',', $rowData[$this->getCol('targetGroups',0)]);
                    foreach ($targetGroups as $targetGroup){
                        $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_vermarktungsart');
                        $result = $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart
                            ->select('uid')
                            ->from('tx_sgestatecore_domain_model_vermarktungsart')
                            ->where($queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->expr()->eq('kuerzel', $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->createNamedParameter(strtoupper($targetGroup))))
                            ->execute();
                        if ($result->rowCount() > 0){
                            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_vermarktungsart');
                            $connection->insert(
                                'tx_sgestatecore_immobilie_vermarktungsart_mm',
                                [
                                    'uid_local' => $uidProperty,
                                    'uid_foreign' => $result->fetch(FetchMode::ASSOCIATIVE)['uid']
                                ]
                            );
                        }
                    }
                }
            }

            // Attachments
            $queryBuilder__tx_sgestatecore_domain_model_anhangtyp = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhangtyp');
            $result = $queryBuilder__tx_sgestatecore_domain_model_anhangtyp
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_anhangtyp')
                ->where($queryBuilder__tx_sgestatecore_domain_model_anhangtyp->expr()->like('kuerzel', $queryBuilder__tx_sgestatecore_domain_model_anhangtyp->createNamedParameter('BILD')))
                ->execute();
            $attachmentTypeUid = $result->fetch()['uid'];
            foreach ($this->getCol('media',0) as $mediaCols) {
                if ($rowData[$mediaCols['name']] != ''){
                    $insertArray = array();
                    $insertArray['sys_language_uid'] = -1;
                    $insertArray['pid'] = $this->storagePid;
                    $insertArray['immobilie'] = $uidProperty;
                    $insertArray['format'] = 'JPG';
                    $insertArray['typ'] = $attachmentTypeUid;
                    $insertArray['titel'] = isset($propertyData['objekttitel']) ? $propertyData['objekttitel'] : '';
                    $filename = uniqid($this->importIdentifier.'_',true).'.jpg';
                    $insertArray['datei'] = $filename;
                    file_put_contents($this->mediaPath.$filename, fopen($rowData[$mediaCols['name']], 'r'));
                    if (is_file($this->mediaPath.$filename)){
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                    }
                }
            }
        }
    }

    private function addAusstattung($category, $item, $propertyUid){
        $table_name = '';
        $table_mm = '';

        switch ($category){
            case 'Generelles':
                break;
            case 'Badezimmer':
                $table_name = "tx_sgestatecore_domain_model_badausstattung";
                $table_mm = "tx_sgestatecore_immobilie_badausstattung_mm";
                break;
            case 'Kochmöglichkeiten':
                $table_name = "tx_sgestatecore_domain_model_kuechenausstattung";
                $table_mm = "tx_sgestatecore_immobilie_kuechenausstattung_mm";
                break;
            default:
                break;
        }

        if ($table_name !== '') {
            $query = "uid FROM " . $table_name." WHERE bezeichner LIKE '".$item."'";

            $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionByName('Default');
            $rows = $connection->createQueryBuilder()->addSelectLiteral($query)->execute()->fetchAll();

            if ($rows->num_rows > 0){
                $ausstattung_id = $rows[0]['uid'];
            }
            else {
                //$insert_query = "INSERT INTO ".$table_name." VALUES (bezeichner, kuerzel) VALUES (".$item.", '')";
                $connection->insert(
                    $table_name,
                    [
                        'bezeichner' => $item,
                        'kuerzel' => ''
                    ]
                );
                $ausstattung_id = (int)$connection->lastInsertId($table_name);
            }

            //$query = "INSERT INTO ".$table_mm." (uid_local, uid_foreign) VALUES ('".$propertyUid."', '".$ausstattung_id."')";
            $connection->insert(
                $table_mm,
                [
                    'uid_local' => $propertyUid,
                    'uid_foreign' => $ausstattung_id
                ]
            );
        }
    }

    private function checkAndSetRequirements(InputInterface $input, OutputInterface $output)
    {
        // Check the importIdentifier
        if (!is_numeric($input->getArgument('importIdentifier'))){
            throw new InvalidArgumentException('importIdentifier MUST be numeric');
        }else{
            $this->importIdentifier = $input->getArgument('importIdentifier');
        }

        // Check the storagePid
        if (!is_numeric($input->getArgument('storagePid'))){
            throw new InvalidArgumentException('storagePid MUST be numeric');
        }else{
            $this->storagePid = $input->getArgument('storagePid');
        }

        // Creates all needed directories if not present
        $basePathForImportCommand = Environment::getPublicPath().'/uploads/tx_sgestateimportswoppen/'. $input->getArgument('importIdentifier');
        if (!is_dir($basePathForImportCommand)){
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/temp');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/archive');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/upload');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/work');
        }

        // Set all Path Variables
        $this->tempPath = $basePathForImportCommand . '/temp/';
        $this->archivePath = $basePathForImportCommand . '/archive/';
        $this->uploadPath = $basePathForImportCommand . '/upload/';
        $this->workPath = $basePathForImportCommand . '/work/';
        $this->mediaPath = Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/';

        if ($input->getArgument('debugMode') != ''){
            $this->debugMode = true;
        }

        $this->connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
        $this->sgEstateCoreServices = GeneralUtility::makeInstance(Services::class);

        $this->addAdditionalMarketingMethods();
    }

    private function addAdditionalMarketingMethods(){
        $additionalMarketingMethods = [
            [
                'kuerzel' => 'FERIENWOHNUNG',
                'bezeichner' => 'Ferienwohnung'
            ],
            [
                'kuerzel' => 'MESSEZIMMER',
                'bezeichner' => 'Messezimmer'
            ],
            [
                'kuerzel' => 'MONTEURZIMMER',
                'bezeichner' => 'Monteurzimmer'
            ],
        ];
        foreach ($additionalMarketingMethods as $additionalMarketingMethod){
            $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_vermarktungsart');
            $result = $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_vermarktungsart')
                ->where($queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->expr()->eq('kuerzel', $queryBuilder__tx_sgestatecore_domain_model_vermarktungsart->createNamedParameter($additionalMarketingMethod['kuerzel'])))
                ->execute();
            if ($result->rowCount() == 0){
                $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_vermarktungsart');
                $connection->insert(
                    'tx_sgestatecore_domain_model_vermarktungsart',
                    [
                        'pid' => $this->storagePid,
                        'kuerzel' => $additionalMarketingMethod['kuerzel'],
                        'bezeichner' => $additionalMarketingMethod['bezeichner']
                    ]
                );
            }
        }

    }

    private function preprocess(InputInterface $input, OutputInterface $output){
        if ($input->getArgument('urlForCsv') != '' && GeneralUtility::isValidUrl($input->getArgument('urlForCsv'))){
            // Retrieve CSV Data from URL
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $input->getArgument('urlForCsv'));
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            if ($input->hasArgument('basicAuthUsername') && $input->getArgument('basicAuthUsername') != ''){
                curl_setopt($ch, CURLOPT_USERPWD, $input->getArgument('basicAuthUsername').':'.$input->getArgument('basicAuthPassword'));
            }
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);

            if(curl_exec($ch) === false)
            {
                throw new \Exception('Curl error: ' . curl_error($ch));
            }

            $result = curl_exec($ch);
            $returnCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            if ($returnCode == 200){
                if (file_put_contents($this->workPath.$this->dataFileName, $result) !== false){
                    $this->importFilePresent = true;
                    return true;
                }else{
                    throw new \Exception('File could not be written');
                }
            }else{
                throw new \Exception('Url could not be retrieved');
            }
        }
    }

    private function postprocess(InputInterface $input, OutputInterface $output){
        if ($this->importFilePresent){
            /**
             * @var $cache FrontendInterface
             */
            // Reset Cache
            $cache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgestate_cache');
            $cache->flush();

            $this->sgEstateCoreServices->sendTemplateEmail(
                explode(',', $input->getArgument('reportReceiverEmails')),
                $input->getArgument('reportSenderEmail'),
                $input->getArgument('reportSubject'),
                'SgEstateImportSwoppenReport',
                array(),
                array(
                    'reportDaten' => $this->reportData
                ),
                array(),
                'html'
            );
        }
        $this->cleanUpTempDirectory();

        $this->cleanUpArchiveFiles($input, $output);

    }

    private function addCity($cityName, $useForSearch = null){
        if ($cityName != ''){
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ort');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_ort')
                ->where($queryBuilder->expr()->like('bezeichner', $queryBuilder->createNamedParameter($cityName)))
                ->execute();

            $data = array();
            $data['pid']        = $this->storagePid;
            $data['bezeichner'] = $cityName;
            if ($useForSearch === true){
                $data['fuer_suche_verwenden'] = 1;
            }
            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_ort');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))){
                $connection->update(
                    'tx_sgestatecore_domain_model_ort',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            }else{
                $connection->insert(
                    'tx_sgestatecore_domain_model_ort',
                    $data
                );
                $uid = (int)$connection->lastInsertId('tx_sgestatecore_domain_model_ort');
            }
            return $uid;
        }else{
            return null;
        }
    }

    /**
     * Cleans up Archive Files
     */
    private function cleanUpArchiveFiles(InputInterface $input, OutputInterface $output)
    {
        $handle = opendir($this->archivePath);
        $now = time();
        $secondsToKeepArchiveFiles = (int)$input->getArgument('daysToKeepArchiveFiles')*24*60*60;
        $borderTimestamp = $now - $secondsToKeepArchiveFiles;

        while ($file = readdir($handle)) {
            $filetime= filemtime($this->archivePath . $file);
            if ($file != '..' && $file != '.' && $filetime < $borderTimestamp){
                @unlink($this->archivePath . $file);
            }
        }
        closedir($handle);
    }

    private function cleanUpTempDirectory(){
        $directoryContent = scandir($this->tempPath);
        foreach ($directoryContent as $entry) {
            if (!is_dir($entry)){
                @unlink($this->tempPath.$entry);
            }
        }
    }

    private function translation(){
        $columns = [
            0 => 'apt_id',
            1 => 'apt_no',
            2 => 'landlord_no',
            3 => 'hs3_no',
            4 => 'street_name',
            5 => 'street_no',
            6 => 'district_code',
            7 => 'city',
            8 => 'latitude',
            9 => 'longitude',
            10 => 'Title',
            11 => 'Content',
            12 => 'Excerpt',
            13 => 'Date',
            14 => 'Städte',
            15 => 'Zielgruppen',
            16 => 'Eigenschaften',
            17 => 'apt_setting',
            18 => 'price_from',
            19 => 'price_to',
            20 => 'size',
            21 => 'rooms',
            22 => 'beds',
            23 => 'beds_single',
            24 => 'beds_double',
            25 => 'beds_couch',
            26 => 'people_from',
            27 => 'people_to',
            28 => 'stays_from',
            29 => 'veroeffentlichungsdatum',
            30 => 'Artikel-Nr.',
            31 => 'Bild1',
            32 => 'Bild2',
            33 => 'Bild3',
            34 => 'Bild4',
            35 => 'Bild5',
            36 => 'Bild6',
            37 => 'Bild7',
            38 => 'Bild9',
            39 => 'Bild9',
            40 => 'Bild10',
            41 => 'youtube',
        ];
        return $columns;
    }

    private function getCol($attribute, $kindOfObject = 0){

        $generalTranslate = array(
            // Immobilienartunabhängig
            'kindOfObject'              => null,
            'objectNumberExternal'      => 30,
            'objectNumberInternal'      => 0,
            'openimmoObjectId'          => 0,
            'contact_salutation'        => null,
            'contact_firstname'         => null,
            'contact_lastname'          => null,
            'contact_street'            => null,
            'contact_housenumber'       => null,
            'contact_zip'               => null,
            'contact_city'              => null,
            'contact_country'           => null,
            'contact_telephone'         => null,
            'contact_mobile'            => null,
            'contact_fax'               => null,
            'contact_email'             => null,
            'contact_url'               => null,
            'contact_company'           => null,
            'showGeo'                   => null,
            'title'                     => 10,
            'brokerageValue'            => null,
            'currency'                  => null,
            'brokerage'                 => null,
            'city'                      => 7,
            'street'                    => 4,
            'housenumber'               => 5,
            'zip'                       => 6,
            'country'                   => null,
            'siteDescription'           => null,
            'equipmentDescription'      => 16,
            'description'               => 11,
            'additionalDescription'     => null,
            'media'                     => array(
                1  => array(
                    'name'        => 31,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                2  => array(
                    'name'        => 32,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                3  => array(
                    'name'        => 33,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                4  => array(
                    'name'        => 34,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                5  => array(
                    'name'        => 35,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                6  => array(
                    'name'        => 36,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                7  => array(
                    'name'        => 37,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                8  => array(
                    'name'        => 38,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                9  => array(
                    'name'        => 39,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
                10 => array(
                    'name'        => 40,
                    'extension'   => null,
                    'type'        => null,
                    'playtime'    => null,
                    'description' => null,
                ),
            ),
            // Immobilienartabhängig
            'kindOfObjectType'          => null,
            'livingSpace'               => 20,
            'totalArea'                 => null,
            'effectiveSurface'          => null,
            'roomCount'                 => 21,
            'bathroomCount'             => null,
            'floor'                     => null,
            'floorCount'                => null,
            'yearOfConstruction'        => null,
            'heatingType'               => null,
            'lift'                      => null,
            'petAllowed'                => null,
            'balconyCount'              => null,
            'gardenUse'                 => null,
            'fittedKitchen'             => null,
            'carportCount'              => null,
            'availableFrom'             => null,
            'accessibleForWheelchairs'  => null,
            'sleepingroomCount'         => null,
            'barrierFree'               => null,
            'beacoaningType'            => null,
            'seperateToilet'            => null,
            'basementAvailable'         => null,
            'netRentExclusiveOfHeating' => null,
            'rentExclusiveOfHeating'    => 18,
            'rent'                      => 19,
            'additionalExpenses'        => null,
            'heatingCosts'              => null,
            'rentIncludesHeatingCosts'  => null,
            'basementSpace'             => null,
            'bathWithShower'            => null,
            'bathWithTub'               => null,
            'bathWithWindow'            => null,
            'broadbandSpeed'            => null,
            'broadbandType'             => null,
            'flooringMaterials'         => null,
            'goodsLift'                 => null,
            'kindsOfUse'                => null,
            'marketingMethods'          => null,
            'operatingCosts'            => null,
            'parkingSpaces'             => null,
            'shortDescription'          => null,
            'updated'                   => null,
            'wohnberechtigungsschein'   => null,
            'district'                  => null,
            'latitude'                  => 8,
            'longitude'                 => 9,
            'targetGroups'              => 15,
            'minimumPersons'            => 26,
            'maximumPersons'            => 27,
            'beds'                      => 22,
            'bedsSingle'                => 23,
            'bedsDouble'                => 24,
            'bedsCouch'                 => 25,
            'youtube'                   => 41,
        );
        /**
         * Mapping for Kind of Object Type 0
         */
        // Wohnung Miete
        $translate[0] = array_merge($generalTranslate, array());
        // Haus Miete
        $translate[1] = array_merge($generalTranslate, array());
        // Wohnung Kauf
        $translate[2] = array_merge($generalTranslate, array());
        // Haus Kauf
        $translate[3] = array_merge($generalTranslate, array());
        // Wohnen auf Zeit
        $translate[4] = array_merge($generalTranslate, array());
        // Grundstücke
        $translate[5] = array_merge($generalTranslate, array());
        //Büros/Praxen
        $translate[7] = array_merge($generalTranslate, array());
        // Einzelhandel
        $translate[8] = array_merge($generalTranslate, array());
        // Gastronomie
        $translate[9] = array_merge($generalTranslate, array());
        // Hallen- und Produktionsflächen
        $translate[10] = array_merge($generalTranslate, array());
        // Sonstige Objekte Gewerbe
        $translate[11] = array_merge($generalTranslate, array());
        // Anlageobjekte
        $translate[12] = array_merge($generalTranslate, array());
        // Stellplatz Kauf
        $translate[17] = array_merge($generalTranslate, array());
        // Stellplatz Miete
        $translate[18] = array_merge($generalTranslate, array());

        if (!is_numeric($kindOfObject)){
            $kindOfObject = 0;
        }
        if (isset($translate[$kindOfObject][$attribute])){
            return $translate[$kindOfObject][$attribute];
        }else{
            return null;
        }

    }
}