<?php
namespace SG\SgEstateImportOpenimmo\Command;

use DateMalformedStringException;
use DateTime;
use Doctrine\DBAL\Exception;
use SG\SgEstateCore\Util\Services;
use SimpleXMLElement;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use UnexpectedValueException;
use ZipArchive;

class ImportCommand extends Command
{
    protected static string $defaultName = 'sg.estate.import.openimmo';

    protected int $storagePid;

    private array $reportData = [];

    private string $logFile = '';

    private string $tempPath = '';

    private string $archivePath = '';

    private string $uploadPath = '';

    private string $workPath = '';

    private string $mediaPath = '';

    private bool $debugMode = false;

    private ?string $filenameDatafile = '';

    private bool $importFilePresent = false;

    private int $importIdentifier;

    private array $alerts = [];

    public function __construct(
        private readonly FrontendInterface $sgEstateCoreCache,
        private readonly ConnectionPool    $connectionPool,
        private readonly Services          $sgEstateCoreServices,
        ?string                            $name = null,
    )
    {
        parent::__construct($name);
    }

    protected function configure(): void
    {
        $this->setName('sg.estate.import.openimmo');
        $this->setDescription('Import von Immobilien aus einer OpenimmoXML-Datei (Bilder und XML Datei in ZIP-Archiv)');
        $this->addArgument('storagePid', InputArgument::REQUIRED, 'Storage Pid for imported realties');
        $this->addArgument('importIdentifier', InputArgument::REQUIRED, 'Unique Identifier for this Import (must be numeric)');
        $this->addArgument('reportSenderEmail', InputArgument::REQUIRED, 'E-Mail Address for sender of report');
        $this->addArgument('reportSubject', InputArgument::REQUIRED, 'Subject of Report E-Mail');
        $this->addArgument('reportReceiverEmails', InputArgument::REQUIRED, 'E-Mail Addresses for receivers of report (commaseparated)');
        $this->addArgument('daysToKeepArchiveFiles', InputArgument::OPTIONAL, 'Days to keep archives Importfiles', '7');
        $this->addArgument('debugMode', InputArgument::OPTIONAL, 'Run in Debug Mode ? ');
        $this->addArgument('logging', InputArgument::OPTIONAL, 'Log imports processes');
        $this->addArgument('forceImportModeFull', InputArgument::OPTIONAL, 'Force "full" import mode and remove all estate items of this task before processing the current import file.');
    }

    /**
     * @throws TransportExceptionInterface
     * @throws Exception
     * @throws \TYPO3\CMS\Backend\Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->reportData['importStart'] = date('d.m.Y H:i:s');
        $this->checkAndSetRequirements($input);
        if ($this->preprocess()) {
            $files = scandir($this->workPath);
            sort($files);
            $this->reportData['importFilesCount'] = count($files);
            $i = 0;
            foreach ($files as $file) {
                $tempData = pathinfo($this->workPath . $file);
                if (strtolower($tempData['extension']) === 'zip') {
                    $this->reportData['importFileToWork'][$i] = $file;
                    if ($this->unzip($this->workPath . $file, $this->tempPath)) {
                        $this->reportData['importFileUnziped'][$i] = 'OK';
                        $this->filenameDatafile = null;
                        $tempFiles = scandir($this->tempPath);
                        foreach ($tempFiles as $dataFile) {
                            $dataFileInfo = pathinfo($this->tempPath . $dataFile);
                            if (strtolower($dataFileInfo['extension']) === 'xml') {
                                $this->reportData['process'][$i]['importFileXMLFound'] = $dataFile;
                                $this->filenameDatafile = $dataFile;
                                $this->importFilePresent = true;
                                $this->reportData['process'][$i]['importFileXMLStartImport'] = 'OK';
                                $this->import($input, $i);
                                $this->reportData['process'][$i]['importFileXMLEndImport'] = 'OK';
                            }
                        }
                        $this->cleanUpTempDirectory();
                    } else {
                        $this->addAlert('Fehler', 'Datei ' . $file . ' konnte nicht entpackt werden');
                    }
                    rename($this->workPath . $file, $this->archivePath . $tempData['filename'] . '-' . uniqid('', true) . '.zip');
                    $i++;
                }
            }
        }
        $this->reportData['importEnde'] = date('d.m.Y H:i:s');
        $this->postprocess($input);
        return 0;
    }

    /**
     * @throws \TYPO3\CMS\Backend\Exception
     */
    private function checkAndSetRequirements(InputInterface $input): void
    {
        // Check the importIdentifier
        if (!is_numeric($input->getArgument('importIdentifier'))) {
            throw new InvalidArgumentException('importIdentifier MUST be numeric', 8259919113);
        }

        $this->importIdentifier = $input->getArgument('importIdentifier');

        // Check the storagePid
        if (!is_numeric($input->getArgument('storagePid'))) {
            throw new InvalidArgumentException('storagePid MUST be numeric', 6626447679);
        }

        $this->storagePid = $input->getArgument('storagePid');

        if (!class_exists(ZipArchive::class)) {
            throw new \TYPO3\CMS\Backend\Exception('ZipArchive is not installed', 8763580871);
        }

        // Creates all needed directories if not present
        $basePathForImportCommand = Environment::getVarPath() . '/uploads/tx_sgestateimportopenimmo/' . $input->getArgument('importIdentifier');
        if (!is_dir($basePathForImportCommand)) {
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/temp');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/archive');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/upload');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/work');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/log');
        }

        // Set all Path Variables
        $this->tempPath = $basePathForImportCommand . '/temp/';
        $this->archivePath = $basePathForImportCommand . '/archive/';
        $this->uploadPath = $basePathForImportCommand . '/upload/';
        $this->workPath = $basePathForImportCommand . '/work/';
        $this->logFile = $basePathForImportCommand . '/log/log.json';
        $this->mediaPath = Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/';

        if ((string)$input->getArgument('debugMode') !== '') {
            $this->debugMode = true;
        }
    }

    private function preprocess(): bool
    {
        $files = scandir($this->uploadPath);
        $atLeastOneFilePresentForImport = false;
        foreach ($files as $file) {
            $atLeastOneFilePresentForImport = true;
            $tempData = pathinfo($this->uploadPath . $file);
            if (strtolower($tempData['extension']) === 'zip') {
                if ($this->debugMode) {
                    copy($this->uploadPath . $file, $this->workPath . $file);
                } else {
                    rename($this->uploadPath . $file, $this->workPath . $file);
                }
            }
        }
        return $atLeastOneFilePresentForImport;
    }

    /**
     * Sends the Import Report
     * Removes Tempfiles
     * @throws TransportExceptionInterface
     */
    private function postprocess(InputInterface $input): void
    {
        if ($this->importFilePresent) {
            /**
             * @var $cache FrontendInterface
             */
            // Reset Cache
            $this->sgEstateCoreCache->flush();

            // @todo: Add when sg-estate-base is updated
//            $estateBaseCache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgestatebase_realtycache');
//            $estateBaseCache->flush();

            $this->sgEstateCoreServices->sendTemplateEmail(
                explode(',', $input->getArgument('reportReceiverEmails')),
                $input->getArgument('reportSenderEmail'),
                $input->getArgument('reportSubject'),
                'SgEstateImportOpenimmoReport',
                [],
                [
                    'reportDaten' => $this->reportData,
                    'alerts' => $this->alerts //@todo: Integrate in Mailtemplate
                ]
            );
        }

        $this->cleanUpTempDirectory();

        $this->cleanUpArchiveFiles($input);
    }

    /**
     * Cleans up Archive Files
     */
    private function cleanUpArchiveFiles(InputInterface $input): void
    {
        $handle = opendir($this->archivePath);
        $now = time();
        $secondsToKeepArchiveFiles = (int)$input->getArgument('daysToKeepArchiveFiles')*24*60*60;
        $borderTimestamp = $now - $secondsToKeepArchiveFiles;

        while ($file = readdir($handle)) {
            $filetime= filemtime($this->archivePath . $file);
            if ($file !== '..' && $file !== '.' && $filetime < $borderTimestamp) {
                @unlink($this->archivePath . $file);
            }
        }
        closedir($handle);
    }

    /**
     * @throws Exception
     * @throws \Exception
     */
    private function import(InputInterface $input, $i): void
    {
        $dataImport = new SimpleXMLElement(file_get_contents($this->tempPath . $this->filenameDatafile));
        $this->reportData['process'][$i]['importFileXMLDataRead'] = 'OK';
        // Your can force the usage of a full mode if you pass anything in the
        // "forceImportModeFull" attr.
        $isImportModeFull = true;
        $this->reportData['process'][$i]['importFileXMLImportModel_IS'] = 'FULL';
        $forceImportModeFull = false;
        $this->reportData['process'][$i]['importFileXMLImportModel_FORCE'] = 'NO';
        if ((string)$dataImport->uebertragung[0]->attributes()->umfang !== 'VOLL') {
            $isImportModeFull = false;
            $this->reportData['process'][$i]['importFileXMLImportModel_IS'] = 'PARTIAL';
        }
        // Keep in mind that "forceImportModeFull" overrides the import mode in the file!
        if ($input->getArgument('forceImportModeFull') !== '') {
            $isImportModeFull = true;
            $this->reportData['process'][$i]['importFileXMLImportModel_IS'] = 'FULL';
            $forceImportModeFull = true;
            $this->reportData['process'][$i]['importFileXMLImportModel_FORCE'] = 'YES';
        }

        if ($isImportModeFull) {
            $this->reportData['process'][$i]['importFileXMLImportModelFull_deleteAllItemsTry'] = 'OK';
            $this->sgEstateCoreServices->deleteAllUpdatableRealties($this->importIdentifier);
            $this->reportData['process'][$i]['importFileXMLImportModelFull_allItemsDeleted'] = 'OK';
            $this->reportData['importType'] = 'Full Import';
            if ($forceImportModeFull) {
                $this->reportData['importType'] .= ' (forced)';
            }
        } else {
            $this->reportData['importType'] = 'Partial Import';
        }
        $this->reportData['process'][$i]['importFileXMLImportReadXML_start'] = 'OK';
        foreach ($dataImport->anbieter[0]->immobilie as $item) {
            /**
             * @var SimpleXMLElement $item
             */
            $action = (string)$item->verwaltung_techn->aktion->attributes()->aktionart;
            $this->reportData['process'][$i]['importFileXMLImportTypeFound'] = $action;
            $this->reportData['process'][$i]['importFileXMLImportExternNrFound'] = $item->verwaltung_techn->objektnr_extern;
            $this->reportData['process'][$i]['importFileXMLImportInternNrFound'] = $item->verwaltung_techn->objektnr_intern;
            switch ($action) {
                case 'DELETE':
                    $this->reportData['process'][$i]['importFileXMLImportStartFindRealtyByExternNr'] = 'OK';
                    $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
                    $queryBuilder->getRestrictions()->removeAll();
                    $result = $queryBuilder->select('uid', 'kontaktperson')
                        ->from('tx_sgestatecore_domain_model_immobilie')->where($queryBuilder->expr()->eq('objektnr_extern', $queryBuilder->createNamedParameter($item->verwaltung_techn->objektnr_extern)))->executeQuery();
                    if ($result->rowCount() >= 1) {
                        $this->reportData['process'][$i]['importFileXMLImportFindRealtyByExternNrQty'] = $result->rowCount();
                        $this->reportData['process'][$i]['importFileXMLImportStartDeleteByExternNr'] = 'OK (it is in CORE)';
                        $out = $this->sgEstateCoreServices->deleteRealties($result->fetchAllAssociative());
                        $this->reportData['process'][$i]['importCoreOutput'] = $out;
                        $this->sgEstateCoreServices->deleteRealties($result->fetchAllAssociative());
                        $this->reportData['process'][$i]['importFileXMLImportStopDeleteByExternNr'] = 'OK (it is in CORE)';
                    } else {
                        //$this->reportData['process'][$i]['importRealtyForDeleteNotFound'] = 'ERROR (objektnr_intern)';
                        $this->reportData['process'][$i]['importRealtyForDeleteNotFound'] = 'ERROR (objektnr_extern)';
                    }
                    //$this->reportData['deleted'][] = strval($item->verwaltung_techn->objektnr_intern);
                    break;
                case 'CHANGE':
                default:
                    $this->reportData['process'][$i]['importFileXMLImportStartUpdateOrInsert'] = 'OK';
                    $log = [];
                    $propertyUid = $this->addProperty($item, $log);
                    if($propertyUid > 0) {
                        $this->reportData['process'][$i]['importUpdateOrInsert'][$propertyUid] = $log;
                        $this->reportData['process'][$i]['importFileXMLImportStopUpdateOrInsert'] = 'OK';
                        $this->reportData['process'][$i]['importFileXMLImportStartUpdateOrInsertForManyToOneRel'] = 'OK';
                        $this->addN1Connections($item, $propertyUid);
                        $this->reportData['process'][$i]['importFileXMLImportStopUpdateOrInsertForManyToOneRel'] = 'OK';
                        $this->reportData['process'][$i]['importFileXMLImportStartUpdateOrInsertForManyToManyRel'] = 'OK';
                        $this->addMnConnections($item, $propertyUid);
                        $this->reportData['process'][$i]['importFileXMLImportStopUpdateOrInsertForManyToManyRel'] = 'OK';
                        $this->reportData['process'][$i]['importFileXMLImportStartUpdateOrInsertForAttachments'] = 'OK';
                        $this->addAttachments($propertyUid, $item->anhaenge->children());
                        $this->reportData['process'][$i]['importFileXMLImportStopUpdateOrInsertForAttachments'] = 'OK';
                    } else {
                        $this->reportData['wrong'][$i]['externNr'] = $item->verwaltung_techn->objektnr_extern;
                        $this->reportData['wrong'][$i]['internNr'] = $item->verwaltung_techn->objektnr_intern;
                        $this->reportData['wrong'][$i]['fileName'] = $this->filenameDatafile;
                    }
                    break;

            }
        }
        $this->reportData['process'][$i]['importFileXMLImportReadXML_stop'] = 'OK';

        if ((string)$input->getArgument('logging') !== '') {
            $current = file_get_contents($this->logFile);
            $current .= "\n\n\n\n\n\n";
            $current .= json_encode(array('import' => $this->reportData), JSON_THROW_ON_ERROR);
            file_put_contents($this->logFile, $current);
        }
    }

    /**
     * @throws Exception
     * @throws \Exception
     */
    private function addProperty(SimpleXMLElement $item, array &$log=[]): mixed
    {
        $log['checkUpdateOrInsertStart'] = 'ok';
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $result = $queryBuilder->select('uid')
            ->from('tx_sgestatecore_domain_model_immobilie')
            //->where($queryBuilder->expr()->eq('objektnr_intern', $queryBuilder->createNamedParameter($item->verwaltung_techn->objektnr_intern)))
            ->where($queryBuilder->expr()->eq('objektnr_extern', $queryBuilder->createNamedParameter($item->verwaltung_techn->objektnr_extern)))->andWhere($queryBuilder->expr()->eq('import_number', $queryBuilder->createNamedParameter($this->importIdentifier)))->executeQuery();
        $log['checkUpdateOrInsertStop'] = 'ok';
        $propertyData = $this->generatePropertyData($item);
        if ($this->inputChecker($propertyData) === 0) {
            // need for sitemap.xml
            $propertyData['tstamp'] = strtotime(date('d.m.Y H:i:s'));
            if ($result->rowCount() >= 1) {
                $log['STATUS'] = 'UPDATE';
                // Update Property
                $uidProperty = $result->fetchAssociative()['uid'];
                $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
                $connectionPoolForProperties->update(
                    'tx_sgestatecore_domain_model_immobilie',
                    $propertyData,
                    ['uid' => $uidProperty]
                );
                //$this->reportData['updated'][] = strval($item->verwaltung_techn->objektnr_intern);
                $this->reportData['updated'][] = (string)$item->verwaltung_techn->objektnr_extern;
            } else {
                $log['STATUS'] = 'INSERT';
                // Add new Property
                $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
                $connectionPoolForProperties->insert(
                    'tx_sgestatecore_domain_model_immobilie',
                    $propertyData
                );
                $uidProperty = (int)$connectionPoolForProperties->lastInsertId();
                //$this->reportData['inserted'][] = strval($item->verwaltung_techn->objektnr_intern);
                $this->reportData['inserted'][] = (string)$item->verwaltung_techn->objektnr_extern;
            }
            return $uidProperty;
        }
        return 0;
    }

    /**
     * @throws Exception|DateMalformedStringException
     */
    private function generatePropertyData(SimpleXMLElement $item): array
    {
        // @todo: objektart_zusatz
        // @todo: unterkellert
        // @todo: stp_carport varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_duplex varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_freiplatz varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_garage varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_parkhaus varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_tiefgarage varchar(255) DEFAULT '' NOT NULL,
        // @todo: user_defined_simple int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: user_defined_extend int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: $temp['zulieferung'] = $this->getBoolValue($item->infrastruktur->zulieferung);

        $temp = [];
        $temp['pid'] = $this->storagePid;
        $temp['updatable'] = 1;
        $temp['import_number'] = $this->importIdentifier;
        foreach ($item->children() as $child1) {
            switch ($child1->getName()) {
                case 'geo':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'plz':
                                $temp['objekt_plz'] = (string)$child2;
                                break;
                            case 'strasse':
                                $temp['objekt_strasse'] = (string)$child2;
                                break;
                            case 'hausnummer':
                                $temp['objekt_hausnummer'] = (string)$child2;
                                break;
                            case 'etage':
                                $temp['objekt_etage'] = (int)$child2;
                                break;
                            case 'lage_im_bau':
                                $temp['objekt_lage_im_bau_links'] = $this->getBoolValue($child2->attributes()->LINKS);
                                $temp['objekt_lage_im_bau_rechts'] = $this->getBoolValue($child2->attributes()->RECHTS);
                                $temp['objekt_lage_im_bau_vorne'] = $this->getBoolValue($child2->attributes()->VORNE);
                                $temp['objekt_lage_im_bau_hinten'] = $this->getBoolValue($child2->attributes()->HINTEN);
                                break;
                            case 'regionaler_zusatz':
                                $temp['objekt_regionaler_zusatz'] = (string)$child2;
                                break;
                            case 'bundesland':
                                $temp['objekt_bundesland'] = (string)$child2;
                                break;
                            case 'gemeindecode':
                                $temp['objekt_gemeindecode'] = (string)$child2;
                                break;
                            case 'flur':
                                $temp['objekt_flur'] = (string)$child2;
                                break;
                            case 'flustueck':
                                $temp['objekt_flurstueck'] = (string)$child2;
                                break;
                            case 'gemarkung':
                                $temp['objekt_gemarkung'] = (string)$child2;
                                break;
                            case 'anzahl_etagen':
                                $temp['objekt_anzahl_etagen'] = (int)$child2;
                                break;
                            case 'wohnungsnr':
                                $temp['objekt_wohnungsnr'] = (string)$child2;
                                break;
                            case 'karten_makro':
                                $temp['objekt_karten_makro'] = $this->getBoolValue($child2);
                                break;
                            case 'karten_mikro':
                                $temp['objekt_karten_mikro'] = $this->getBoolValue($child2);
                                break;
                            case 'virtuelletour':
                                $temp['objekt_virtuelletour'] = $this->getBoolValue($child2);
                                break;
                            case 'luftbilder':
                                $temp['objekt_luftbilder'] = $this->getBoolValue($child2);
                                break;
                            case 'geokoordinaten':
                                $temp['objekt_breitengrad'] = str_replace(',', '.', (string)$child2->attributes()->breitengrad);
                                $temp['objekt_laengengrad'] = str_replace(',', '.', (string)$child2->attributes()->laengengrad);
                                break;
                            case 'ort':
                            case 'land':
                                break;
                        }
                    }
                    break;
                case 'preise':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'kaufpreis':
                                $temp['kaufpreis'] = (float)$child2;
                                break;
                            case 'kaufpreisnetto':
                                $temp['kaufpreis_netto'] = (float)$child2;
                                $temp['kaufpreis_netto_ust'] = (float)$child2->attributes()->kaufpreisust;
                                break;
                            case 'kaufpreisbrutto':
                                $temp['kaufpreis_brutto'] = (float)$child2;
                                break;
                            case 'nettokaltmiete':
                                $temp['nettokaltmiete'] = (float)$child2;
                                break;
                            case 'kaltmiete':
                                $temp['kaltmiete'] = (float)$child2;
                                break;
                            case 'warmmiete':
                                $temp['warmmiete'] = (float)$child2;
                                break;
                            case 'nebenkosten':
                                $temp['nebenkosten'] = (float)$child2;
                                break;
                            case 'heizkosten_enthalten':
                                $temp['heizkosten_enthalten'] = $this->getBoolValue($child2);
                                break;
                            case 'heizkosten':
                                $temp['heizkosten'] = (float)$child2;
                                break;
                            case 'zzg_mehrwertsteuer':
                                $temp['zzgl_mehrwertsteuer'] = $this->getBoolValue($child2);
                                break;
                            case 'mietzuschlaege':
                                $temp['mietzuschlaege'] = (float)$child2;
                                break;
                            case 'hauptmietzinsnetto':
                                $temp['hauptmietzins_netto'] = (float)$child2;
                                $temp['hauptmietzins_netto_ust'] = (float)$child2->attributes()->hauptmietzinsust;
                                break;
                            case 'pauschalmiete':
                                $temp['pauschalmiete'] = (float)$child2;
                                break;
                            case 'betriebskostennetto':
                                $temp['betriebskosten_netto'] = (float)$child2;
                                $temp['betriebskosten_netto_ust'] = (float)$child2->attributes()->betriebskostenust;
                                break;
                            case 'evbnetto':
                                $temp['evb_netto'] = (float)$child2;
                                $temp['evb_netto_ust'] = (float)$child2;
                                break;
                            case 'gesamtmietenetto':
                                $temp['gesamtmiete_netto'] = (float)$child2;
                                $temp['gesamtmiete_netto_ust'] = (float)$child2->attributes()->gesamtmieteust;
                                break;
                            case 'gesamtmietebrutto':
                                $temp['gesamtmiete_brutto'] = (float)$child2;
                                break;
                            case 'gesamtbelastungnetto':
                                $temp['gesamtbelastung_netto'] = (float)$child2;
                                $temp['gesamtbelastung_netto_ust'] = (float)$child2->attributes()->gesamtbelastungust;
                                break;
                            case 'gesamtbelastungbrutto':
                                $temp['gesamtbelastung_brutto'] = (float)$child2;
                                break;
                            case 'gesamtkostenprom2von':
                                $temp['gesamtkosten_pro_m2_von'] = (float)$child2;
                                $temp['gesamtkosten_pro_m2_bis'] = (float)$child2->attributes()->gesamtkostenprom2bis;
                                break;
                            case 'heizkostennetto':
                                $temp['heizkosten_netto'] = (float)$child2;
                                $temp['heizkosten_netto_ust'] = (float)$child2->attributes()->heizkostenust;
                                break;
                            case 'monatlichekostennetto':
                                $temp['monatlichekosten_netto'] = (float)$child2;
                                $temp['monatlichekosten_netto_ust'] = (float)$child2->attributes()->monatlichekostenust;
                                break;
                            case 'monatlichekostenbrutto':
                                $temp['monatlichekostenbrutto'] = (float)$child2;
                                break;
                            case 'nebenkostenprom2von':
                                $temp['nebenkosten_pro_m2_von'] = (float)$child2;
                                $temp['nebenkosten_pro_m2_bis'] = (float)$child2->attributes()->nebenkostenprom2bis;
                                break;
                            case 'ruecklagenetto':
                                $temp['ruecklage_netto'] = (float)$child2;
                                $temp['ruecklage_netto_ust'] = (float)$child2->attributes()->ruecklageust;
                                break;
                            case 'sonstigekostennetto':
                                $temp['sonstigekosten_netto'] = (float)$child2;
                                $temp['sonstigekosten_netto_ust'] = (float)$child2->attributes()->sonstigekostenust;
                                break;
                            case 'sonstigemietenetto':
                                $temp['sonstigemiete_netto'] = (float)$child2;
                                $temp['sonstigemiete_netto_ust'] = (float)$child2->attributes()->sonstigemieteust;
                                break;
                            case 'summemietenetto':
                                $temp['summemiete_netto'] = (float)$child2;
                                $temp['summemiete_netto_ust'] = (float)$child2->attributes()->sonstigemieteust;
                                break;
                            case 'nettomieteprom2von':
                                $temp['nettomiete_pro_m2_von'] = (float)$child2;
                                $temp['nettomiete_pro_m2_bis'] = (float)$child2->attributes()->nettomieteprom2bis;
                                break;
                            case 'pacht':
                                $temp['pacht'] = (float)$child2;
                                break;
                            case 'erbpacht':
                                $temp['erbpacht'] = (float)$child2;
                                break;
                            case 'hausgeld':
                                $temp['hausgeld'] = (float)$child2;
                                break;
                            case 'abstand':
                                $temp['abstand'] = (float)$child2;
                                break;
                            case 'preis_zeitraum_von':
                                $temp['preis_zeitraum_von'] = $this->getDateValue($child2);
                                break;
                            case 'preis_zeitraum_bis':
                                $temp['preis_zeitraum_bis'] = $this->getDateValue($child2);
                                break;
                            case 'mietpreis_pro_qm':
                                $temp['mietpreis_pro_m2'] = (float)$child2;
                                break;
                            case 'kaufpreis_pro_qm':
                                $temp['kaufpreis_pro_m2'] = (float)$child2;
                                break;
                            case 'provisionspflichtig':
                                $temp['provisionspflichtig'] = $this->getBoolValue($child2);
                                break;
                            case 'innen_courtage':
                                $temp['innencourtage'] = (string)$child2;
                                $temp['innencourtage_mit_mwst'] = $this->getBoolValue($child2->attributes()->mit_mwst);
                                break;
                            case 'aussen_courtage':
                                $temp['aussencourtage'] = (string)$child2;
                                $temp['aussencourtage_mit_mwst'] = $this->getBoolValue($child2->attributes()->mit_mwst);
                                break;
                            case 'courtage_hinweis':
                                $temp['courtage_hinweis'] = (string)$child2;
                                break;
                            case 'provisionnetto':
                                $temp['provision_netto'] = (float)$child2;
                                $temp['provision_netto_ust'] = (float)$child2->attributes()->provisionust;
                                break;
                            case 'provisionbrutto':
                                $temp['provision_brutto'] = (float)$child2;
                                break;
                            case 'mwst_satz':
                                $temp['mwst_satz'] = (float)$child2;
                                break;
                            case 'mwst_gesamt':
                                $temp['mwst_gesamt'] = (float)$child2;
                                break;
                            case 'freitext_preis':
                                $temp['freitext_preis'] = nl2br($child2);
                                break;
                            case 'x_fache':
                                $temp['xfache'] = (string)$child2;
                                break;
                            case 'nettorendite':
                                $temp['nettorendite'] = (float)$child2;
                                break;
                            case 'nettorendite_soll':
                                $temp['nettorendite_soll'] = (float)$child2;
                                break;
                            case 'nettorendite_ist':
                                $temp['nettorendite_ist'] = (float)$child2;
                                break;
                            case 'mieteinnahmen_ist':
                                $temp['mieteinnahmen_ist'] = (float)$child2;
                                break;
                            case 'mieteinnahmen_soll':
                                $temp['mieteinnahmen_soll'] = (float)$child2;
                                break;
                            case 'erschliessungskosten':
                                $temp['erschliessungskosten'] = (float)$child2;
                                break;
                            case 'kaution':
                                $temp['kaution'] = (float)$child2;
                                break;
                            case 'kaution_text':
                                $temp['kaution_text'] = (string)$child2;
                                break;
                            case 'geschaeftsguthaben':
                                $temp['geschaeftsguthaben'] = (float)$child2;
                                break;
                            case 'richtpreis':
                                $temp['richtpreis'] = (float)$child2;
                                break;
                            case 'richtpreisprom2':
                                $temp['richtpreis_pro_m2'] = (float)$child2;
                                break;
                            case 'user_defined_simplefield':
                                $attribute = (string)$child2->attributes()->feldname;
                                switch ($attribute) {
                                    case 'anteile':
                                        $temp['genossenschaftsanteile'] = (float)$child2;
                                        break;
                                    case 'vertragsabschlussgebuehr':
                                        $temp['vertragsabschlussgebuehr'] = (float)$child2;
                                        break;
                                    default:
                                        //echo var_dump($child2);die();
                                        break;
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'flaechen':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'wohnflaeche':
                                $temp['wohnflaeche'] = (float)$item->flaechen->wohnflaeche;
                                break;
                            case 'nutzflaeche':
                                $temp['nutzflaeche'] = (float)$item->flaechen->nutzflaeche;
                                break;
                            case 'gesamtflaeche':
                                $temp['gesamtflaeche'] = (float)$item->flaechen->gesamtflaeche;
                                break;
                            case 'ladenflaeche':
                                $temp['ladenflaeche'] = (float)$item->flaechen->ladenflaeche;
                                break;
                            case 'lagerflaeche':
                                $temp['lagerflaeche'] = (float)$item->flaechen->lagerflaeche;
                                break;
                            case 'verkaufsflaeche':
                                $temp['verkaufsflaeche'] = (float)$item->flaechen->verkaufsflaeche;
                                break;
                            case 'freiflaeche':
                                $temp['freiflaeche'] = (float)$item->flaechen->freiflaeche;
                                break;
                            case 'bueroflaeche':
                                $temp['bueroflaeche'] = (float)$item->flaechen->bueroflaeche;
                                break;
                            case 'bueroteilflaeche':
                                $temp['bueroteilflaeche'] = (float)$item->flaechen->bueroteilflaeche;
                                break;
                            case 'fensterfront':
                                $temp['fensterfront'] = (float)$item->flaechen->fensterfront;
                                break;
                            case 'verwaltungsflaeche':
                                $temp['verwaltungsflaeche'] = (float)$item->flaechen->verwaltungsflaeche;
                                break;
                            case 'gastroflaeche':
                                $temp['gastroflaeche'] = (float)$item->flaechen->gastroflaeche;
                                break;
                            case 'grz':
                                $temp['grundflaechenzahl'] = (string)$item->flaechen->grz;
                                break;
                            case 'gfz':
                                $temp['geschossflaechenzahl'] = (string)$item->flaechen->gfz;
                                break;
                            case 'bmz':
                                $temp['baumassenzahl'] = (string)$item->flaechen->bmz;
                                break;
                            case 'bgf':
                                $temp['bruttogeschossflaechenzahl'] = (float)$item->flaechen->bgf;
                                break;
                            case 'grundstuecksflaeche':
                                $temp['grundstuecksflaeche'] = (float)$item->flaechen->grundstuecksflaeche;
                                break;
                            case 'sonstflaeche':
                                $temp['sonstflaeche'] = (float)$item->flaechen->sonstflaeche;
                                break;
                            case 'anzahl_zimmer':
                                $temp['anzahl_zimmer'] = (float)$item->flaechen->anzahl_zimmer;
                                break;
                            case 'anzahl_schlafzimmer':
                                $temp['anzahl_schlafzimmer'] = (float)$item->flaechen->anzahl_schlafzimmer;
                                break;
                            case 'anzahl_badezimmer':
                                $temp['anzahl_badezimmer'] = (float)$item->flaechen->anzahl_badezimmer;
                                break;
                            case 'anzahl_sep_wc':
                                $temp['anzahl_separate_wc'] = (float)$item->flaechen->anzahl_sep_wc;
                                break;
                            case 'balkon_terrasse_flaeche':
                                $temp['balkon_terasse_flaeche'] = (float)$item->flaechen->balkon_terrasse_flaeche;
                                break;
                            case 'anzahl_wohn_schlafzimmer':
                                $temp['anzahl_wohn_schlafzimmer'] = (float)$item->flaechen->anzahl_wohn_schlafzimmer;
                                break;
                            case 'gartenflaeche':
                                $temp['gartenflaeche'] = (float)$item->flaechen->gartenflaeche;
                                break;
                            case 'anzahl_balkone':
                                $temp['anzahl_balkone'] = (float)$item->flaechen->anzahl_balkone;
                                break;
                            case 'anzahl_terrassen':
                                $temp['anzahl_terassen'] = (float)$item->flaechen->anzahl_terrassen;
                                break;
                            case 'anzahl_logia':
                                $temp['anzahl_logia'] = (float)$item->flaechen->anzahl_logia;
                                break;
                            case 'fensterfront_qm':
                                $temp['fensterfront_m2'] = (float)$item->flaechen->fensterfront_qm;
                                break;
                            case 'grundstuecksfront':
                                $temp['grundstuecksfront'] = (float)$item->flaechen->grundstuecksfront;
                                break;
                            case 'dachbodenflaeche':
                                $temp['dachbodenflaeche'] = (float)$item->flaechen->dachbodenflaeche;
                                break;
                            case 'teilbar_ab':
                                $temp['teilbar_ab'] = (float)$item->flaechen->teilbar_ab;
                                break;
                            case 'kellerflaeche':
                                $temp['kellerflaeche'] = (float)$item->flaechen->kellerflaeche;
                                break;
                            case 'beheizbare_flaeche':
                                $temp['beheizbare_flaeche'] = (float)$item->flaechen->beheizbare_flaeche;
                                break;
                            case 'anzahl_stellplaetze':
                                $temp['anzahl_stellplaetze'] = (float)$item->flaechen->anzahl_stellplaetze;
                                break;
                            case 'plaetze_gastraum':
                                $temp['plaetze_gastraum'] = (float)$item->flaechen->plaetze_gastraum;
                                break;
                            case 'anzahl_betten':
                                $temp['anzahl_betten'] = (float)$item->flaechen->anzahl_betten;
                                break;
                            case 'anzahl_tagungsraeume':
                                $temp['anzahl_tagungsraeume'] = (float)$item->flaechen->anzahl_tagungsraeume;
                                break;
                            case 'vermietbare_flaeche':
                                $temp['vermietbare_flaeche'] = (float)$item->flaechen->vermietbare_flaeche;
                                break;
                            case 'anzahl_wohneinheiten':
                                $temp['anzahl_wohneinheiten'] = (float)$item->flaechen->anzahl_wohneinheiten;
                                break;
                            case 'anzahl_gewerbeeinheiten':
                                $temp['anzahl_gewerbeeinheiten'] = (float)$item->flaechen->anzahl_gewerbeeinheiten;
                                break;
                            case 'einliegerwohnung':
                                $temp['einliegerwohnung'] = $this->getBoolValue($item->flaechen->einliegerwohnung);
                                break;
                            case 'kubatur':
                                $temp['kubatur'] = (float)$item->flaechen->kubatur;
                                break;
                            case 'ausnuetzungsziffer':
                                $temp['ausnuetzungsziffer'] = (float)$item->flaechen->ausnuetzungsziffer;
                                break;
                            case 'flaechevon':
                                $temp['flaechevon'] = (float)$item->flaechen->flaechevon;
                                break;
                            case 'flaechebis':
                                $temp['flaechebis'] = (float)$item->flaechen->flaechebis;
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'ausstattung':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'wg_geeignet':
                                $temp['wg_geeignet'] = $this->getBoolValue($item->ausstattung->wg_geeignet);
                                break;
                            case 'raeume_veraenderbar':
                                $temp['raeume_veraenderbar'] = $this->getBoolValue($item->ausstattung->raeume_veraenderbar);
                                break;
                            case 'kamin':
                                $temp['kamin'] = $this->getBoolValue($item->ausstattung->kamin);
                                break;
                            case 'klimatisiert':
                                $temp['klimatisiert'] = $this->getBoolValue($item->ausstattung->klimatisiert);
                                break;
                            case 'fahrstuhl':
                                $temp['aufzug_personen'] = $this->getBoolValue($item->ausstattung->fahrstuhl->attributes()->PERSONEN);
                                //$temp['aufzug_personen'] = $this->getBoolValue($item->ausstattung->fahrstuhl->attributes()->LASTEN);
                                break;
                            case 'gartennutzung':
                                $temp['gartennutzung'] = $this->getBoolValue($item->ausstattung->gartennutzung);
                                break;
                            case 'moebliert':
                                $temp['moebliert'] = (string)$item->ausstattung->moebliert->attributes()->moeb;
                                break;
                            case 'rollstuhlgerecht':
                                $temp['rollstuhlgerecht'] = $this->getBoolValue($item->ausstattung->rollstuhlgerecht);
                                break;
                            case 'kabel_sat_tv':
                                $temp['kabel_sat_tv'] = $this->getBoolValue($item->ausstattung->kabel_sat_tv);
                                $temp['kabelfernsehen'] = $this->getBoolValue($item->ausstattung->kabel_sat_tv);
                                break;
                            case 'dvbt':
                                $temp['dvbt'] = $this->getBoolValue($item->ausstattung->dvbt);
                                break;
                            case 'barrierefrei':
                                $temp['barrierefrei'] = $this->getBoolValue($item->ausstattung->barrierefrei);
                                break;
                            case 'sauna':
                                $temp['sauna'] = $this->getBoolValue($item->ausstattung->sauna);
                                break;
                            case 'swimmingpool':
                                $temp['swimmingpool'] = $this->getBoolValue($item->ausstattung->swimmingpool);
                                break;
                            case 'wasch_trockenraum':
                                $temp['wasch_trockenraum'] = $this->getBoolValue($item->ausstattung->wasch_trockenraum);
                                break;
                            case 'wintergarten':
                                $temp['wintergarten'] = $this->getBoolValue($item->ausstattung->wintergarten);
                                break;
                            case 'dv_verkabelung':
                                $temp['dv_verkabelung'] = $this->getBoolValue($item->ausstattung->dv_verkabelung);
                                break;
                            case 'rampe':
                                $temp['rampe'] = $this->getBoolValue($item->ausstattung->rampe);
                                break;
                            case 'hebebuehne':
                                $temp['hebebuehne'] = $this->getBoolValue($item->ausstattung->hebebuehne);
                                break;
                            case 'kran':
                                $temp['kran'] = $this->getBoolValue($item->ausstattung->kran);
                                break;
                            case 'gastterrasse':
                                $temp['gastterrasse'] = $this->getBoolValue($item->ausstattung->gastterrasse);
                                break;
                            case 'stromanschlusswert':
                                $temp['stromanschlusswert'] = (float)$item->ausstattung->stromanschlusswert;
                                break;
                            case 'kantine_cafeteria':
                                $temp['katine_cafeteria'] = $this->getBoolValue($item->ausstattung->kantine_cafeteria);
                                break;
                            case 'teekueche':
                                $temp['teekueche'] = $this->getBoolValue($item->ausstattung->teekueche);
                                break;
                            case 'hallenhoehe':
                                $temp['hallenhoehe'] = (float)$item->ausstattung->hallenhoehe;
                                break;
                            case 'angeschl_gastronomie':
                                $temp['hotelrestaurant_angeschlossen'] = $this->getBoolValue($item->ausstattung->angeschl_gastronomie->attributes()->HOTELRESTAURANT);
                                $temp['bar_angeschlossen'] = $this->getBoolValue($item->ausstattung->angeschl_gastronomie->attributes()->BAR);
                                break;
                            case 'brauereibindung':
                                $temp['brauereibindung'] = $this->getBoolValue($item->ausstattung->brauereibindung);
                                break;
                            case 'sporteinrichtungen':
                                $temp['sporteinrichtungen'] = $this->getBoolValue($item->ausstattung->sporteinrichtungen);
                                break;
                            case 'wellnessbereich':
                                $temp['wellnessbereich'] = $this->getBoolValue($item->ausstattung->wellnessbereich);
                                break;
                            case 'serviceleistungen':
                                $temp['service_betreutes_wohnen'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->BETREUTES_WOHNEN);
                                $temp['service_catering'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->CATERING);
                                $temp['service_reinigung'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->REINIGUNG);
                                $temp['service_einkauf'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->EINKAUF);
                                $temp['service_wachdienst'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->WACHDIENST);
                                break;
                            case 'telefon_ferienimmobilie':
                                $temp['telefon_ferienimmobilie'] = $this->getBoolValue($item->ausstattung->telefon_ferienimmobilie);
                                break;
                            case 'breitband_zugang':
                                $temp['breitband_zugang_art'] = (string)$item->ausstattung->breitband_zugang->attributes()->art;
                                $temp['breitband_zugang_speed'] = (float)$item->ausstattung->breitband_zugang->attributes()->speed;
                                break;
                            case 'umts_empfang':
                                $temp['umts_empfang'] = $this->getBoolValue($item->ausstattung->umts_empfang);
                                break;
                            case 'sicherheitstechnik':
                                $temp['sicherheit_alarmanlage'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->ALARMANLAGE);
                                $temp['sicherheit_kamera'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->KAMERA);
                                $temp['sicherheit_polizeiruf'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->POLIZEIRUF);
                                break;
                            case 'abstellraum':
                                $temp['abstellraum'] = $this->getBoolValue($item->ausstattung->abstellraum);
                                break;
                            case 'fahrradraum':
                                $temp['fahrradraum'] = $this->getBoolValue($item->ausstattung->fahrradraum);
                                break;
                            case 'rolladen':
                                $temp['rolladen'] = $this->getBoolValue($item->ausstattung->rolladen);
                                break;
                            case 'bibliothek':
                                $temp['bibliothek'] = $this->getBoolValue($item->ausstattung->bibliothek);
                                break;
                            case 'dachboden':
                                $temp['dachboden'] = $this->getBoolValue($item->ausstattung->dachboden);
                                break;
                            case 'gaestewc':
                                $temp['gaestewc'] = $this->getBoolValue($item->ausstattung->gaestewc);
                                break;
                            case 'kabelkanaele':
                                $temp['kabelkanaele'] = $this->getBoolValue($item->ausstattung->kabelkanaele);
                                break;
                            case 'seniorengerecht':
                                $temp['seniorengerecht'] = $this->getBoolValue($item->ausstattung->seniorengerecht);
                                break;
                            case 'unterkellert':
                                $temp['unterkellert'] = $this->getBoolValue($item->ausstattung->unterkellert->attributes()->keller);
                                break;
                            case 'user_defined_simplefield':
                                foreach ($child2->attributes() as $attribute) {
                                    switch ((string)$attribute) {
                                        case 'gegensprechanlage':
                                            $temp['gegensprechanlage'] = $this->getBoolValue($child2);
                                            break;
                                        case 'leicht_erreichbar':
                                            $temp['leicht_erreichbar'] = $this->getBoolValue($child2);
                                            break;
                                        case 'bad_mit_fenster':
                                            if (!$child1->bad) {
                                                $child1->addChild('bad');
                                            }
                                            if ($child2 == 'true') {
                                                $child1->bad->addAttribute('FENSTER', 'true');
                                            }
                                            break;
                                        case 'wunschwohnung':
                                            if (!($child1->kampagne)) {
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true') {
                                                $child1->kampagne->addAttribute('WUNSCHWOHNUNG', 'true');
                                            }
                                            break;
                                        case 'junges_wohnen':
                                            if (!($child1->kampagne)) {
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true') {
                                                $child1->kampagne->addAttribute('JUNGESWOHNEN', 'true');
                                            }
                                            break;
                                        case 'bad_modernisiert':
                                            $temp['bad_modernisiert'] = $this->getBoolValue($child2);
                                            break;
                                        case 'hd_satellitenfernsehen':
                                            $temp['satellitenfernsehen'] = $this->getBoolValue($child2);
                                            break;
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'zustand_angaben':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'baujahr':
                                $temp['baujahr'] = (string)$item->zustand_angaben->baujahr;
                                break;
                            case 'letztemodernisierung':
                                $temp['letztemodernisierung'] = (string)$item->zustand_angaben->letztemodernisierung;
                                break;
                            case 'bauzone':
                                $temp['bauzone'] = (string)$item->zustand_angaben->bauzone;
                                break;
                            case 'altlasten':
                                $temp['altlasten'] = (string)$item->zustand_angaben->altlasten;
                                break;
                            case 'energiepass':
                                $temp['energiepass_energieverbrauchkennwert'] = (string)$item->zustand_angaben->energiepass->energieverbrauchkennwert;
                                $temp['energiepass_mitwarmwasser'] = $this->getBoolValue($item->zustand_angaben->energiepass->mitwarmwasser);
                                $temp['energiepass_endenergiebedarf'] = (string)$item->zustand_angaben->energiepass->endenergiebedarf;
                                $temp['energiepass_baujahr'] = (string)$item->zustand_angaben->energiepass->baujahr;
                                $temp['energiepass_wertklasse'] = (string)$item->zustand_angaben->energiepass->wertklasse;
                                $temp['energiepass_ausstelldatum'] = $this->getDateValue($item->zustand_angaben->energiepass->ausstelldatum);

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassart')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->energiepass->epart)))->executeQuery();

                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['energiepass_art'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_befeuerungsart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_befeuerungsart')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->energiepass->primaerenergietraeger)))->executeQuery();

                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['energiepass_primaerenergietraeger'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassjahrgang');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassjahrgang')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->energiepass->jahrgang)))->executeQuery();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['energiepass_jahrgang'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassgebaeudeart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassgebaeudeart')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->energiepass->gebaeudeart)))->executeQuery();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['energiepass_gebaeudeart'] = $resultItemUid;
                                }
                                $result = null;

                                break;
                            case 'alter':
                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_altertyp');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_altertyp')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->alter->attributes()->alter_attr)))->executeQuery();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['altertyp'] = $resultItemUid;
                                }
                                $result = null;
                                break;
                            case 'verkaufstatus':
                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_verkaufstatus');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_verkaufstatus')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->verkaufstatus)))->executeQuery();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                                    $temp['verkaufstatus'] = $resultItemUid;
                                }
                                $result = null;
                                break;
                            case 'user_defined_simplefield':
                            default:
                                break;
                        }
                    }
                    break;
                case 'freitexte':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'objekttitel':
                                $temp['objekttitel'] = nl2br($item->freitexte->objekttitel);
                                break;
                            case 'dreizeiler':
                                $temp['kurzbeschreibung'] = nl2br($item->freitexte->dreizeiler);
                                break;
                            case 'lage':
                                $temp['lage'] = nl2br($item->freitexte->lage);
                                break;
                            case 'ausstatt_beschr':
                                $temp['austattungsbeschreibung'] = nl2br($item->freitexte->ausstatt_beschr);
                                break;
                            case 'objektbeschreibung':
                                $temp['objektbeschreibung'] = nl2br($item->freitexte->objektbeschreibung);
                                break;
                            case 'sonstige_angaben':
                                $temp['sonstige_angaben'] = nl2br($item->freitexte->sonstige_angaben);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_objekt':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'objektadresse_freigeben':
                                $temp['objektadresse_freigeben'] = $this->getBoolValue($item->verwaltung_objekt->objektadresse_freigeben);
                                break;
                            case 'verfuegbar_ab':
                                $temp['verfuegbar_ab'] = (string)$item->verwaltung_objekt->verfuegbar_ab;
                                break;
                            case 'abdatum':
                                $temp['verfuegbar_ab_datum'] = $this->getDateValue($item->verwaltung_objekt->abdatum);
                                break;
                            case 'bisdatum':
                                $temp['verfuegbar_bis_datum'] = $this->getDateValue($item->verwaltung_objekt->bisdatum);
                                break;
                            case 'versteigerungstermin':
                                $temp['versteigerungstermin'] = $this->getDateValue($item->verwaltung_objekt->versteigerungstermin);
                                break;
                            case 'wbs_sozialwohnung':
                                $temp['wbs_erforderlich'] = $this->getBoolValue($item->verwaltung_objekt->wbs_sozialwohnung);
                                break;
                            case 'vermietet':
                                $temp['vermietet'] = $this->getBoolValue($item->verwaltung_objekt->vermietet);
                                break;
                            case 'gruppennummer':
                                $temp['gruppennummer'] = (string)$item->verwaltung_objekt->gruppennummer;
                                break;
                            case 'zugang':
                                $temp['zugang'] = (string)$item->verwaltung_objekt->zugang;
                                break;
                            case 'laufzeit':
                                $temp['laufzeit'] = (float)$item->verwaltung_objekt->laufzeit;
                                break;
                            case 'max_personen':
                                $temp['maximale_personenanzahl'] = (int)$item->verwaltung_objekt->max_personen;
                                break;
                            case 'nichtraucher':
                                $temp['nichtraucher'] = $this->getBoolValue($item->verwaltung_objekt->nichtraucher);
                                break;
                            case 'haustiere':
                                $temp['haustiere'] = $this->getBoolValue($item->verwaltung_objekt->haustiere);
                                break;
                            case 'denkmalgeschuetzt':
                                $temp['denkmalgeschuetzt'] = $this->getBoolValue($item->verwaltung_objekt->denkmalgeschuetzt);
                                break;
                            case 'als_ferien':
                                $temp['als_ferien'] = $this->getBoolValue($item->verwaltung_objekt->als_ferien);
                                break;
                            case 'gewerbliche_nutzung':
                                $temp['gewerbliche_nutzung'] = $this->getBoolValue($item->verwaltung_objekt->gewerbliche_nutzung);
                                break;
                            case 'branchen':
                                $temp['branchen'] = (string)$item->verwaltung_objekt->branchen;
                                break;
                            case 'hochhaus':
                                $temp['hochhaus'] = $this->getBoolValue($item->verwaltung_objekt->hochhaus);
                                break;
                            case 'aktiv_von':
                                $temp['anzeige_ab'] = $this->getDateValue($item->verwaltung_objekt->aktiv_von);
                                break;
                            case 'aktiv_bis':
                                $temp['anzeige_bis'] = $this->getDateValue($item->verwaltung_objekt->aktiv_bis);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_techn':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'openimmo_obid':
                                $temp['openimmo_objid'] = (string)$item->verwaltung_techn->openimmo_obid;
                                break;
                            case 'objektnr_intern':
                                $temp['objektnr_intern'] = (string)$item->verwaltung_techn->objektnr_intern;
                                break;
                            case 'objektnr_extern':
                                $temp['objektnr_extern'] = (string)$item->verwaltung_techn->objektnr_extern;
                                break;
                            case 'kennung_ursprung':
                                $temp['kennung_ursprung'] = (string)$item->verwaltung_techn->kennung_ursprung;
                                break;
                            case 'stand_vom':
                                $temp['stand_vom'] = $this->getDateValue($item->verwaltung_techn->stand_vom);
                                break;
                            case 'weitergabe_generell':
                                $temp['weitergabe_generell'] = $this->getBoolValue($item->verwaltung_techn->weitergabe_generell);
                                break;
                            case 'weitergabe_positivliste':
                                $temp['weitergabe_positivliste'] = (string)$item->verwaltung_techn->weitergabe_positivliste;
                                break;
                            case 'weitergabe_negativliste':
                                $temp['weitergabe_negativliste'] = (string)$item->verwaltung_techn->weitergabe_negativliste;
                                break;
                            case 'gruppen_kennung':
                                $temp['gruppen_kennung'] = (string)$item->verwaltung_techn->gruppen_kennung;
                                break;
                            case 'master':
                                $temp['master_objekt'] = (string)$item->verwaltung_techn->master;
                                $temp['master_visible'] = $this->getBoolValue($item->verwaltung_techn->master->attributes()->visible);
                                break;
                            case 'sprache':
                                $temp['sprache'] = (string)$item->verwaltung_techn->sprache;
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'kontaktperson':
                case 'anhaenge':
                case 'objektkategorie':
                    break;
                default:
                    throw new UnexpectedValueException('Unexpected value');
            }
        }
        return $temp;
    }

    /**
     * @throws Exception
     */
    private function addN1Connections($item, $propertyUid): void
    {
        $updateArray = [];

        if (($item->infrastruktur->ausblick)&&((string)$item->infrastruktur->ausblick->attributes()->blick !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ausblick');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_ausblick')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->infrastruktur->ausblick->attributes()->blick)))->executeQuery();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['ausblick'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->geschlecht)&&((string)$item->verwaltung_objekt->geschlecht->attributes()->geschl_attr !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_geschlecht');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_geschlecht')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->verwaltung_objekt->geschlecht->attributes()->geschl_attr)))->executeQuery();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['geschlecht'] = $resultItemUid;
            }
            $result = null;
        }
        if ($item->objektkategorie->objektart) {
            foreach ($item->objektkategorie->objektart->children() as $child) {
                /**
                 * @var $child SimpleXMLElement
                 */
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektart');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_objektart')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strtoupper($child->getName()))))->executeQuery();
                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                    $updateArray['objektart'] = $resultItemUid;
                }




//                $_param = !empty(strval($child[$child->getName() . 'typ']))
//                    ? strval($child[$child->getName() . 'typ'])
//                    : !empty(strval($child[$child->getName() . '_typ']))
//                        ? strval($child[$child->getName() . '_typ'])
//                        : '';

                $_param = '';
                $attrs = $child->attributes();
                $search = $child->getName() . 'typ';
                $_search = $child->getName() . '_typ';
                if($attrs && !empty($attrs[$search])) {
                    $_param = $attrs[$search];
                }
                if($attrs && !empty($attrs[$_search])) {
                    $_param = $attrs[$_search];
                }


                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektarttyp');
                $result = $queryBuilder
                    ->select('uid')->from('tx_sgestatecore_domain_model_objektarttyp')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strtoupper($_param))))->executeQuery();
                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                    $updateArray['objektarttyp'] = $resultItemUid;
                }
            }
        }
        $updateArray['objekt_ort'] = $this->addCity((string)$item->geo->ort, true);
        if ((string)$item->geo->regionaler_zusatz === '') {
            foreach ($item->geo->children() as $subnode) {
                /**
                 * @var $subnode SimpleXMLElement
                 */
                if ($subnode->getName() === 'user_defined_simplefield') {
                    foreach ($subnode->attributes() as $attribute) {
                        switch ((string)$attribute) {
                            case 'SearchRegion': // Immosolve District
                            case 'ZONE': //Immoblue District
                                if ((string)$subnode !== '') {
                                    $updateArray['objekt_stadtteil'] = $this->addDistrict((string)$subnode, $updateArray['objekt_ort'], true);
                                }
                                break;
                        }
                    }
                }
            }
        } else {
            $updateArray['objekt_stadtteil'] = $this->addDistrict($item->geo->regionaler_zusatz, $updateArray['objekt_ort'], true);
        }

        if (($item->geo->land)&&((string)$item->geo->land->attributes()->iso_land !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_land');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_land')->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter((string)$item->geo->land->attributes()->iso_land)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['objekt_land'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->geo->lage_gebiet)&&((string)$item->geo->lage_gebiet->attributes()->gebiete !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_lage');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_lage')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->geo->lage_gebiet->attributes()->gebiete)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['objekt_lage'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->preis_zeiteinheit)&&((string)$item->preise->preis_zeiteinheit->attributes()->zeiteinheit !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->preise->preis_zeiteinheit->attributes()->zeiteinheit)))->executeQuery();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['preis_zeiteinheit'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_ist)&&((string)$item->preise->mieteinnahmen_ist->attributes()->periode !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->preise->mieteinnahmen_ist->attributes()->periode)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['mieteinnahmen_ist_periode'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_soll)&&((string)$item->preise->mieteinnahmen_soll->attributes()->periode !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->preise->mieteinnahmen_soll->attributes()->periode)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['mieteinnahmen_soll_periode'] = $resultItemUid;
            }
            $result = null;
        }
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_waehrung');
        if (($item->preise->waehrung)&&((string)$item->preise->waehrung->attributes()->iso_waehrung !== '')) {
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_waehrung')->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter((string)$item->preise->waehrung->attributes()->iso_waehrung)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['waehrung'] = $resultItemUid;
            }
            $result = null;
        } else {
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_waehrung')->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter('EUR')))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['waehrung'] = $resultItemUid;
            } else {
                die('Keine Währung angelegt.');
            }
        }
        if ((string)$item->ausstattung->ausstatt_kategorie !== '') {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_austattungskategorie');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_austattungskategorie')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->ausstattung->ausstatt_kategorie)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['ausstattungs_kategorie'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->zustand)&&((string)$item->zustand_angaben->zustand->attributes()->zustand_art !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_zustand');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_zustand')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->zustand->attributes()->zustand_art)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['zustand'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->bebaubar_nach)&&((string)$item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_bebaubarnach');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_bebaubarnach')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['bebaubar_nach'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung)&&((string)$item->zustand_angaben->erschliessung->attributes()->erschl_attr !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_erschliessung');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_erschliessung')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->erschliessung->attributes()->erschl_attr)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['erschliessung'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung_umfang)&&((string)$item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_erschliessungdetails');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_erschliessungdetails')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['erschliessung_umfang'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->min_mietdauer)&&((string)$item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer !== '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['minimale_mietdauer'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->max_mietdauer)&&((string)$item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer !== '')) {
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter((string)$item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer)))->executeQuery();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $updateArray['maximale_mietdauer'] = $resultItemUid;
            }
            $result = null;
        }
        $updateArray['kontaktperson'] = $this->addContactperson($item);

        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
        $connection->update(
            'tx_sgestatecore_domain_model_immobilie',
            $updateArray,
            ['uid' => (int)$propertyUid]
        );
    }

    /**
     * @throws Exception
     */
    private function addMnConnections($item, $propertyUid): void
    {
        if (is_object($item->objektkategorie->nutzungsart)) {
            $this->setMM($propertyUid, $item->objektkategorie->nutzungsart, 'tx_sgestatecore_domain_model_nutzungsart', 'tx_sgestatecore_immobilie_nutzungsart_mm');
        }

        if (is_object($item->objektkategorie->vermarktungsart)) {
            $this->setMM($propertyUid, $item->objektkategorie->vermarktungsart, 'tx_sgestatecore_domain_model_vermarktungsart', 'tx_sgestatecore_immobilie_vermarktungsart_mm');
        }

        if (is_object($item->ausstattung->bad)) {
            $this->setMM($propertyUid, $item->ausstattung->bad, 'tx_sgestatecore_domain_model_badausstattung', 'tx_sgestatecore_immobilie_badausstattung_mm');
        }

        if (is_object($item->ausstattung->kueche)) {
            $this->setMM($propertyUid, $item->ausstattung->kueche, 'tx_sgestatecore_domain_model_kuechenausstattung', 'tx_sgestatecore_immobilie_kuechenausstattung_mm');
        }

        if (is_object($item->ausstattung->boden)) {
            $this->setMM($propertyUid, $item->ausstattung->boden, 'tx_sgestatecore_domain_model_bodenbelag', 'tx_sgestatecore_immobilie_bodenbelag_mm');
        }

        if (is_object($item->ausstattung->heizungsart)) {
            $this->setMM($propertyUid, $item->ausstattung->heizungsart, 'tx_sgestatecore_domain_model_heizungsart', 'tx_sgestatecore_immobilie_heizungsart_mm');
        }

        if (is_object($item->ausstattung->befeuerung)) {
            $this->setMM($propertyUid, $item->ausstattung->befeuerung, 'tx_sgestatecore_domain_model_befeuerungsart', 'tx_sgestatecore_immobilie_befeuerungsart_mm');
        }

        // @todo: ausricht_balkon_terasse int(11) unsigned DEFAULT '0' NOT NULL, Anscheinend Himmelrichtung falsch verMMt
        // $this->setMM($propertyUid, $item->ausstattung->ausricht_balkon_terrasse, '', '');

        if (is_object($item->ausstattung->dachform)) {
            $this->setMM($propertyUid, $item->ausstattung->dachform, 'tx_sgestatecore_domain_model_dachform', 'tx_sgestatecore_immobilie_dachform_mm');
        }

        if (is_object($item->ausstattung->bauweise)) {
            $this->setMM($propertyUid, $item->ausstattung->bauweise, 'tx_sgestatecore_domain_model_bauweise', 'tx_sgestatecore_immobilie_bauweise_mm');
        }

        if (is_object($item->ausstattung->ausbaustufe)) {
            $this->setMM($propertyUid, $item->ausstattung->ausbaustufe, 'tx_sgestatecore_domain_model_ausbaustufe', 'tx_sgestatecore_immobilie_ausbaustufe_mm');
        }

        if (is_object($item->ausstattung->kampagne)) {
            $this->setMM($propertyUid, $item->ausstattung->kampagne, 'tx_sgestatecore_domain_model_kampagne', 'tx_sgestatecore_immobilie_kampagne_mm');
        }

        // @todo: energietyp int(11) unsigned DEFAULT '0', n:1 verknüpft auf M:n umstellen
        // $this->setMM($propertyUid, $item->ausstattung->energietyp, 'tx_sgestatecore_domain_model_energietyp', '');

        // @todo: weitere_adressen int(11) unsigned DEFAULT '0' NOT NULL,
    }

    /**
     * @throws Exception
     */
    private function setMM($propertyUid, SimpleXMLElement $xmlNode, $table, $tableMM): void
    {
        if ($xmlNode->attributes() !== null) {
            $kuerzel = [];

            foreach ($xmlNode->attributes() as $key => $value) {
                if (((string)$value === '1')||(strtolower((string)$value) === 'true')) {
                    $kuerzel[] = '"' . $key . '"';
                }
            }
            if (count($kuerzel)>0) {
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable($table);
                $result = $queryBuilder
                    ->select('uid')
                    ->from($table)->where($queryBuilder->expr()->in('kuerzel', implode(',', $kuerzel)))->executeQuery();
                $rows = $result->fetchAllAssociative();

                $queryBuilder = $this->connectionPool->getQueryBuilderForTable($tableMM);
                $queryBuilder
                    ->delete($tableMM)->where($queryBuilder->expr()->eq('uid_local', $queryBuilder->createNamedParameter($propertyUid)))->executeStatement();
                $connection = $this->connectionPool->getConnectionForTable($tableMM);
                foreach ($rows as $item) {
                    $connection->insert(
                        $tableMM,
                        [
                            'uid_local' => $propertyUid,
                            'uid_foreign' => $item['uid']
                        ]
                    );
                }
            }
        }
    }

    /**
     * @throws Exception
     */
    private function addAttachments($propertyUid, $xmlAttachments): void
    {
        if (is_numeric($propertyUid)) {
            // Remove old Attachments
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhang');
            $result = $queryBuilder
                ->select('uid', 'datei')
                ->from('tx_sgestatecore_domain_model_anhang')->where($queryBuilder->expr()->eq('immobilie', $queryBuilder->createNamedParameter((int)$propertyUid)))->executeQuery();
            $rows = $result->fetchAllAssociative();

            foreach ($rows as $item) {
                @unlink($this->mediaPath . $item['datei']);
                $queryBuilder
                    ->delete('tx_sgestatecore_domain_model_anhang')->where($queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($item['uid'])))->executeStatement();
            }

            if ($xmlAttachments) {
                foreach ($xmlAttachments as $attachment) {
                    /**
                     * @var $attachment SimpleXMLElement
                     */
                    $title = $format = $tempFile = $type = '';
                    $source = (string)$attachment->attributes()->location;
                    switch ($source) {
                        case 'REMOTE':
                        case 'INTERN':
                        case 'EXTERN':
                            $title = (string)$attachment->anhangtitel;
                            $format = (string)$attachment->format;
                            $tempFile = (string)$attachment->daten->pfad;
                            $type = (string)$attachment->attributes()->gruppe;
                            break;
                    }
                    $this->addAttachment($propertyUid, $title, $format, $tempFile, $type);
                }
            }
        }
    }

    /**
     * @throws Exception
     */
    private function addAttachment($propertyUid, string $title = '', string $format = '', string $tempFile = '', string $type = ''): void
    {
        if (($tempFile !== '')&&(is_numeric($propertyUid))) {
            if (in_array(strtoupper($format), ['JPG', 'GIF', 'PNG', 'PDF', 'IMAGE/JPEG', 'IMAGE/GIF', 'IMAGE/PNG', 'APPLICATION/PDF'])) {
                // @todo: Standardtyp
                if ($type === '') {
                    $type = 'DOKUMENTE';
                }
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhangtyp');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_anhangtyp')->where($queryBuilder->expr()->like('kuerzel', $queryBuilder->createNamedParameter($type)))->executeQuery();

                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                    $insertArray = [];
                    $insertArray['pid']         = $this->storagePid;
                    $insertArray['typ']         = $resultItemUid;
                    $insertArray['format']      = strtoupper($format);
                    $insertArray['titel']       = $title;
                    $insertArray['immobilie']   = $propertyUid;

                    $path_info = pathinfo($tempFile);
                    $new_filename = $path_info['filename'].'_'.uniqid('', true).'.'.$path_info['extension'];

                    if ((is_file($this->tempPath . $tempFile))&&(rename($this->tempPath . $tempFile, $this->mediaPath.$new_filename))) {
                        $insertArray['datei']       = $new_filename;
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                    }

                    if (is_file($this->mediaPath . $tempFile)) {
                        $insertArray['datei']       = $tempFile;
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                    }

                    $this->addAlert('Fehler bei der Anhangzuordnung', 'Datei konnte nicht kopiert werden');
                } else {
                    $this->addAlert('Fehler bei der Anhangzuordnung', 'Es wurde ein ungültiger Typ übergeben: ' . $type);
                }
            } else {
                if ($type = 'LINKS') {
                    $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhangtyp');
                    $result = $queryBuilder
                        ->select('uid')
                        ->from('tx_sgestatecore_domain_model_anhangtyp')->where($queryBuilder->expr()->like('kuerzel', $queryBuilder->createNamedParameter($type)))->executeQuery();

                    if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                        $insertArray = [];
                        $insertArray['pid']         = $this->storagePid;
                        $insertArray['typ']         = $resultItemUid;
                        $insertArray['format']      = strtoupper($format);
                        $insertArray['titel']       = $title;
                        $insertArray['immobilie']   = $propertyUid;

                        $insertArray['datei']       = $tempFile;
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                    }
                }
                $this->addAlert('Fehler bei der Anhangzuordnung', 'Der Anhang ist in keinem zulässigen Format');
            }
        } else {
            $this->addAlert('Fehler bei der Anhangzuordnung', 'Es wurde keine Immobilie übergeben');
        }
    }

    /**
     * @throws Exception
     */
    private function addCity($cityName, $useForSearch = null)
    {
        if ($cityName !== '') {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ort');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_ort')->where($queryBuilder->expr()->like('bezeichner', $queryBuilder->createNamedParameter($cityName)))->executeQuery();

            $data = [];
            $data['pid']        = $this->storagePid;
            $data['bezeichner'] = $cityName;
            if ($useForSearch === true) {
                $data['fuer_suche_verwenden'] = 1;
            }
            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_ort');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_ort',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_ort',
                    $data
                );
                $uid = (int)$connection->lastInsertId();
            }
            return $uid;
        }

        return null;
    }

    /**
     * @throws Exception
     */
    private function addDistrict($districtName, $cityUid, $useForSearch = null)
    {
        if (($districtName !== '')&&(is_numeric($cityUid))) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_stadtteil');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_stadtteil')
                ->where($queryBuilder->expr()->like('bezeichner', $queryBuilder->createNamedParameter($districtName)))->andWhere($queryBuilder->expr()->eq('ort', $queryBuilder->createNamedParameter($cityUid)))->executeQuery();

            $data = [];
            $data['pid']        = $this->storagePid;
            $data['ort']        = $cityUid;
            $data['bezeichner'] = $districtName;
            if ($useForSearch === true) {
                $data['fuer_suche_verwenden'] = 1;
            }

            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_stadtteil');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_stadtteil',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_stadtteil',
                    $data
                );
                $uid = (int)$connection->lastInsertId();
            }
            return $uid;
        }

        return null;
    }

    /**
     * @throws Exception
     */
    private function addContactperson(SimpleXMLElement $item)
    {
        $kontaktpersonDaten = $item->kontaktperson;
        if (is_object($kontaktpersonDaten)) {
            $data['pid']                            = $this->storagePid;
            $data['name']                           = (string)$kontaktpersonDaten->name;
            $data['vorname']                        = (string)$kontaktpersonDaten->vorname;
            $data['titel']                          = (string)$kontaktpersonDaten->titel;
            $data['anrede']                         = (string)$kontaktpersonDaten->anrede;
            $data['position']                       = (string)$kontaktpersonDaten->position;
            $data['anrede_brief']                   = (string)$kontaktpersonDaten->anrede_brief;
            $data['firma']                          = (string)$kontaktpersonDaten->firma;
            $data['zusatzfeld']                     = (string)$kontaktpersonDaten->zusatzfeld;
            $data['strasse']                        = (string)$kontaktpersonDaten->strasse;
            $data['hausnummer']                     = (string)$kontaktpersonDaten->hausnummer;
            $data['plz']                            = (string)$kontaktpersonDaten->plz;
            $data['postfach']                       = (string)$kontaktpersonDaten->postfach;
            $data['postfach_plz']                   = (string)$kontaktpersonDaten->postf_plz;
            $data['email_zentrale']                 = (string)$kontaktpersonDaten->email_zentrale;
            $data['email_direkt']                   = (string)$kontaktpersonDaten->email_direkt;
            $data['email_privat']                   = (string)$kontaktpersonDaten->email_privat;
            $data['email_sonstige']                 = (string)$kontaktpersonDaten->email_sonstige;
            $data['email_feedback']                 = (string)$kontaktpersonDaten->email_feedback;
            $data['telefon_zentrale']               = (string)$kontaktpersonDaten->tel_zentrale;
            $data['telefon_durchwahl']              = (string)$kontaktpersonDaten->tel_durchw;
            $data['telefon_fax']                    = (string)$kontaktpersonDaten->tel_fax;
            $data['telefon_handy']                  = (string)$kontaktpersonDaten->tel_handy;
            $data['telefon_privat']                 = (string)$kontaktpersonDaten->tel_privat;
            $data['telefon_sonstige']               = (string)$kontaktpersonDaten->tel_sonstige;
            $data['url']                            = (string)$kontaktpersonDaten->url;
            $data['adressfreigabe']                 = 0;
            $data['personennummer']                 = (string)$kontaktpersonDaten->personennummer;
            $data['immobilientreuhaenderid']        = (string)$kontaktpersonDaten->immobilientreuhaenderid;
            $data['freitextfeld']                   = (string)$kontaktpersonDaten->freitextfeld;
            $data['ort']                            = $this->addCity((string)$kontaktpersonDaten->ort);
            if ((!empty($kontaktpersonDaten->xpath('(land)[1]')))&&((string)$kontaktpersonDaten->land->attributes()->iso_land !== '')) {
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_land');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_land')->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter((string)$item->geo->land->attributes()->iso_land)))->executeQuery();
                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                    $data['land'] = $resultItemUid;
                }
                $result = null;
            }

            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_kontaktperson');
            $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_kontaktperson');

            // If a personnumber is provided this will be used for check if a contactperson is already present
            // if not firstname and lastname will be used
            if ((string)$kontaktpersonDaten->personennummer !== '') {
                $queryBuilder->where($queryBuilder->expr()->like('personennummer', $queryBuilder->createNamedParameter((string)$kontaktpersonDaten->personennummer)));
            } else {
                if ((string)$kontaktpersonDaten->vorname !== '') {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('vorname', $queryBuilder->createNamedParameter((string)$kontaktpersonDaten->vorname)));
                }
                if ((string)$kontaktpersonDaten->name !== '') {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('name', $queryBuilder->createNamedParameter((string)$kontaktpersonDaten->name)));
                }
            }
            $result = $queryBuilder->executeQuery();

            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_kontaktperson');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetchAssociative()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    $data
                );
                $uid = (int)$connection->lastInsertId();
            }
            return $uid;
        }

        return null;
    }

    private function getBoolValue($value): int
    {
        if ((string)$value !== '') {
            if ((strtolower((string)$value) === 'true')||(strtolower((string)$value) === 'ja')||(strtolower((string)$value) === 'j')) {
                return 1;
            }

            if ((strtolower((string)$value) === 'false')||(strtolower((string)$value) === 'n')) {
                return -1;
            }

            return 0;
        }

        return 0;
    }

    /**
     * @throws DateMalformedStringException
     */
    private function getDateValue($value): ?string
    {
        return (new DateTime($value))->format('U');
    }

    /**
     * Unzips filenameZip to pathExtract
     */
    private function unzip(string $filenameZip, string $pathExtract): bool
    {
        $zip = new ZipArchive;
        if ($zip->open($filenameZip) === true) {
            $zip->extractTo($pathExtract);
            $zip->close();
            return true;
        }

        return false;
    }

    private function cleanUpTempDirectory(): void
    {
        $directoryContent = scandir($this->tempPath);
        foreach ($directoryContent as $entry) {
            if (!is_dir($entry)) {
                @unlink($this->tempPath . $entry);
            }
        }
    }

    /**
     * Adds an Entry to the internal Log Array
     */
    private function addAlert($title, $content): void
    {
        $this->alerts[] = [
            'titel' => $title,
            'text' => $content
        ];
    }

    protected function inputChecker(array $propertyData): int
    {
        $errorsQty = 0;
        if(!preg_match("/[0-9]{5}$/", $propertyData['objekt_plz'])) {
            $errorsQty++;
        }
        if(empty($propertyData['objekt_strasse'])) {
            $errorsQty++;
        }
        if(empty($propertyData['objekt_hausnummer'])) {
            $errorsQty++;
        }
        if(!empty($propertyData['objektarttyp']) && (int)$propertyData['objektarttyp'] === 131) {
            $errorsQty++;
        }
        return $errorsQty;
    }
}
