<?php
namespace SG\SgEstateImportOpenimmo\Tasks;

use DateTime;
use Exception;
use RuntimeException;
use SimpleXMLElement;
use TYPO3\CMS\Core\Cache\Exception\NoSuchCacheException;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\Exception\InvalidTypeException;
use TYPO3\CMS\Scheduler\Task\AbstractTask;
use TYPO3\CMS\Extbase\Object\ObjectManager;
use SG\SgEstateCore\Util\Services;
use TYPO3\CMS\Typo3DbLegacy\Database\DatabaseConnection;
use TYPO3\CMS\Core\Cache\CacheManager;
use ZipArchive;

class Import extends AbstractTask{

    /**************************************
     * BEGIN
     * Members for Task-Settings
     ***************************************/

    /**
     * An email address to be used during the process
     *
     * @var integer $storagePid
     */
    protected $storagePid;

    /**
     * @var bool
     */
    protected $debugMode = false;

    /**
     * @var string
     */
    protected $reportSenderEmail = '';

    /**
     * @var array
     */
    protected $reportReceiverEmails = array();

    /**
     * @var string
     */
    protected $reportSubject = '';

    /**
     * @var string
     */
    protected $reportTemplatePath = '';

    /**
     * @var int
     */
    protected $daysToKeepArchiveFiles = 7;

    /********************************
     * END
     * Members for Task-Settings
     ********************************/

    /**
     * @var array
     */
    private $reportDaten = array();

    /**
     * @var array
     */
    private $fehlerBeiAusfuehrung = array();

    /**
     * Extension Configuration
     *
     * @var array
     */
    private $extConf = array();

    /**
     * Array zur Speicherung von Warnungen während der Ausführung um diese nachher gesammelt zu Versenden
     *
     * Aufbau
     * [] = array(
     *      'titel' => '',
     *      'text'  => '',
     * )
     *
     * @var array
     */
    private $warnungen = array();

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var Services
     */
    private $sgEstateCoreServices;

    /**
     * @var string $uploadPfad
     */
    protected $uploadPfad = '';

    /**
     * @var string
     */
    protected $tempPfad = '';

    /**
     * @var string
     */
    protected $filenameDatafile = '';

    /**
     * @var string
     */
    protected $archivPfad = '';

    /**
     * @var string
     */
    protected $mediaPfad = '';

    /**
     * @var string
     */
    protected $workPath = '';

    /**
     * @var DatabaseConnection
     */
    protected $database;

    /**
     * @var bool
     */
    protected $importFilePresent = false;

    /**
     * Main Task Function started by the scheduler
     *
     * @return bool
     * @throws InvalidTypeException
     * @throws NoSuchCacheException
     */
    public function execute()
    {
        $this->reportDaten['importStart'] = date('d.m.Y H:i:s');
        $this->initialize();

        if ($this->preprocess()){
            $files = scandir($this->workPath);
            sort($files);
            foreach( $files as $file ) {
                $tempData = pathinfo($this->workPath.$file);
                if (strtolower($tempData['extension']) == 'zip') {
                    if ($this->unzip($this->workPath . $file, $this->tempPfad)) {
                        $this->filenameDatafile = null;
                        $tempFiles = scandir($this->tempPfad);
                        foreach ($tempFiles as $dataFile) {
                            $dataFileInfo = pathinfo($this->tempPfad . $dataFile);
                            if (strtolower($dataFileInfo['extension']) == 'xml') {
                                $this->filenameDatafile = $dataFile;
                                $this->importFilePresent = true;
                                $this->import();
                            }
                        }
                        $this->cleanUpTempDirectory();
                    } else {
                        $this->addWarnung('Fehler', 'Datei ' . $file . ' konnte nicht entpackt werden');
                    }
                    rename($this->workPath.$file, $this->archivPfad.$tempData['filename'].'-'.uniqid().'.zip');
                }
            }
        }

        $this->reportDaten['importEnde'] = date('d.m.Y H:i:s');
        $this->postprocess();

        if (count($this->fehlerBeiAusfuehrung) > 0){
            throw new RuntimeException(implode(', ', $this->fehlerBeiAusfuehrung), 4711);
        }else{
            return true;
        }
    }

    /**
     * Initializes All needed Variables
     */
    private function initialize(){
        if (!$this->systemcheck()){
            array_push($this->fehlerBeiAusfuehrung,'System-Check fehlgeschlagen');
            throw new RuntimeException(implode(', ', $this->fehlerBeiAusfuehrung), 4711);
        }
        $this->database = $GLOBALS['TYPO3_DB'];
        $this->fehlerBeiAusfuehrung = array();
        $this->extConf = unserialize($GLOBALS['TYPO3_CONF_VARS']['EXT']['extConf']['sg_estate_import_csv']);

        $this->objectManager = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Object\ObjectManager');
        $this->sgEstateCoreServices = $this->objectManager->get('SG\SgEstateCore\Util\Services');

        $this->tempPfad = PATH_site . 'uploads/tx_sgestateimportopenimmo/temp/';
        $this->archivPfad = PATH_site . 'uploads/tx_sgestateimportopenimmo/archiv/';
        $this->uploadPfad = PATH_site . 'uploads/tx_sgestateimportopenimmo/upload/';
        $this->workPath = PATH_site . 'uploads/tx_sgestateimportopenimmo/work/';

        $this->mediaPfad = PATH_site . 'uploads/tx_sgestatecore/media/';
    }

    /**
     * Cleans up Archive Files
     */
    private function cleanUpArchiveFiles()
    {
        $handle = opendir($this->archivPfad);
        $now = time();
        $secondsToKeepArchiveFiles = $this->getDaysToKeepArchiveFiles()*24*60*60;
        $borderTimestamp = $now - $secondsToKeepArchiveFiles;

        while ($file = readdir($handle)) {
            $filetime= filemtime($this->archivPfad . $file);
            if ($file != '..' && $file != '.' && $filetime < $borderTimestamp){
                @unlink($this->archivPfad . $file);
            }
        }
        closedir($handle);
    }

    private function preprocess(){
        $files = scandir($this->uploadPfad);
        $atLeastOneFilePresentForImport = false;
        foreach( $files as $file ) {
            $atLeastOneFilePresentForImport = true;
            $tempData = pathinfo($this->uploadPfad.$file);
            if (strtolower($tempData['extension']) == 'zip'){
                if ($this->isDebugMode()){
                    copy($this->uploadPfad.$file, $this->workPath.$file);
                }else{
                    rename($this->uploadPfad.$file, $this->workPath.$file);
                }
            }
        }
        return $atLeastOneFilePresentForImport;
    }

    /**
     * Sends the Import Report
     * Removes Tempfiles
     * @throws NoSuchCacheException
     */
    private function postprocess(){
        if ($this->importFilePresent){
            /**
             * @var $cache FrontendInterface
             */
            // Reset Cache
            $cache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgestate_cache');
            $cache->flush();

            $this->sgEstateCoreServices->sendTemplateEmail(
                $this->getReportReceiverEmails(),
                array(),
                $this->getReportSenderEmail(),
                $this->getReportSubject(),
                $this->getReportTemplatePath() == '' ? 'EXT:sg_estate_import_openimmo/Resources/Private/Templates/Email/' : $this->getReportTemplatePath(),
                'Report',
                array(
                    'reportDaten' => $this->reportDaten
                ),
                array(),
                'html'
            );
        }

        $this->cleanUpTempDirectory();

        $this->cleanUpArchiveFiles();

    }

    private function cleanUpTempDirectory(){
        $directoryContent = scandir($this->tempPfad);
        foreach ($directoryContent as $entry) {
            if (!is_dir($entry)){
                @unlink($this->tempPfad.$entry);
            }
        }
    }

    /**
     * Does the Import
     * @throws InvalidTypeException
     */
    private function import(){
        $dataImport = new SimpleXMLElement(file_get_contents($this->tempPfad.$this->filenameDatafile));
        if ((string)$dataImport->uebertragung[0]->attributes()->umfang == 'VOLL'){
            $this->sgEstateCoreServices->deleteAllUpdatableRealties($this->taskUid);
            $this->reportDaten['importTyp'] = 'Vollabgleich';
        }else{
            $this->reportDaten['importTyp'] = 'Teilabgleich';
        }

        foreach ($dataImport->anbieter[0]->immobilie as $item) {
            /**
             * @var SimpleXMLElement $item
             */
            switch (strval($item->verwaltung_techn->aktion->attributes()->aktionart)){
                case 'DELETE':
                    $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                        'uid,kontaktperson',
                        'tx_sgestatecore_domain_model_immobilie',
                        'objektnr_intern = "'.$item->verwaltung_techn->objektnr_intern.'"'
                    );

                    if (count($result) >= 1){
                        $this->sgEstateCoreServices->deleteRealties($result);
                    }
                    $this->reportDaten['deleted'][] = strval($item->verwaltung_techn->objektnr_intern);
                    break;
                case 'CHANGE':
                default:
                    $uid = $this->addImmobilie($item);
                    $this->addN1Abhaengigkeiten($item, $uid);
                    $this->addMnAbhaengigkeiten($item, $uid);
                    $this->addAnhaenge($uid, $item->anhaenge->children());
                    break;

            }
        }
    }

    /**
     * @param SimpleXMLElement $item
     * @return mixed
     * @throws InvalidTypeException
     */
    private function addImmobilie(SimpleXMLElement $item){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_immobilie',
            'objektnr_intern = "'.$item->verwaltung_techn->objektnr_intern.'" AND import_number='.$this->taskUid
        );

        $immobilienDaten = $this->generateImmobilienDaten($item);

        $GLOBALS['TYPO3_DB']->debugOutput = true;
        if (count($result)>=1){
            // Datensatz vorhanden - aktualisieren
            $uidImmobilie = $result[0]['uid'];
            $GLOBALS['TYPO3_DB']->exec_UPDATEquery(
                'tx_sgestatecore_domain_model_immobilie',
                'uid = '.$uidImmobilie,
                $immobilienDaten
            );
            $this->reportDaten['updated'][] = strval($item->verwaltung_techn->objektnr_intern);
        }else{
            // Neuen Datensatz anlegen
            $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                'tx_sgestatecore_domain_model_immobilie',
                $immobilienDaten
            );
            $uidImmobilie = $GLOBALS['TYPO3_DB']->sql_insert_id();
            $this->reportDaten['inserted'][] = strval($item->verwaltung_techn->objektnr_intern);
        }

        return $uidImmobilie;
    }

    /**
     * @param SimpleXMLElement $item
     * @return array
     * @throws InvalidTypeException
     */
    private function generateImmobilienDaten(SimpleXMLElement $item){
        // @todo: objektart_zusatz
        // @todo: unterkellert
        // @todo: stp_carport varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_duplex varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_freiplatz varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_garage varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_parkhaus varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_tiefgarage varchar(255) DEFAULT '' NOT NULL,
        // @todo: user_defined_simple int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: user_defined_extend int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: $temp['zulieferung'] = $this->getBoolWert($item->infrastruktur->zulieferung);

        $temp = array();
        $temp['pid'] = $this->storagePid;
        $temp['updatable'] = 1;
        $temp['import_number'] = $this->taskUid;
        foreach($item->children() as $child1){
            switch ($child1->getName()){
                case 'geo':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'plz':
                                $temp['objekt_plz'] = strval($child2);
                                break;
                            case 'strasse':
                                $temp['objekt_strasse'] = strval($child2);
                                break;
                            case 'hausnummer':
                                $temp['objekt_hausnummer'] = strval($child2);
                                break;
                            case 'etage':
                                $temp['objekt_etage'] = intval($child2);
                                break;
                            case 'lage_im_bau':
                                $temp['objekt_lage_im_bau_links'] = $this->getBoolWert($child2->attributes()->LINKS);
                                $temp['objekt_lage_im_bau_rechts'] = $this->getBoolWert($child2->attributes()->RECHTS);
                                $temp['objekt_lage_im_bau_vorne'] = $this->getBoolWert($child2->attributes()->VORNE);
                                $temp['objekt_lage_im_bau_hinten'] = $this->getBoolWert($child2->attributes()->HINTEN);
                                break;
                            case 'regionaler_zusatz':
                                $temp['objekt_regionaler_zusatz'] = strval($child2);
                                break;
                            case 'bundesland':
                                $temp['objekt_bundesland'] = strval($child2);
                                break;
                            case 'gemeindecode':
                                $temp['objekt_gemeindecode'] = strval($child2);
                                break;
                            case 'flur':
                                $temp['objekt_flur'] = strval($child2);
                                break;
                            case 'flustueck':
                                $temp['objekt_flurstueck'] = strval($child2);
                                break;
                            case 'gemarkung':
                                $temp['objekt_gemarkung'] = strval($child2);
                                break;
                            case 'anzahl_etagen':
                                $temp['objekt_anzahl_etagen'] = intval($child2);
                                break;
                            case 'wohnungsnr':
                                $temp['objekt_wohnungsnr'] = strval($child2);
                                break;
                            case 'karten_makro':
                                $temp['objekt_karten_makro'] = $this->getBoolWert($child2);
                                break;
                            case 'karten_mikro':
                                $temp['objekt_karten_mikro'] = $this->getBoolWert($child2);
                                break;
                            case 'virtuelletour':
                                $temp['objekt_virtuelletour'] = $this->getBoolWert($child2);
                                break;
                            case 'luftbilder':
                                $temp['objekt_luftbilder'] = $this->getBoolWert($child2);
                                break;
                            case 'geokoordinaten':
                                $temp['objekt_breitengrad'] = str_replace(',','.',strval($child2->attributes()->breitengrad));
                                $temp['objekt_laengengrad'] = str_replace(',','.',strval($child2->attributes()->laengengrad));
                                break;
                            case 'user_defined_simplefield':
                                foreach ($child2->attributes()->feldname as $attribute){
                                    switch ($attribute){
                                        case 'SearchRegion':
                                            //$temp['search_region'] = strval($child2);
                                            break;
                                        default:
                                            //echo var_dump($child2);die();
                                            break;
                                    }
                                }
                                break;
                            case 'ort':
                            case 'land':
                                break;
                        }
                    }
                    break;
                case 'preise':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'kaufpreis':
                                $temp['kaufpreis'] = floatval($child2);
                                break;
                            case 'kaufpreisnetto':
                                $temp['kaufpreis_netto'] = floatval($child2);
                                $temp['kaufpreis_netto_ust'] = floatval($child2->attributes()->kaufpreisust);
                                break;
                            case 'kaufpreisbrutto':
                                $temp['kaufpreis_brutto'] = floatval($child2);
                                break;
                            case 'nettokaltmiete':
                                $temp['nettokaltmiete'] = floatval($child2);
                                break;
                            case 'kaltmiete':
                                $temp['kaltmiete'] = floatval($child2);
                                break;
                            case 'warmmiete':
                                $temp['warmmiete'] = floatval($child2);
                                break;
                            case 'nebenkosten':
                                $temp['nebenkosten'] = floatval($child2);
                                break;
                            case 'heizkosten_enthalten':
                                $temp['heizkosten_enthalten'] = $this->getBoolWert($child2);
                                break;
                            case 'heizkosten':
                                $temp['heizkosten'] = floatval($child2);
                                break;
                            case 'zzg_mehrwertsteuer':
                                $temp['zzgl_mehrwertsteuer'] = $this->getBoolWert($child2);
                                break;
                            case 'mietzuschlaege':
                                $temp['mietzuschlaege'] = floatval($child2);
                                break;
                            case 'hauptmietzinsnetto':
                                $temp['hauptmietzins_netto'] = floatval($child2);
                                $temp['hauptmietzins_netto_ust'] = floatval($child2->attributes()->hauptmietzinsust);
                                break;
                            case 'pauschalmiete':
                                $temp['pauschalmiete'] = floatval($child2);
                                break;
                            case 'betriebskostennetto':
                                $temp['betriebskosten_netto'] = floatval($child2);
                                $temp['betriebskosten_netto_ust'] = floatval($child2->attributes()->betriebskostenust);
                                break;
                            case 'evbnetto':
                                $temp['evb_netto'] = floatval($child2);
                                $temp['evb_netto_ust'] = floatval($child2);
                                break;
                            case 'gesamtmietenetto':
                                $temp['gesamtmiete_netto'] = floatval($child2);
                                $temp['gesamtmiete_netto_ust'] = floatval($child2->attributes()->gesamtmieteust);
                                break;
                            case 'gesamtmietebrutto':
                                $temp['gesamtmiete_brutto'] = floatval($child2);
                                break;
                            case 'gesamtbelastungnetto':
                                $temp['gesamtbelastung_netto'] = floatval($child2);
                                $temp['gesamtbelastung_netto_ust'] = floatval($child2->attributes()->gesamtbelastungust);
                                break;
                            case 'gesamtbelastungbrutto':
                                $temp['gesamtbelastung_brutto'] = floatval($child2);
                                break;
                            case 'gesamtkostenprom2von':
                                $temp['gesamtkosten_pro_m2_von'] = floatval($child2);
                                $temp['gesamtkosten_pro_m2_bis'] = floatval($child2->attributes()->gesamtkostenprom2bis);
                                break;
                            case 'heizkostennetto':
                                $temp['heizkosten_netto'] = floatval($child2);
                                $temp['heizkosten_netto_ust'] = floatval($child2->attributes()->heizkostenust);
                                break;
                            case 'monatlichekostennetto':
                                $temp['monatlichekosten_netto'] = floatval($child2);
                                $temp['monatlichekosten_netto_ust'] = floatval($child2->attributes()->monatlichekostenust);
                                break;
                            case 'monatlichekostenbrutto':
                                $temp['monatlichekostenbrutto'] = floatval($child2);
                                break;
                            case 'nebenkostenprom2von':
                                $temp['nebenkosten_pro_m2_von'] = floatval($child2);
                                $temp['nebenkosten_pro_m2_bis'] = floatval($child2->attributes()->nebenkostenprom2bis);
                                break;
                            case 'ruecklagenetto':
                                $temp['ruecklage_netto'] = floatval($child2);
                                $temp['ruecklage_netto_ust'] = floatval($child2->attributes()->ruecklageust);
                                break;
                            case 'sonstigekostennetto':
                                $temp['sonstigekosten_netto'] = floatval($child2);
                                $temp['sonstigekosten_netto_ust'] = floatval($child2->attributes()->sonstigekostenust);
                                break;
                            case 'sonstigemietenetto':
                                $temp['sonstigemiete_netto'] = floatval($child2);
                                $temp['sonstigemiete_netto_ust'] = floatval($child2->attributes()->sonstigemieteust);
                                break;
                            case 'summemietenetto':
                                $temp['summemiete_netto'] = floatval($child2);
                                $temp['summemiete_netto_ust'] = floatval($child2->attributes()->sonstigemieteust);
                                break;
                            case 'nettomieteprom2von':
                                $temp['nettomiete_pro_m2_von'] = floatval($child2);
                                $temp['nettomiete_pro_m2_bis'] = floatval($child2->attributes()->nettomieteprom2bis);
                                break;
                            case 'pacht':
                                $temp['pacht'] = floatval($child2);
                                break;
                            case 'erbpacht':
                                $temp['erbpacht'] = floatval($child2);
                                break;
                            case 'hausgeld':
                                $temp['hausgeld'] = floatval($child2);
                                break;
                            case 'abstand':
                                $temp['abstand'] = floatval($child2);
                                break;
                            case 'preis_zeitraum_von':
                                $temp['preis_zeitraum_von'] = $this->getDateWert($child2);
                                break;
                            case 'preis_zeitraum_bis':
                                $temp['preis_zeitraum_bis'] = $this->getDateWert($child2);
                                break;
                            case 'mietpreis_pro_qm':
                                $temp['mietpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'kaufpreis_pro_qm':
                                $temp['kaufpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'provisionspflichtig':
                                $temp['provisionspflichtig'] = $this->getBoolWert($child2);
                                break;
                            case 'innen_courtage':
                                $temp['innencourtage'] = strval($child2);
                                $temp['innencourtage_mit_mwst'] = $this->getBoolWert($child2->attributes()->mit_mwst);
                                break;
                            case 'aussen_courtage':
                                $temp['aussencourtage'] = strval($child2);
                                $temp['aussencourtage_mit_mwst'] = $this->getBoolWert($child2->attributes()->mit_mwst);
                                break;
                            case 'courtage_hinweis':
                                $temp['courtage_hinweis'] = strval($child2);
                                break;
                            case 'provisionnetto':
                                $temp['provision_netto'] = floatval($child2);
                                $temp['provision_netto_ust'] = floatval($child2->attributes()->provisionust);
                                break;
                            case 'provisionbrutto':
                                $temp['provision_brutto'] = floatval($child2);
                                break;
                            case 'mwst_satz':
                                $temp['mwst_satz'] = floatval($child2);
                                break;
                            case 'mwst_gesamt':
                                $temp['mwst_gesamt'] = floatval($child2);
                                break;
                            case 'freitext_preis':
                                $temp['freitext_preis'] = nl2br($child2);
                                break;
                            case 'x_fache':
                                $temp['xfache'] = strval($child2);
                                break;
                            case 'nettorendite':
                                $temp['nettorendite'] = floatval($child2);
                                break;
                            case 'nettorendite_soll':
                                $temp['nettorendite_soll'] = floatval($child2);
                                break;
                            case 'nettorendite_ist':
                                $temp['nettorendite_ist'] = floatval($child2);
                                break;
                            case 'mieteinnahmen_ist':
                                $temp['mieteinnahmen_ist'] = floatval($child2);
                                break;
                            case 'mieteinnahmen_soll':
                                $temp['mieteinnahmen_soll'] = floatval($child2);
                                break;
                            case 'erschliessungskosten':
                                $temp['erschliessungskosten'] = floatval($child2);
                                break;
                            case 'kaution':
                                $temp['kaution'] = floatval($child2);
                                break;
                            case 'kaution_text':
                                $temp['kaution_text'] = strval($child2);
                                break;
                            case 'geschaeftsguthaben':
                                $temp['geschaeftsguthaben'] = floatval($child2);
                                break;
                            case 'richtpreis':
                                $temp['richtpreis'] = floatval($child2);
                                break;
                            case 'richtpreisprom2':
                                $temp['richtpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'user_defined_simplefield':
                                $attribute = strval($child2->attributes()->feldname);
                                switch ($attribute){
                                    case 'anteile':
                                        $temp['genossenschaftsanteile'] = floatval($child2);
                                        break;
                                    case 'vertragsabschlussgebuehr':
                                        $temp['vertragsabschlussgebuehr'] = floatval($child2);
                                        break;
                                    default:
                                        //echo var_dump($child2);die();
                                        break;
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'flaechen':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'wohnflaeche':
                                $temp['wohnflaeche'] = floatval($item->flaechen->wohnflaeche);
                                break;
                            case 'nutzflaeche':
                                $temp['nutzflaeche'] = floatval($item->flaechen->nutzflaeche);
                                break;
                            case 'gesamtflaeche':
                                $temp['gesamtflaeche'] = floatval($item->flaechen->gesamtflaeche);
                                break;
                            case 'ladenflaeche':
                                $temp['ladenflaeche'] = floatval($item->flaechen->ladenflaeche);
                                break;
                            case 'lagerflaeche':
                                $temp['lagerflaeche'] = floatval($item->flaechen->lagerflaeche);
                                break;
                            case 'verkaufsflaeche':
                                $temp['verkaufsflaeche'] = floatval($item->flaechen->verkaufsflaeche);
                                break;
                            case 'freiflaeche':
                                $temp['freiflaeche'] = floatval($item->flaechen->freiflaeche);
                                break;
                            case 'bueroflaeche':
                                $temp['bueroflaeche'] = floatval($item->flaechen->bueroflaeche);
                                break;
                            case 'bueroteilflaeche':
                                $temp['bueroteilflaeche'] = floatval($item->flaechen->bueroteilflaeche);
                                break;
                            case 'fensterfront':
                                $temp['fensterfront'] = floatval($item->flaechen->fensterfront);
                                break;
                            case 'verwaltungsflaeche':
                                $temp['verwaltungsflaeche'] = floatval($item->flaechen->verwaltungsflaeche);
                                break;
                            case 'gastroflaeche':
                                $temp['gastroflaeche'] = floatval($item->flaechen->gastroflaeche);
                                break;
                            case 'grz':
                                $temp['grundflaechenzahl'] = strval($item->flaechen->grz);
                                break;
                            case 'gfz':
                                $temp['geschossflaechenzahl'] = strval($item->flaechen->gfz);
                                break;
                            case 'bmz':
                                $temp['baumassenzahl'] = strval($item->flaechen->bmz);
                                break;
                            case 'bgf':
                                $temp['bruttogeschossflaechenzahl'] = floatval($item->flaechen->bgf);
                                break;
                            case 'grundstuecksflaeche':
                                $temp['grundstuecksflaeche'] = floatval($item->flaechen->grundstuecksflaeche);
                                break;
                            case 'sonstflaeche':
                                $temp['sonstflaeche'] = floatval($item->flaechen->sonstflaeche);
                                break;
                            case 'anzahl_zimmer':
                                $temp['anzahl_zimmer'] = floatval($item->flaechen->anzahl_zimmer);
                                break;
                            case 'anzahl_schlafzimmer':
                                $temp['anzahl_schlafzimmer'] = floatval($item->flaechen->anzahl_schlafzimmer);
                                break;
                            case 'anzahl_badezimmer':
                                $temp['anzahl_badezimmer'] = floatval($item->flaechen->anzahl_badezimmer);
                                break;
                            case 'anzahl_sep_wc':
                                $temp['anzahl_separate_wc'] = floatval($item->flaechen->anzahl_sep_wc);
                                break;
                            case 'balkon_terrasse_flaeche':
                                $temp['balkon_terasse_flaeche'] = floatval($item->flaechen->balkon_terrasse_flaeche);
                                break;
                            case 'anzahl_wohn_schlafzimmer':
                                $temp['anzahl_wohn_schlafzimmer'] = floatval($item->flaechen->anzahl_wohn_schlafzimmer);
                                break;
                            case 'gartenflaeche':
                                $temp['gartenflaeche'] = floatval($item->flaechen->gartenflaeche);
                                break;
                            case 'anzahl_balkone':
                                $temp['anzahl_balkone'] = floatval($item->flaechen->anzahl_balkone);
                                break;
                            case 'anzahl_terrassen':
                                $temp['anzahl_terassen'] = floatval($item->flaechen->anzahl_terrassen);
                                break;
                            case 'anzahl_logia':
                                $temp['anzahl_logia'] = floatval($item->flaechen->anzahl_logia);
                                break;
                            case 'fensterfront_qm':
                                $temp['fensterfront_m2'] = floatval($item->flaechen->fensterfront_qm);
                                break;
                            case 'grundstuecksfront':
                                $temp['grundstuecksfront'] = floatval($item->flaechen->grundstuecksfront);
                                break;
                            case 'dachbodenflaeche':
                                $temp['dachbodenflaeche'] = floatval($item->flaechen->dachbodenflaeche);
                                break;
                            case 'teilbar_ab':
                                $temp['teilbar_ab'] = floatval($item->flaechen->teilbar_ab);
                                break;
                            case 'kellerflaeche':
                                $temp['kellerflaeche'] = floatval($item->flaechen->kellerflaeche);
                                break;
                            case 'beheizbare_flaeche':
                                $temp['beheizbare_flaeche'] = floatval($item->flaechen->beheizbare_flaeche);
                                break;
                            case 'anzahl_stellplaetze':
                                $temp['anzahl_stellplaetze'] = floatval($item->flaechen->anzahl_stellplaetze);
                                break;
                            case 'plaetze_gastraum':
                                $temp['plaetze_gastraum'] = floatval($item->flaechen->plaetze_gastraum);
                                break;
                            case 'anzahl_betten':
                                $temp['anzahl_betten'] = floatval($item->flaechen->anzahl_betten);
                                break;
                            case 'anzahl_tagungsraeume':
                                $temp['anzahl_tagungsraeume'] = floatval($item->flaechen->anzahl_tagungsraeume);
                                break;
                            case 'vermietbare_flaeche':
                                $temp['vermietbare_flaeche'] = floatval($item->flaechen->vermietbare_flaeche);
                                break;
                            case 'anzahl_wohneinheiten':
                                $temp['anzahl_wohneinheiten'] = floatval($item->flaechen->anzahl_wohneinheiten);
                                break;
                            case 'anzahl_gewerbeeinheiten':
                                $temp['anzahl_gewerbeeinheiten'] = floatval($item->flaechen->anzahl_gewerbeeinheiten);
                                break;
                            case 'einliegerwohnung':
                                $temp['einliegerwohnung'] = $this->getBoolWert($item->flaechen->einliegerwohnung);
                                break;
                            case 'kubatur':
                                $temp['kubatur'] = floatval($item->flaechen->kubatur);
                                break;
                            case 'ausnuetzungsziffer':
                                $temp['ausnuetzungsziffer'] = floatval($item->flaechen->ausnuetzungsziffer);
                                break;
                            case 'flaechevon':
                                $temp['flaechevon'] = floatval($item->flaechen->flaechevon);
                                break;
                            case 'flaechebis':
                                $temp['flaechebis'] = floatval($item->flaechen->flaechebis);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'ausstattung':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'wg_geeignet':
                                $temp['wg_geeignet'] = $this->getBoolWert($item->ausstattung->wg_geeignet);
                                break;
                            case 'raeume_veraenderbar':
                                $temp['raeume_veraenderbar'] = $this->getBoolWert($item->ausstattung->raeume_veraenderbar);
                                break;
                            case 'kamin':
                                $temp['kamin'] = $this->getBoolWert($item->ausstattung->kamin);
                                break;
                            case 'klimatisiert':
                                $temp['klimatisiert'] = $this->getBoolWert($item->ausstattung->klimatisiert);
                                break;
                            case 'fahrstuhl':
                                $temp['aufzug_personen'] = $this->getBoolWert($item->ausstattung->fahrstuhl->attributes()->PERSONEN);
                                //$temp['aufzug_personen'] = $this->getBoolWert($item->ausstattung->fahrstuhl->attributes()->LASTEN);
                                break;
                            case 'gartennutzung':
                                $temp['gartennutzung'] = $this->getBoolWert($item->ausstattung->gartennutzung);
                                break;
                            case 'moebliert':
                                $temp['moebliert'] = strval($item->ausstattung->moebliert->attributes()->moeb);
                                break;
                            case 'rollstuhlgerecht':
                                $temp['rollstuhlgerecht'] = $this->getBoolWert($item->ausstattung->rollstuhlgerecht);
                                break;
                            case 'kabel_sat_tv':
                                $temp['kabel_sat_tv'] = $this->getBoolWert($item->ausstattung->kabel_sat_tv);
                                $temp['kabelfernsehen'] = $this->getBoolWert($item->ausstattung->kabel_sat_tv);
                                break;
                            case 'dvbt':
                                $temp['dvbt'] = $this->getBoolWert($item->ausstattung->dvbt);
                                break;
                            case 'barrierefrei':
                                $temp['barrierefrei'] = $this->getBoolWert($item->ausstattung->barrierefrei);
                                break;
                            case 'sauna':
                                $temp['sauna'] = $this->getBoolWert($item->ausstattung->sauna);
                                break;
                            case 'swimmingpool':
                                $temp['swimmingpool'] = $this->getBoolWert($item->ausstattung->swimmingpool);
                                break;
                            case 'wasch_trockenraum':
                                $temp['wasch_trockenraum'] = $this->getBoolWert($item->ausstattung->wasch_trockenraum);
                                break;
                            case 'wintergarten':
                                $temp['wintergarten'] = $this->getBoolWert($item->ausstattung->wintergarten);
                                break;
                            case 'dv_verkabelung':
                                $temp['dv_verkabelung'] = $this->getBoolWert($item->ausstattung->dv_verkabelung);
                                break;
                            case 'rampe':
                                $temp['rampe'] = $this->getBoolWert($item->ausstattung->rampe);
                                break;
                            case 'hebebuehne':
                                $temp['hebebuehne'] = $this->getBoolWert($item->ausstattung->hebebuehne);
                                break;
                            case 'kran':
                                $temp['kran'] = $this->getBoolWert($item->ausstattung->kran);
                                break;
                            case 'gastterrasse':
                                $temp['gastterrasse'] = $this->getBoolWert($item->ausstattung->gastterrasse);
                                break;
                            case 'stromanschlusswert':
                                $temp['stromanschlusswert'] = floatval($item->ausstattung->stromanschlusswert);
                                break;
                            case 'kantine_cafeteria':
                                $temp['katine_cafeteria'] = $this->getBoolWert($item->ausstattung->kantine_cafeteria);
                                break;
                            case 'teekueche':
                                $temp['teekueche'] = $this->getBoolWert($item->ausstattung->teekueche);
                                break;
                            case 'hallenhoehe':
                                $temp['hallenhoehe'] = floatval($item->ausstattung->hallenhoehe);
                                break;
                            case 'angeschl_gastronomie':
                                $temp['hotelrestaurant_angeschlossen'] = $this->getBoolWert($item->ausstattung->angeschl_gastronomie->attributes()->HOTELRESTAURANT);
                                $temp['bar_angeschlossen'] = $this->getBoolWert($item->ausstattung->angeschl_gastronomie->attributes()->BAR);
                                break;
                            case 'brauereibindung':
                                $temp['brauereibindung'] = $this->getBoolWert($item->ausstattung->brauereibindung);
                                break;
                            case 'sporteinrichtungen':
                                $temp['sporteinrichtungen'] = $this->getBoolWert($item->ausstattung->sporteinrichtungen);
                                break;
                            case 'wellnessbereich':
                                $temp['wellnessbereich'] = $this->getBoolWert($item->ausstattung->wellnessbereich);
                                break;
                            case 'serviceleistungen':
                                $temp['service_betreutes_wohnen'] = $this->getBoolWert($item->ausstattung->serviceleistungen->attributes()->BETREUTES_WOHNEN);
                                $temp['service_catering'] = $this->getBoolWert($item->ausstattung->serviceleistungen->attributes()->CATERING);
                                $temp['service_reinigung'] = $this->getBoolWert($item->ausstattung->serviceleistungen->attributes()->REINIGUNG);
                                $temp['service_einkauf'] = $this->getBoolWert($item->ausstattung->serviceleistungen->attributes()->EINKAUF);
                                $temp['service_wachdienst'] = $this->getBoolWert($item->ausstattung->serviceleistungen->attributes()->WACHDIENST);
                                break;
                            case 'telefon_ferienimmobilie':
                                $temp['telefon_ferienimmobilie'] = $this->getBoolWert($item->ausstattung->telefon_ferienimmobilie);
                                break;
                            case 'breitband_zugang':
                                $temp['breitband_zugang_art'] = strval($item->ausstattung->breitband_zugang->attributes()->art);
                                $temp['breitband_zugang_speed'] = floatval($item->ausstattung->breitband_zugang->attributes()->speed);
                                break;
                            case 'umts_empfang':
                                $temp['umts_empfang'] = $this->getBoolWert($item->ausstattung->umts_empfang);
                                break;
                            case 'sicherheitstechnik':
                                $temp['sicherheit_alarmanlage'] = $this->getBoolWert($item->ausstattung->sicherheitstechnik->attributes()->ALARMANLAGE);
                                $temp['sicherheit_kamera'] = $this->getBoolWert($item->ausstattung->sicherheitstechnik->attributes()->KAMERA);
                                $temp['sicherheit_polizeiruf'] = $this->getBoolWert($item->ausstattung->sicherheitstechnik->attributes()->POLIZEIRUF);
                                break;
                            case 'abstellraum':
                                $temp['abstellraum'] = $this->getBoolWert($item->ausstattung->abstellraum);
                                break;
                            case 'fahrradraum':
                                $temp['fahrradraum'] = $this->getBoolWert($item->ausstattung->fahrradraum);
                                break;
                            case 'rolladen':
                                $temp['rolladen'] = $this->getBoolWert($item->ausstattung->rolladen);
                                break;
                            case 'bibliothek':
                                $temp['bibliothek'] = $this->getBoolWert($item->ausstattung->bibliothek);
                                break;
                            case 'dachboden':
                                $temp['dachboden'] = $this->getBoolWert($item->ausstattung->dachboden);
                                break;
                            case 'gaestewc':
                                $temp['gaestewc'] = $this->getBoolWert($item->ausstattung->gaestewc);
                                break;
                            case 'kabelkanaele':
                                $temp['kabelkanaele'] = $this->getBoolWert($item->ausstattung->kabelkanaele);
                                break;
                            case 'seniorengerecht':
                                $temp['seniorengerecht'] = $this->getBoolWert($item->ausstattung->seniorengerecht);
                                break;
                            case 'unterkellert':
                                $temp['unterkellert'] = $this->getBoolWert($item->ausstattung->unterkellert->attributes()->keller);
                                break;
                            case 'user_defined_simplefield':
                                foreach ($child2->attributes() as $attribute){
                                    switch (strval($attribute)){
                                        case 'gegensprechanlage':
                                            $temp['gegensprechanlage'] = $this->getBoolWert($child2);
                                            break;
                                        case 'leicht_erreichbar':
                                            $temp['leicht_erreichbar'] = $this->getBoolWert($child2);
                                            break;
                                        case 'bad_mit_fenster':
                                            if ($child1->bad){
                                                if ($child2 == 'true'){$child1->bad->addAttribute('FENSTER','true');}
                                            }else{
                                                $child1->addChild('bad');
                                                if ($child2 == 'true'){$child1->bad->addAttribute('FENSTER','true');}
                                            }
                                            break;
                                        case 'wunschwohnung':
                                            if (!($child1->kampagne)){
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true'){
                                                $child1->kampagne->addAttribute('WUNSCHWOHNUNG','true');
                                            }
                                            break;
                                        case 'junges_wohnen':
                                            if (!($child1->kampagne)){
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true'){
                                                $child1->kampagne->addAttribute('JUNGESWOHNEN','true');
                                            }
                                            break;
                                        case 'bad_modernisiert':
                                            $temp['bad_modernisiert'] = $this->getBoolWert($child2);
                                            break;
                                        case 'hd_satellitenfernsehen':
                                            $temp['satellitenfernsehen'] = $this->getBoolWert($child2);
                                            break;
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'zustand_angaben':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'baujahr':
                                $temp['baujahr'] = strval($item->zustand_angaben->baujahr);
                                break;
                            case 'letztemodernisierung':
                                $temp['letztemodernisierung'] = strval($item->zustand_angaben->letztemodernisierung);
                                break;
                            case 'bauzone':
                                $temp['bauzone'] = strval($item->zustand_angaben->bauzone);
                                break;
                            case 'altlasten':
                                $temp['altlasten'] = strval($item->zustand_angaben->altlasten);
                                break;
                            case 'energiepass':
                                $temp['energiepass_energieverbrauchkennwert'] = strval($item->zustand_angaben->energiepass->energieverbrauchkennwert);
                                $temp['energiepass_mitwarmwasser'] = $this->getBoolWert($item->zustand_angaben->energiepass->mitwarmwasser);
                                $temp['energiepass_endenergiebedarf'] = strval($item->zustand_angaben->energiepass->endenergiebedarf);
                                $temp['energiepass_baujahr'] = strval($item->zustand_angaben->energiepass->baujahr);
                                $temp['energiepass_wertklasse'] = strval($item->zustand_angaben->energiepass->wertklasse);
                                $temp['energiepass_ausstelldatum'] = $this->getDateWert($item->zustand_angaben->energiepass->ausstelldatum);

                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_energiepassart',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->epart).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['energiepass_art'] = $result[0]['uid'];
                                }
                                $result = null;

                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_befeuerungsart',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->primaerenergietraeger).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['energiepass_primaerenergietraeger'] = $result[0]['uid'];
                                }
                                $result = null;

                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_energiepassjahrgang',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->jahrgang).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['energiepass_jahrgang'] = $result[0]['uid'];
                                }
                                $result = null;


                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_energiepassgebaeudeart',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->gebaeudeart).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['energiepass_gebaeudeart'] = $result[0]['uid'];
                                }
                                $result = null;



                                break;
                            case 'alter':
                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_altertyp',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->alter->attributes()->alter_attr).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['altertyp'] = $result[0]['uid'];
                                }
                                $result = null;
                                break;
                            case 'verkaufstatus':
                                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                    'uid',
                                    'tx_sgestatecore_domain_model_verkaufstatus',
                                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->verkaufstatus).'"'
                                );
                                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                                    $temp['verkaufstatus'] = $result[0]['uid'];
                                }
                                $result = null;
                                break;
                            case 'user_defined_simplefield':
                            default:
                                break;
                        }
                    }
                    break;
                case 'freitexte':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'objekttitel':
                                $temp['objekttitel'] = nl2br($item->freitexte->objekttitel);
                                break;
                            case 'dreizeiler':
                                $temp['kurzbeschreibung'] = nl2br($item->freitexte->dreizeiler);
                                break;
                            case 'lage':
                                $temp['lage'] = nl2br($item->freitexte->lage);
                                break;
                            case 'ausstatt_beschr':
                                $temp['austattungsbeschreibung'] = nl2br($item->freitexte->ausstatt_beschr);
                                break;
                            case 'objektbeschreibung':
                                $temp['objektbeschreibung'] = nl2br($item->freitexte->objektbeschreibung);
                                break;
                            case 'sonstige_angaben':
                                $temp['sonstige_angaben'] = nl2br($item->freitexte->sonstige_angaben);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_objekt':
                    foreach ($child1->children() as $child2){
                        switch($child2->getName()){
                            case 'objektadresse_freigeben':
                                $temp['objektadresse_freigeben'] = $this->getBoolWert($item->verwaltung_objekt->objektadresse_freigeben);
                                break;
                            case 'verfuegbar_ab':
                                $temp['verfuegbar_ab'] = strval($item->verwaltung_objekt->verfuegbar_ab);
                                break;
                            case 'abdatum':
                                $temp['verfuegbar_ab_datum'] = $this->getDateWert($item->verwaltung_objekt->abdatum);
                                break;
                            case 'bisdatum':
                                $temp['verfuegbar_bis_datum'] = $this->getDateWert($item->verwaltung_objekt->bisdatum);
                                break;
                            case 'versteigerungstermin':
                                $temp['versteigerungstermin'] = $this->getDateWert($item->verwaltung_objekt->versteigerungstermin);
                                break;
                            case 'wbs_sozialwohnung':
                                $temp['wbs_erforderlich'] = $this->getBoolWert($item->verwaltung_objekt->wbs_sozialwohnung);
                                break;
                            case 'vermietet':
                                $temp['vermietet'] = $this->getBoolWert($item->verwaltung_objekt->vermietet);
                                break;
                            case 'gruppennummer':
                                $temp['gruppennummer'] = strval($item->verwaltung_objekt->gruppennummer);
                                break;
                            case 'zugang':
                                $temp['zugang'] = strval($item->verwaltung_objekt->zugang);
                                break;
                            case 'laufzeit':
                                $temp['laufzeit'] = floatval($item->verwaltung_objekt->laufzeit);
                                break;
                            case 'max_personen':
                                $temp['maximale_personenanzahl'] = intval($item->verwaltung_objekt->max_personen);
                                break;
                            case 'nichtraucher':
                                $temp['nichtraucher'] = $this->getBoolWert($item->verwaltung_objekt->nichtraucher);
                                break;
                            case 'haustiere':
                                $temp['haustiere'] = $this->getBoolWert($item->verwaltung_objekt->haustiere);
                                break;
                            case 'denkmalgeschuetzt':
                                $temp['denkmalgeschuetzt'] = $this->getBoolWert($item->verwaltung_objekt->denkmalgeschuetzt);
                                break;
                            case 'als_ferien':
                                $temp['als_ferien'] = $this->getBoolWert($item->verwaltung_objekt->als_ferien);
                                break;
                            case 'gewerbliche_nutzung':
                                $temp['gewerbliche_nutzung'] = $this->getBoolWert($item->verwaltung_objekt->gewerbliche_nutzung);
                                break;
                            case 'branchen':
                                $temp['branchen'] = strval($item->verwaltung_objekt->branchen);
                                break;
                            case 'hochhaus':
                                $temp['hochhaus'] = $this->getBoolWert($item->verwaltung_objekt->hochhaus);
                                break;
                            case 'aktiv_von':
                                $temp['anzeige_ab'] = $this->getDateWert($item->verwaltung_objekt->aktiv_von);
                                break;
                            case 'aktiv_bis':
                                $temp['anzeige_bis'] = $this->getDateWert($item->verwaltung_objekt->aktiv_bis);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_techn':
                    foreach ($child1->children() as $child2){
                        switch ($child2->getName()){
                            case 'openimmo_obid':
                                $temp['openimmo_objid'] = strval($item->verwaltung_techn->openimmo_obid);
                                break;
                            case 'objektnr_intern':
                                $temp['objektnr_intern'] = strval($item->verwaltung_techn->objektnr_intern);
                                break;
                            case 'objektnr_extern':
                                $temp['objektnr_extern'] = strval($item->verwaltung_techn->objektnr_extern);
                                break;
                            case 'kennung_ursprung':
                                $temp['kennung_ursprung'] = strval($item->verwaltung_techn->kennung_ursprung);
                                break;
                            case 'stand_vom':
                                $temp['stand_vom'] = $this->getDateWert($item->verwaltung_techn->stand_vom);
                                break;
                            case 'weitergabe_generell':
                                $temp['weitergabe_generell'] = $this->getBoolWert($item->verwaltung_techn->weitergabe_generell);
                                break;
                            case 'weitergabe_positivliste':
                                $temp['weitergabe_positivliste'] = strval($item->verwaltung_techn->weitergabe_positivliste);
                                break;
                            case 'weitergabe_negativliste':
                                $temp['weitergabe_negativliste'] = strval($item->verwaltung_techn->weitergabe_negativliste);
                                break;
                            case 'gruppen_kennung':
                                $temp['gruppen_kennung'] = strval($item->verwaltung_techn->gruppen_kennung);
                                break;
                            case 'master':
                                $temp['master_objekt'] = strval($item->verwaltung_techn->master);
                                $temp['master_visible'] = $this->getBoolWert($item->verwaltung_techn->master->attributes()->visible);
                                break;
                            case 'sprache':
                                $temp['sprache'] = strval($item->verwaltung_techn->sprache);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'kontaktperson':
                case 'anhaenge':
                case 'objektkategorie':
                    break;
            }
        }
        return $temp;
    }

    private function addN1Abhaengigkeiten($item, $immobilie){
        $updateArray = array();


        if (($item->infrastruktur->ausblick)&&(strval($item->infrastruktur->ausblick->attributes()->blick) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_ausblick',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->infrastruktur->ausblick->attributes()->blick).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['ausblick'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->geschlecht)&&(strval($item->verwaltung_objekt->geschlecht->attributes()->geschl_attr) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_geschlecht',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->verwaltung_objekt->geschlecht->attributes()->geschl_attr).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['geschlecht'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (strval($item->zustand_angaben->energiepass->epart) != ''){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_energiepassart',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->epart).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['energiepass_art'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (strval($item->zustand_angaben->energiepass->primaerenergietraeger) != ''){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_befeuerungsart',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->energiepass->primaerenergietraeger).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['energiepass_primaerenergietraeger'] = $result[0]['uid'];
            }
            $result = null;
        }
        if ($item->objektkategorie->objektart){
            foreach ($item->objektkategorie->objektart->children() as $child) {
                /**
                 * @var $child SimpleXMLElement
                 */
                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                    'uid',
                    'tx_sgestatecore_domain_model_objektart',
                    'sys_language_uid = 0 AND kuerzel="'.strval($child->getName()).'"'
                );
                if (count($result) > 0){
                    $updateArray['objektart'] = $result[0]['uid'];
                }

                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                    'uid',
                    'tx_sgestatecore_domain_model_objektarttyp',
                    'sys_language_uid = 0 AND kuerzel = "'.strval($child[$child->getName().'typ']).'"'
                );
                if (count($result) > 0){
                    $updateArray['objektarttyp'] = $result[0]['uid'];
                }else{
                    $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                        'uid',
                        'tx_sgestatecore_domain_model_objektarttyp',
                        'sys_language_uid = 0 AND kuerzel = "'.strval($child[$child->getName().'_typ']).'"'
                    );
                    if (count($result) > 0){
                        $updateArray['objektarttyp'] = $result[0]['uid'];
                    }
                }
            }
        }

        $updateArray['objekt_ort'] = $this->addOrt(strval($item->geo->ort), true);
        if (strval($item->geo->regionaler_zusatz) == ''){
            foreach ($item->geo->children() as $subnode){
                /**
                 * @var $subnode SimpleXMLElement
                 */
                switch ($subnode->getName()){
                    case 'user_defined_simplefield':
                        foreach ($subnode->attributes() as $attribute) {
                            switch (strval($attribute)) {
                                case 'SearchRegion': // Immosolve District
                                case 'ZONE': //Immoblue District
                                    if (strval($subnode) != ''){
                                        $updateArray['objekt_stadtteil'] = $this->addStadtteil(strval($subnode), $updateArray['objekt_ort'], true);
                                    }
                                    break;
                            }
                        }
                        break;
                }
            }
        }else{
            $updateArray['objekt_stadtteil'] = $this->addStadtteil($item->geo->regionaler_zusatz, $updateArray['objekt_ort'],true);
        }


        if (($item->geo->land)&&(strval($item->geo->land->attributes()->iso_land) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_land',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND iso_kennzeichen = "'.strval($item->geo->land->attributes()->iso_land).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['objekt_land'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->geo->lage_gebiet)&&(strval($item->geo->lage_gebiet->attributes()->gebiete) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_lage',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->geo->lage_gebiet->attributes()->gebiete).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['objekt_lage'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->preise->preis_zeiteinheit)&&(strval($item->preise->preis_zeiteinheit->attributes()->zeiteinheit) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_periode',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->preise->preis_zeiteinheit->attributes()->zeiteinheit).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['preis_zeiteinheit'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_ist)&&(strval($item->preise->mieteinnahmen_ist->attributes()->periode) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_periode',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->preise->mieteinnahmen_ist->attributes()->periode).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['mieteinnahmen_ist_periode'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_soll)&&(strval($item->preise->mieteinnahmen_soll->attributes()->periode) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_periode',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->preise->mieteinnahmen_soll->attributes()->periode).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['mieteinnahmen_soll_periode'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->preise->waehrung)&&(strval($item->preise->waehrung->attributes()->iso_waehrung) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_waehrung',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND iso_kennzeichen = "'.strval($item->preise->waehrung->attributes()->iso_waehrung).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['waehrung'] = $result[0]['uid'];
            }
            $result = null;
        }else{
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_waehrung',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND iso_kennzeichen = "EUR"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['waehrung'] = $result[0]['uid'];
            }else{
                die('Keine Währung angelegt.');
            }
        }
        if (strval($item->ausstattung->ausstatt_kategorie) != ''){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_austattungskategorie',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->ausstattung->ausstatt_kategorie).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['ausstattungs_kategorie'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->zustand_angaben->zustand)&&(strval($item->zustand_angaben->zustand->attributes()->zustand_art) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_zustand',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->zustand->attributes()->zustand_art).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['zustand'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->zustand_angaben->bebaubar_nach)&&(strval($item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_bebaubarnach',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['bebaubar_nach'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung)&&(strval($item->zustand_angaben->erschliessung->attributes()->erschl_attr) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_erschliessung',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->erschliessung->attributes()->erschl_attr).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['erschliessung'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung_umfang)&&(strval($item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_erschliessungdetails',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['erschliessung_umfang'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->min_mietdauer)&&(strval($item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_periode',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['minimale_mietdauer'] = $result[0]['uid'];
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->max_mietdauer)&&(strval($item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer) != '')){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_periode',
                'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND kuerzel = "'.strval($item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer).'"'
            );
            if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                $updateArray['maximale_mietdauer'] = $result[0]['uid'];
            }
            $result = null;
        }
        $updateArray['kontaktperson'] = $this->addKontaktperson($item);

        $GLOBALS['TYPO3_DB']->exec_UPDATEquery(
            'tx_sgestatecore_domain_model_immobilie',
            'uid='.intval($immobilie),
            $updateArray
        );

    }

    private function addMnAbhaengigkeiten($item, $immobilie){
        $this->setMM($immobilie, $item->objektkategorie->nutzungsart, 'tx_sgestatecore_domain_model_nutzungsart', 'tx_sgestatecore_immobilie_nutzungsart_mm');
        $this->setMM($immobilie, $item->objektkategorie->vermarktungsart, 'tx_sgestatecore_domain_model_vermarktungsart', 'tx_sgestatecore_immobilie_vermarktungsart_mm');
        $this->setMM($immobilie, $item->ausstattung->bad, 'tx_sgestatecore_domain_model_badausstattung', 'tx_sgestatecore_immobilie_badausstattung_mm');
        $this->setMM($immobilie, $item->ausstattung->kueche, 'tx_sgestatecore_domain_model_kuechenausstattung', 'tx_sgestatecore_immobilie_kuechenausstattung_mm');

        $this->setMM($immobilie, $item->ausstattung->boden, 'tx_sgestatecore_domain_model_bodenbelag', 'tx_sgestatecore_immobilie_bodenbelag_mm');
        $this->setMM($immobilie, $item->ausstattung->heizungsart, 'tx_sgestatecore_domain_model_heizungsart','tx_sgestatecore_immobilie_heizungsart_mm');
        $this->setMM($immobilie, $item->ausstattung->befeuerung, 'tx_sgestatecore_domain_model_befeuerungsart', 'tx_sgestatecore_immobilie_befeuerungsart_mm');

        // @todo: ausricht_balkon_terasse int(11) unsigned DEFAULT '0' NOT NULL, Anscheinend Himmelrichtung falsch verMMt
        // $this->setMM($immobilie, $item->ausstattung->ausricht_balkon_terrasse, '', '');

        $this->setMM($immobilie, $item->ausstattung->dachform, 'tx_sgestatecore_domain_model_dachform', 'tx_sgestatecore_immobilie_dachform_mm');
        $this->setMM($immobilie, $item->ausstattung->bauweise, 'tx_sgestatecore_domain_model_bauweise', 'tx_sgestatecore_immobilie_bauweise_mm');
        $this->setMM($immobilie, $item->ausstattung->ausbaustufe, 'tx_sgestatecore_domain_model_ausbaustufe', 'tx_sgestatecore_immobilie_ausbaustufe_mm');
        $this->setMM($immobilie, $item->ausstattung->kampagne, 'tx_sgestatecore_domain_model_kampagne', 'tx_sgestatecore_immobilie_kampagne_mm');

        // @todo: energietyp int(11) unsigned DEFAULT '0', n:1 verknüpft auf M:n umstellen
        // $this->setMM($immobilie, $item->ausstattung->energietyp, 'tx_sgestatecore_domain_model_energietyp', '');



        // @todo: weitere_adressen int(11) unsigned DEFAULT '0' NOT NULL,

    }

    private function setMM($immobilie, $xmlNode, $table, $tableMM ){
        /**
         * @var $xmlNode SimpleXMLElement
         */
        if ($xmlNode){
            $kuerzel = $rows = array();
            $item = null;

            foreach ($xmlNode->attributes() as $key => $value) {
                if ((strval($value) == '1')||(strtolower(strval($value)) == 'true')){
                    array_push($kuerzel, '"'.$key.'"');
                }
            }
            if (count($kuerzel)>0){
                $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows('uid',$table,'sys_language_uid = 0 AND kuerzel in ('.implode(',',$kuerzel).')');
                $GLOBALS['TYPO3_DB']->exec_DELETEquery($tableMM,'uid_local = '.$immobilie);
                foreach ($rows as $item){
                    $GLOBALS['TYPO3_DB']->exec_INSERTquery($tableMM,array('uid_local' => $immobilie,'uid_foreign' => $item['uid']));
                }
            }
        }
    }

    private function addKontaktperson(SimpleXMLElement $item){
        $kontaktpersonDaten = $item->kontaktperson;
        if (is_object($kontaktpersonDaten)){
            $data['pid']                            = $this->storagePid;
            $data['name']                           = strval($kontaktpersonDaten->name);
            $data['vorname']                        = strval($kontaktpersonDaten->vorname);
            $data['titel']                          = strval($kontaktpersonDaten->titel);
            $data['anrede']                         = strval($kontaktpersonDaten->anrede);
            $data['position']                       = strval($kontaktpersonDaten->position);
            $data['anrede_brief']                   = strval($kontaktpersonDaten->anrede_brief);
            $data['firma']                          = strval($kontaktpersonDaten->firma);
            $data['zusatzfeld']                     = strval($kontaktpersonDaten->zusatzfeld);
            $data['strasse']                        = strval($kontaktpersonDaten->strasse);
            $data['hausnummer']                     = strval($kontaktpersonDaten->hausnummer);
            $data['plz']                            = strval($kontaktpersonDaten->plz);
            $data['postfach']                       = strval($kontaktpersonDaten->postfach);
            $data['postfach_plz']                   = strval($kontaktpersonDaten->postf_plz);
            $data['email_zentrale']                 = strval($kontaktpersonDaten->email_zentrale);
            $data['email_direkt']                   = strval($kontaktpersonDaten->email_direkt);
            $data['email_privat']                   = strval($kontaktpersonDaten->email_privat);
            $data['email_sonstige']                 = strval($kontaktpersonDaten->email_sonstige);
            $data['email_feedback']                 = strval($kontaktpersonDaten->email_feedback);
            $data['telefon_zentrale']               = strval($kontaktpersonDaten->tel_zentrale);
            $data['telefon_durchwahl']              = strval($kontaktpersonDaten->tel_durchw);
            $data['telefon_fax']                    = strval($kontaktpersonDaten->tel_fax);
            $data['telefon_handy']                  = strval($kontaktpersonDaten->tel_handy);
            $data['telefon_privat']                 = strval($kontaktpersonDaten->tel_privat);
            $data['telefon_sonstige']               = strval($kontaktpersonDaten->tel_sonstige);
            $data['url']                            = strval($kontaktpersonDaten->url);
            $data['adressfreigabe']                 = 0;
            $data['personennummer']                 = strval($kontaktpersonDaten->personennummer);
            $data['immobilientreuhaenderid']        = strval($kontaktpersonDaten->immobilientreuhaenderid);
            $data['freitextfeld']                   = strval($kontaktpersonDaten->freitextfeld);
            $data['ort']                            = $this->addOrt(strval($kontaktpersonDaten->ort));
            if ((isset($kontaktpersonDaten) && !empty($kontaktpersonDaten->xpath('(land)[1]')))&&(strval($kontaktpersonDaten->land->attributes()->iso_land) != '')){
                $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                    'uid',
                    'tx_sgestatecore_domain_model_land',
                    'hidden=0 AND deleted=0 AND sys_language_uid = 0 AND iso_kennzeichen = "'.strval($item->geo->land->attributes()->iso_land).'"'
                );
                if ((count($result) > 0)&&(is_numeric($result[0]['uid']))){
                    $data['land'] = $result[0]['uid'];
                }
                $result = null;
            }



            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows('uid','tx_sgestatecore_domain_model_kontaktperson','sys_language_uid = 0 AND personennummer LIKE "'.strval($kontaktpersonDaten->personennummer).'"');
            if (count($rows) > 0){
                $uid = intval($rows[0]['uid']);
                $GLOBALS['TYPO3_DB']->exec_UPDATEquery(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    'uid='. $uid,
                    $data
                );
            }else{
                $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    $data
                );
                $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();
            }
            return $uid;
        }else{
            return null;
        }
    }

    private function addOrt($ortname, $fuerSucheVerwenden = null){
        if ($ortname != ''){
            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_ort',
                'sys_language_uid = 0 AND bezeichner LIKE "'.$ortname.'"'
            );
            $data = array();
            $data['pid']        = $this->storagePid;
            $data['bezeichner'] = $ortname;
            if ($fuerSucheVerwenden === true){
                $data['fuer_suche_verwenden'] = 1;
            }



            if (count($rows) > 0){
                $uid = intval($rows[0]['uid']);
                $GLOBALS['TYPO3_DB']->exec_UPDATEquery(
                    'tx_sgestatecore_domain_model_ort',
                    'uid='. $uid,
                    $data
                );
            }else{
                $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                    'tx_sgestatecore_domain_model_ort',
                    $data
                );
                $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();
            }
            return $uid;
        }else{
            return null;
        }

    }

    private function addStadtteil($stadtteilName, $ortUid, $fuerSucheVerwenden = null ){
        if (($stadtteilName != '')&&(is_numeric($ortUid))){
            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_stadtteil',
                'sys_language_uid = 0 AND bezeichner LIKE "'.$stadtteilName.'" AND ort = '.$ortUid
            );
            $data = array();
            $data['pid']        = $this->storagePid;
            $data['ort']        = $ortUid;
            $data['bezeichner'] = $stadtteilName;
            if ($fuerSucheVerwenden === true){
                $data['fuer_suche_verwenden'] = 1;
            }

            if (count($rows) > 0){
                $uid = intval($rows[0]['uid']);
                $GLOBALS['TYPO3_DB']->exec_UPDATEquery(
                    'tx_sgestatecore_domain_model_stadtteil',
                    'uid='. $uid,
                    $data
                );
            }else{
                $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                    'tx_sgestatecore_domain_model_stadtteil',
                    $data
                );
                $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();
            }
            return $uid;
        }else{
            return null;
        }
    }

    private function addAnhaenge($immobilie, $xmlAnhaenge){
        if (is_numeric($immobilie)){
            // Alte Anhänge entfernen
            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid, datei',
                'tx_sgestatecore_domain_model_anhang',
                'immobilie = '.intval($immobilie)
            );
            foreach($rows as $item){
                @unlink($this->mediaPfad.$item['datei']);
                $GLOBALS['TYPO3_DB']->exec_DELETEquery('tx_sgestatecore_domain_model_anhang','uid = '.$item['uid']);
            }


            if ($xmlAnhaenge){
                foreach ($xmlAnhaenge as $anhang){
                    /**
                     * @var $anhang SimpleXMLElement
                     */
                    $quelle = strval($anhang->attributes()->location);
                    switch($quelle){
                        case 'EXTERN':
                            $titel = strval($anhang->anhangtitel);
                            $format = strval($anhang->format);
                            $tempDatei = strval($anhang->daten->pfad);
                            $typ = strval($anhang->attributes()->gruppe);
                            break;
                        case 'INTERN':
                        case 'REMOTE':
                            // @todo: Implementieren
                            break;
                    }
                    $this->addAnhang($immobilie, $titel, $format, $tempDatei, $typ);
                    $titel = $format = $tempDatei = $typ = null;
                }
            }
            return true;
        }else{
            return false;
        }
    }

    private function addAnhang($immobilie, $titel, $format, $tempDatei, $typ){

        if ((strval($tempDatei)!= '')&&(is_numeric($immobilie))){
            if (in_array(strtoupper($format), array('JPG','GIF','PNG','PDF','IMAGE/JPEG','IMAGE/GIF','IMAGE/PNG','APPLICATION/PDF'))){
                // @todo: Standardtyp
                if ($typ == ''){$typ = 'DOKUMENTE';}
                $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                    'uid',
                    'tx_sgestatecore_domain_model_anhangtyp',
                    'sys_language_uid = 0 AND kuerzel LIKE "'.$typ.'"'
                );

                if (count($rows) > 0){
                    $insertArray = array();
                    $insertArray['pid']         = $this->storagePid;
                    $insertArray['typ']         = $rows[0]['uid'];
                    $insertArray['format']      = strtoupper($format);
                    $insertArray['titel']       = $titel;
                    $insertArray['immobilie']   = $immobilie;

                    if ((is_file($this->tempPfad.$tempDatei))&&(rename($this->tempPfad.$tempDatei, $this->mediaPfad.$tempDatei))){
                        $insertArray['datei']       = $tempDatei;
                        $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                        return $GLOBALS['TYPO3_DB']->sql_insert_id();

                    }elseif(is_file($this->mediaPfad.$tempDatei)){
                        $insertArray['datei']       = $tempDatei;
                        $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                        return $GLOBALS['TYPO3_DB']->sql_insert_id();
                    }else{
                        $this->addWarnung('Fehler bei der Anhangzuordnung','Datei konnte nicht kopiert werden');
                    }
                }else{
                    $this->addWarnung('Fehler bei der Anhangzuordnung', 'Es wurde ein ungültiger Typ übergeben: '. $typ);
                }
            }else{
                $this->addWarnung('Fehler bei der Anhangzuordnung', 'Der Anhang ist in keinem zulässigen Format');
            }
        }else{
            $this->addWarnung('Fehler bei der Anhangzuordnung','Es wurde keine Immobilie übergeben');
        }
        return null;
    }

    private function getBoolWert($value){
        if (strval($value) != ''){
            if ((strtolower(strval($value)) == 'true')||(strtolower(strval($value)) == 'ja')||(strtolower(strval($value)) == 'j')){
                return 1;
            }elseif ((strtolower(strval($value)) == 'false')||(strtolower(strval($value)) == 'n')){
                return -1;
            }else{
                return 0;
            }
        }else{
            return 0;
        }
    }

    /**
     * @param $value
     * @return string
     * @throws InvalidTypeException
     */
    private function getDateWert($value){
        try {
            $temp = new DateTime($value);
            return $temp->format('U');
        } catch (Exception $e) {
            throw new InvalidTypeException();
        }
    }

    /**
     * Adds an Entry to the internal Log Array
     *
     * @param $titel
     * @param $text
     */
    private function addWarnung($titel, $text){
        $this->warnungen[] = array(
            'titel' => $titel,
            'text' => $text
        );
    }

    /**
     * Unzips filenameZip to pathExtract
     *
     * @param string $filenameZip
     * @param string $pathExtract
     * @return bool
     */
    private function unzip($filenameZip, $pathExtract){
        $zip = new ZipArchive;
        if ($zip->open($filenameZip) === TRUE){
            $zip->extractTo($pathExtract);
            $zip->close();
            return true;
        }else {
            return false;
        }
    }

    /**
     * Check if all necessary functions are there
     * @return bool
     */
    private function systemcheck(){
        return class_exists('\ZipArchive');
    }

    /********************************************************************************
     * BEGIN
     *
     * Getter, Setter and Function for Scheduler/Task
     ********************************************************************************/

    /**
     * This method returns the destination mail address as additional information
     *
     * @return string Information to display
     */
    public function getAdditionalInformation() {
        $return = 'Immobilienspeicher:'.$this->getStoragePid();
        if ($this->isDebugMode()){
            $return .= ', DEBUGMODE aktiviert';
        }
        return $return;
    }

    /**
     * @param int $storagePid
     */
    public function setStoragePid($storagePid){$this->storagePid = $storagePid;}

    /**
     * @return int
     */
    public function getStoragePid(){return $this->storagePid;}

    /**
     * @return boolean
     */
    public function isDebugMode(){return $this->debugMode;}

    /**
     * @param boolean $debugMode
     * @return $this
     */
    public function setDebugMode($debugMode){$this->debugMode = $debugMode;return $this;}

    /**
     * @return string
     */
    public function getReportSenderEmail()
    {
        return $this->reportSenderEmail;
    }

    /**
     * @param string $reportSenderEmail
     * @return $this
     */
    public function setReportSenderEmail($reportSenderEmail)
    {
        $this->reportSenderEmail = $reportSenderEmail;
        return $this;
    }

    /**
     * @return array
     */
    public function getReportReceiverEmails()
    {
        return $this->reportReceiverEmails;
    }

    /**
     * @param array $reportReceiverEmails
     * @return $this
     */
    public function setReportReceiverEmails($reportReceiverEmails)
    {
        $this->reportReceiverEmails = $reportReceiverEmails;
        return $this;
    }

    /**
     * @return string
     */
    public function getReportSubject()
    {
        return $this->reportSubject;
    }

    /**
     * @param string $reportSubject
     * @return $this
     */
    public function setReportSubject($reportSubject)
    {
        $this->reportSubject = $reportSubject;
        return $this;
    }

    /**
     * @return string
     */
    public function getReportTemplatePath()
    {
        return $this->reportTemplatePath;
    }

    /**
     * @param string $reportTemplatePath
     * @return $this
     */
    public function setReportTemplatePath($reportTemplatePath)
    {
        $this->reportTemplatePath = $reportTemplatePath;
        return $this;
    }

    /**
     * @return int
     */
    public function getDaysToKeepArchiveFiles()
    {
        return $this->daysToKeepArchiveFiles;
    }

    /**
     * @param int $daysToKeepArchiveFiles
     * @return $this
     */
    public function setDaysToKeepArchiveFiles($daysToKeepArchiveFiles)
    {
        $this->daysToKeepArchiveFiles = $daysToKeepArchiveFiles;
        return $this;
    }

    /**
     * @return string
     */
    public function getWorkPath(): string
    {
        return $this->workPath;
    }

    /**
     * @param string $workPath
     */
    public function setWorkPath(string $workPath): void
    {
        $this->workPath = $workPath;
    }

    /********************************************************************************
     * END
     *
     * Getter, Setter and Function for Scheduler/Task
     ********************************************************************************/

}
