<?php
namespace SG\SgEstateImportOpenimmo\Command;

use Doctrine\DBAL\FetchMode;
use SG\SgEstateCore\Util\Services;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\InvalidArgumentException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Cache\Exception\NoSuchCacheException;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Exception;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\Exception\InvalidTypeException;

class ImportCommand extends Command
{
    protected static $defaultName = 'sg.estate.import.openimmo';

    /**
     * @var int $storagePid
     */
    protected $storagePid;

    /**
     * @var array
     */
    private $reportData = [];

    /**
     * @var Services
     */
    private $sgEstateCoreServices;

    /**
     * @var string
     */
    private $tempPath = '';

    /**
     * @var string
     */
    private $archivePath = '';

    /**
     * @var string
     */
    private $uploadPath = '';

    /**
     * @var string
     */
    private $workPath = '';

    /**
     * @var string
     */
    private $mediaPath = '';

    /**
     * @var bool
     */
    private $debugMode = false;

    /**
     * @var string
     */
    private $filenameDatafile = '';

    /**
     * @var bool
     */
    private $importFilePresent = false;

    /**
     * @var int
     */
    private $importIdentifier;

    /**
     * @var ConnectionPool
     */
    private $connectionPool;
    /**
     * @var array
     */
    private $alerts = [];

    protected function configure()
    {
        $this->setName('sg.estate.import.openimmo');
        $this->setDescription('Import von Immobilien aus einer OpenimmoXML-Datei (Bilder und XML Datei in ZIP-Archiv)');
        $this->addArgument('storagePid', InputArgument::REQUIRED, 'Storage Pid for imported realties');
        $this->addArgument('importIdentifier', InputArgument::REQUIRED, 'Unique Identifier for this Import (must be numeric)');
        $this->addArgument('reportSenderEmail', InputArgument::REQUIRED, 'E-Mail Address for sender of report');
        $this->addArgument('reportSubject', InputArgument::REQUIRED, 'Subject of Report E-Mail');
        $this->addArgument('reportReceiverEmails', InputArgument::REQUIRED, 'E-Mail Addresses for receivers of report (commaseparated)');
        $this->addArgument('daysToKeepArchiveFiles', InputArgument::OPTIONAL, 'Days to keep archives Importfiles', '7');
        $this->addArgument('debugMode', InputArgument::OPTIONAL, 'Run in Debug Mode ? ');
        $this->addArgument('forceImportModeFull', InputArgument::OPTIONAL, 'Force "full" import mode and remove all estate items of this task before processing the current import file.');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $this->reportData['importStart'] = date('d.m.Y H:i:s');
        $this->checkAndSetRequirements($input, $output);
        if ($this->preprocess()) {
            $files = scandir($this->workPath);
            sort($files);
            foreach ($files as $file) {
                $tempData = pathinfo($this->workPath . $file);
                if (strtolower($tempData['extension']) == 'zip') {
                    if ($this->unzip($this->workPath . $file, $this->tempPath)) {
                        $this->filenameDatafile = null;
                        $tempFiles = scandir($this->tempPath);
                        foreach ($tempFiles as $dataFile) {
                            $dataFileInfo = pathinfo($this->tempPath . $dataFile);
                            if (strtolower($dataFileInfo['extension']) == 'xml') {
                                $this->filenameDatafile = $dataFile;
                                $this->importFilePresent = true;
                                $this->import($input, $output);
                            }
                        }
                        $this->cleanUpTempDirectory();
                    } else {
                        $this->addAlert('Fehler', 'Datei ' . $file . ' konnte nicht entpackt werden');
                    }
                    rename($this->workPath . $file, $this->archivePath . $tempData['filename'] . '-' . uniqid() . '.zip');
                }
            }
        }
        $this->reportData['importEnde'] = date('d.m.Y H:i:s');
        $this->postprocess($input, $output);
        return 0;
    }

    private function checkAndSetRequirements(InputInterface $input, OutputInterface $output)
    {
        // Check the importIdentifier
        if (!is_numeric($input->getArgument('importIdentifier'))) {
            throw new InvalidArgumentException('importIdentifier MUST be numeric');
        } else {
            $this->importIdentifier = $input->getArgument('importIdentifier');
        }

        // Check the storagePid
        if (!is_numeric($input->getArgument('storagePid'))) {
            throw new InvalidArgumentException('storagePid MUST be numeric');
        } else {
            $this->storagePid = $input->getArgument('storagePid');
        }

        if (!class_exists(\ZipArchive::class)) {
            throw new \TYPO3\CMS\Backend\Exception('ZipArchive is not installed');
        }

        // Creates all needed directories if not present
        $basePathForImportCommand = Environment::getPublicPath() . '/uploads/tx_sgestateimportopenimmo/' . $input->getArgument('importIdentifier');
        if (!is_dir($basePathForImportCommand)) {
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/temp');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/archive');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/upload');
            GeneralUtility::mkdir_deep($basePathForImportCommand . '/work');
        }

        // Set all Path Variables
        $this->tempPath = $basePathForImportCommand . '/temp/';
        $this->archivePath = $basePathForImportCommand . '/archive/';
        $this->uploadPath = $basePathForImportCommand . '/upload/';
        $this->workPath = $basePathForImportCommand . '/work/';
        $this->mediaPath = Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/';

        if ($input->getArgument('debugMode') != '') {
            $this->debugMode = true;
        }

        $this->connectionPool = GeneralUtility::makeInstance(ConnectionPool::class);
        $this->sgEstateCoreServices = GeneralUtility::makeInstance(Services::class);
    }

    private function preprocess()
    {
        $files = scandir($this->uploadPath);
        $atLeastOneFilePresentForImport = false;
        foreach ($files as $file) {
            $atLeastOneFilePresentForImport = true;
            $tempData = pathinfo($this->uploadPath . $file);
            if (strtolower($tempData['extension']) == 'zip') {
                if ($this->debugMode) {
                    copy($this->uploadPath . $file, $this->workPath . $file);
                } else {
                    rename($this->uploadPath . $file, $this->workPath . $file);
                }
            }
        }
        return $atLeastOneFilePresentForImport;
    }

    /**
     * Sends the Import Report
     * Removes Tempfiles
     * @throws NoSuchCacheException
     */
    private function postprocess(InputInterface $input, OutputInterface $output)
    {
        if ($this->importFilePresent) {
            /**
             * @var $cache FrontendInterface
             */
            // Reset Cache
            $estateBaseCache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgestatebase_realtycache');
            $estateCache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgestate_cache');
            $estateBaseCache->flush();
            $estateCache->flush();

            $this->sgEstateCoreServices->sendTemplateEmail(
                explode(',', $input->getArgument('reportReceiverEmails')),
                $input->getArgument('reportSenderEmail'),
                $input->getArgument('reportSubject'),
                'SgEstateImportOpenimmoReport',
                [],
                [
                    'reportDaten' => $this->reportData
                ],
                [],
                'html'
            );
        }

        $this->cleanUpTempDirectory();

        $this->cleanUpArchiveFiles($input, $output);
    }

    /**
     * Cleans up Archive Files
     */
    private function cleanUpArchiveFiles(InputInterface $input, OutputInterface $output)
    {
        $handle = opendir($this->archivePath);
        $now = time();
        $secondsToKeepArchiveFiles = (int)$input->getArgument('daysToKeepArchiveFiles')*24*60*60;
        $borderTimestamp = $now - $secondsToKeepArchiveFiles;

        while ($file = readdir($handle)) {
            $filetime= filemtime($this->archivePath . $file);
            if ($file != '..' && $file != '.' && $filetime < $borderTimestamp) {
                @unlink($this->archivePath . $file);
            }
        }
        closedir($handle);
    }

    private function import(InputInterface $input, OutputInterface $output)
    {
        $dataImport = new \SimpleXMLElement(file_get_contents($this->tempPath . $this->filenameDatafile));

        // Your can force the usage of a full mode if you pass anything in the
        // "forceImportModeFull" attr.
        $isImportModeFull = true;
        $forceImportModeFull = false;
        if ((string)$dataImport->uebertragung[0]->attributes()->umfang !== 'VOLL') {
            $isImportModeFull = false;
        }
        // Keep in mind that "forceImportModeFull" overrides the import mode in the file!
        if ($input->getArgument('forceImportModeFull') !== '') {
            $isImportModeFull = true;
            $forceImportModeFull = true;
        }

        if ($isImportModeFull) {
            $this->sgEstateCoreServices->deleteAllUpdatableRealties($this->importIdentifier);
            $this->reportData['importType'] = 'Full Import';
            if ($forceImportModeFull) {
                $this->reportData['importType'] .= ' (forced)';
            }
        } else {
            $this->reportData['importType'] = 'Partial Import';
        }

        foreach ($dataImport->anbieter[0]->immobilie as $item) {
            /**
             * @var \SimpleXMLElement $item
             */
            switch (strval($item->verwaltung_techn->aktion->attributes()->aktionart)) {
                case 'DELETE':
                    $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');

                    $queryBuilder->getRestrictions()->removeAll();

                    $result = $queryBuilder->select('uid', 'kontaktperson')
                        ->from('tx_sgestatecore_domain_model_immobilie')
                        ->where($queryBuilder->expr()->eq('objektnr_intern', $queryBuilder->createNamedParameter($item->verwaltung_techn->objektnr_intern)))
                        ->execute();

                    if ($result->rowCount() >= 1) {
                        $this->sgEstateCoreServices->deleteRealties2($result->fetchAll(FetchMode::ASSOCIATIVE));
                    }
                    $this->reportData['deleted'][] = strval($item->verwaltung_techn->objektnr_intern);
                    break;
                case 'CHANGE':
                default:
                    $propertyUid = $this->addProperty($item);
                    $this->addN1Connections($item, $propertyUid);
                    $this->addMnConnections($item, $propertyUid);
                    $this->addAttachments($propertyUid, $item->anhaenge->children());
                    break;

            }
        }
    }

    /**
     * @param \SimpleXMLElement $item
     * @return mixed
     * @throws InvalidTypeException
     */
    private function addProperty(\SimpleXMLElement $item)
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $result = $queryBuilder->select('uid')
            ->from('tx_sgestatecore_domain_model_immobilie')
            ->where($queryBuilder->expr()->eq('objektnr_intern', $queryBuilder->createNamedParameter($item->verwaltung_techn->objektnr_intern)))
            ->andWhere($queryBuilder->expr()->eq('import_number', $queryBuilder->createNamedParameter($this->importIdentifier)))
            ->execute();

        $propertyData = $this->generatePropertyData($item);

        // need for sitemap.xml
        $propertyData['tstamp'] = strtotime(date('d.m.Y H:i:s'));

        if ($result->rowCount() >=1) {
            // Update Property
            $uidProperty = $result->fetch()['uid'];
            $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
            $connectionPoolForProperties->update(
                'tx_sgestatecore_domain_model_immobilie',
                $propertyData,
                ['uid' => $uidProperty]
            );
            $this->reportData['updated'][] = strval($item->verwaltung_techn->objektnr_intern);
        } else {
            // Add new Property
            $connectionPoolForProperties = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
            $connectionPoolForProperties->insert(
                'tx_sgestatecore_domain_model_immobilie',
                $propertyData
            );
            $uidProperty = (int)$connectionPoolForProperties->lastInsertId('tx_sgestatecore_domain_model_immobilie');
            $this->reportData['inserted'][] = strval($item->verwaltung_techn->objektnr_intern);
        }

        return $uidProperty;
    }

    private function generatePropertyData(\SimpleXMLElement $item)
    {
        // @todo: objektart_zusatz
        // @todo: unterkellert
        // @todo: stp_carport varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_duplex varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_freiplatz varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_garage varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_parkhaus varchar(255) DEFAULT '' NOT NULL,
        // @todo: stp_tiefgarage varchar(255) DEFAULT '' NOT NULL,
        // @todo: user_defined_simple int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: user_defined_extend int(11) unsigned DEFAULT '0' NOT NULL,
        // @todo: $temp['zulieferung'] = $this->getBoolValue($item->infrastruktur->zulieferung);

        $temp = [];
        $temp['pid'] = $this->storagePid;
        $temp['updatable'] = 1;
        $temp['import_number'] = $this->importIdentifier;
        foreach ($item->children() as $child1) {
            switch ($child1->getName()) {
                case 'geo':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'plz':
                                $temp['objekt_plz'] = strval($child2);
                                break;
                            case 'strasse':
                                $temp['objekt_strasse'] = strval($child2);
                                break;
                            case 'hausnummer':
                                $temp['objekt_hausnummer'] = strval($child2);
                                break;
                            case 'etage':
                                $temp['objekt_etage'] = intval($child2);
                                break;
                            case 'lage_im_bau':
                                $temp['objekt_lage_im_bau_links'] = $this->getBoolValue($child2->attributes()->LINKS);
                                $temp['objekt_lage_im_bau_rechts'] = $this->getBoolValue($child2->attributes()->RECHTS);
                                $temp['objekt_lage_im_bau_vorne'] = $this->getBoolValue($child2->attributes()->VORNE);
                                $temp['objekt_lage_im_bau_hinten'] = $this->getBoolValue($child2->attributes()->HINTEN);
                                break;
                            case 'regionaler_zusatz':
                                $temp['objekt_regionaler_zusatz'] = strval($child2);
                                break;
                            case 'bundesland':
                                $temp['objekt_bundesland'] = strval($child2);
                                break;
                            case 'gemeindecode':
                                $temp['objekt_gemeindecode'] = strval($child2);
                                break;
                            case 'flur':
                                $temp['objekt_flur'] = strval($child2);
                                break;
                            case 'flustueck':
                                $temp['objekt_flurstueck'] = strval($child2);
                                break;
                            case 'gemarkung':
                                $temp['objekt_gemarkung'] = strval($child2);
                                break;
                            case 'anzahl_etagen':
                                $temp['objekt_anzahl_etagen'] = intval($child2);
                                break;
                            case 'wohnungsnr':
                                $temp['objekt_wohnungsnr'] = strval($child2);
                                break;
                            case 'karten_makro':
                                $temp['objekt_karten_makro'] = $this->getBoolValue($child2);
                                break;
                            case 'karten_mikro':
                                $temp['objekt_karten_mikro'] = $this->getBoolValue($child2);
                                break;
                            case 'virtuelletour':
                                $temp['objekt_virtuelletour'] = $this->getBoolValue($child2);
                                break;
                            case 'luftbilder':
                                $temp['objekt_luftbilder'] = $this->getBoolValue($child2);
                                break;
                            case 'geokoordinaten':
                                $temp['objekt_breitengrad'] = str_replace(',', '.', strval($child2->attributes()->breitengrad));
                                $temp['objekt_laengengrad'] = str_replace(',', '.', strval($child2->attributes()->laengengrad));
                                break;
                            case 'user_defined_simplefield':
                                foreach ($child2->attributes()->feldname as $attribute) {
                                    switch ($attribute) {
                                        case 'SearchRegion':
                                            //$temp['search_region'] = strval($child2);
                                            break;
                                        default:
                                            //echo var_dump($child2);die();
                                            break;
                                    }
                                }
                                break;
                            case 'ort':
                            case 'land':
                                break;
                        }
                    }
                    break;
                case 'preise':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'kaufpreis':
                                $temp['kaufpreis'] = floatval($child2);
                                break;
                            case 'kaufpreisnetto':
                                $temp['kaufpreis_netto'] = floatval($child2);
                                $temp['kaufpreis_netto_ust'] = floatval($child2->attributes()->kaufpreisust);
                                break;
                            case 'kaufpreisbrutto':
                                $temp['kaufpreis_brutto'] = floatval($child2);
                                break;
                            case 'nettokaltmiete':
                                $temp['nettokaltmiete'] = floatval($child2);
                                break;
                            case 'kaltmiete':
                                $temp['kaltmiete'] = floatval($child2);
                                break;
                            case 'warmmiete':
                                $temp['warmmiete'] = floatval($child2);
                                break;
                            case 'nebenkosten':
                                $temp['nebenkosten'] = floatval($child2);
                                break;
                            case 'heizkosten_enthalten':
                                $temp['heizkosten_enthalten'] = $this->getBoolValue($child2);
                                break;
                            case 'heizkosten':
                                $temp['heizkosten'] = floatval($child2);
                                break;
                            case 'zzg_mehrwertsteuer':
                                $temp['zzgl_mehrwertsteuer'] = $this->getBoolValue($child2);
                                break;
                            case 'mietzuschlaege':
                                $temp['mietzuschlaege'] = floatval($child2);
                                break;
                            case 'hauptmietzinsnetto':
                                $temp['hauptmietzins_netto'] = floatval($child2);
                                $temp['hauptmietzins_netto_ust'] = floatval($child2->attributes()->hauptmietzinsust);
                                break;
                            case 'pauschalmiete':
                                $temp['pauschalmiete'] = floatval($child2);
                                break;
                            case 'betriebskostennetto':
                                $temp['betriebskosten_netto'] = floatval($child2);
                                $temp['betriebskosten_netto_ust'] = floatval($child2->attributes()->betriebskostenust);
                                break;
                            case 'evbnetto':
                                $temp['evb_netto'] = floatval($child2);
                                $temp['evb_netto_ust'] = floatval($child2);
                                break;
                            case 'gesamtmietenetto':
                                $temp['gesamtmiete_netto'] = floatval($child2);
                                $temp['gesamtmiete_netto_ust'] = floatval($child2->attributes()->gesamtmieteust);
                                break;
                            case 'gesamtmietebrutto':
                                $temp['gesamtmiete_brutto'] = floatval($child2);
                                break;
                            case 'gesamtbelastungnetto':
                                $temp['gesamtbelastung_netto'] = floatval($child2);
                                $temp['gesamtbelastung_netto_ust'] = floatval($child2->attributes()->gesamtbelastungust);
                                break;
                            case 'gesamtbelastungbrutto':
                                $temp['gesamtbelastung_brutto'] = floatval($child2);
                                break;
                            case 'gesamtkostenprom2von':
                                $temp['gesamtkosten_pro_m2_von'] = floatval($child2);
                                $temp['gesamtkosten_pro_m2_bis'] = floatval($child2->attributes()->gesamtkostenprom2bis);
                                break;
                            case 'heizkostennetto':
                                $temp['heizkosten_netto'] = floatval($child2);
                                $temp['heizkosten_netto_ust'] = floatval($child2->attributes()->heizkostenust);
                                break;
                            case 'monatlichekostennetto':
                                $temp['monatlichekosten_netto'] = floatval($child2);
                                $temp['monatlichekosten_netto_ust'] = floatval($child2->attributes()->monatlichekostenust);
                                break;
                            case 'monatlichekostenbrutto':
                                $temp['monatlichekostenbrutto'] = floatval($child2);
                                break;
                            case 'nebenkostenprom2von':
                                $temp['nebenkosten_pro_m2_von'] = floatval($child2);
                                $temp['nebenkosten_pro_m2_bis'] = floatval($child2->attributes()->nebenkostenprom2bis);
                                break;
                            case 'ruecklagenetto':
                                $temp['ruecklage_netto'] = floatval($child2);
                                $temp['ruecklage_netto_ust'] = floatval($child2->attributes()->ruecklageust);
                                break;
                            case 'sonstigekostennetto':
                                $temp['sonstigekosten_netto'] = floatval($child2);
                                $temp['sonstigekosten_netto_ust'] = floatval($child2->attributes()->sonstigekostenust);
                                break;
                            case 'sonstigemietenetto':
                                $temp['sonstigemiete_netto'] = floatval($child2);
                                $temp['sonstigemiete_netto_ust'] = floatval($child2->attributes()->sonstigemieteust);
                                break;
                            case 'summemietenetto':
                                $temp['summemiete_netto'] = floatval($child2);
                                $temp['summemiete_netto_ust'] = floatval($child2->attributes()->sonstigemieteust);
                                break;
                            case 'nettomieteprom2von':
                                $temp['nettomiete_pro_m2_von'] = floatval($child2);
                                $temp['nettomiete_pro_m2_bis'] = floatval($child2->attributes()->nettomieteprom2bis);
                                break;
                            case 'pacht':
                                $temp['pacht'] = floatval($child2);
                                break;
                            case 'erbpacht':
                                $temp['erbpacht'] = floatval($child2);
                                break;
                            case 'hausgeld':
                                $temp['hausgeld'] = floatval($child2);
                                break;
                            case 'abstand':
                                $temp['abstand'] = floatval($child2);
                                break;
                            case 'preis_zeitraum_von':
                                $temp['preis_zeitraum_von'] = $this->getDateValue($child2);
                                break;
                            case 'preis_zeitraum_bis':
                                $temp['preis_zeitraum_bis'] = $this->getDateValue($child2);
                                break;
                            case 'mietpreis_pro_qm':
                                $temp['mietpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'kaufpreis_pro_qm':
                                $temp['kaufpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'provisionspflichtig':
                                $temp['provisionspflichtig'] = $this->getBoolValue($child2);
                                break;
                            case 'innen_courtage':
                                $temp['innencourtage'] = strval($child2);
                                $temp['innencourtage_mit_mwst'] = $this->getBoolValue($child2->attributes()->mit_mwst);
                                break;
                            case 'aussen_courtage':
                                $temp['aussencourtage'] = strval($child2);
                                $temp['aussencourtage_mit_mwst'] = $this->getBoolValue($child2->attributes()->mit_mwst);
                                break;
                            case 'courtage_hinweis':
                                $temp['courtage_hinweis'] = strval($child2);
                                break;
                            case 'provisionnetto':
                                $temp['provision_netto'] = floatval($child2);
                                $temp['provision_netto_ust'] = floatval($child2->attributes()->provisionust);
                                break;
                            case 'provisionbrutto':
                                $temp['provision_brutto'] = floatval($child2);
                                break;
                            case 'mwst_satz':
                                $temp['mwst_satz'] = floatval($child2);
                                break;
                            case 'mwst_gesamt':
                                $temp['mwst_gesamt'] = floatval($child2);
                                break;
                            case 'freitext_preis':
                                $temp['freitext_preis'] = nl2br($child2);
                                break;
                            case 'x_fache':
                                $temp['xfache'] = strval($child2);
                                break;
                            case 'nettorendite':
                                $temp['nettorendite'] = floatval($child2);
                                break;
                            case 'nettorendite_soll':
                                $temp['nettorendite_soll'] = floatval($child2);
                                break;
                            case 'nettorendite_ist':
                                $temp['nettorendite_ist'] = floatval($child2);
                                break;
                            case 'mieteinnahmen_ist':
                                $temp['mieteinnahmen_ist'] = floatval($child2);
                                break;
                            case 'mieteinnahmen_soll':
                                $temp['mieteinnahmen_soll'] = floatval($child2);
                                break;
                            case 'erschliessungskosten':
                                $temp['erschliessungskosten'] = floatval($child2);
                                break;
                            case 'kaution':
                                $temp['kaution'] = floatval($child2);
                                break;
                            case 'kaution_text':
                                $temp['kaution_text'] = strval($child2);
                                break;
                            case 'geschaeftsguthaben':
                                $temp['geschaeftsguthaben'] = floatval($child2);
                                break;
                            case 'richtpreis':
                                $temp['richtpreis'] = floatval($child2);
                                break;
                            case 'richtpreisprom2':
                                $temp['richtpreis_pro_m2'] = floatval($child2);
                                break;
                            case 'user_defined_simplefield':
                                $attribute = strval($child2->attributes()->feldname);
                                switch ($attribute) {
                                    case 'anteile':
                                        $temp['genossenschaftsanteile'] = floatval($child2);
                                        break;
                                    case 'vertragsabschlussgebuehr':
                                        $temp['vertragsabschlussgebuehr'] = floatval($child2);
                                        break;
                                    default:
                                        //echo var_dump($child2);die();
                                        break;
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'flaechen':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'wohnflaeche':
                                $temp['wohnflaeche'] = floatval($item->flaechen->wohnflaeche);
                                break;
                            case 'nutzflaeche':
                                $temp['nutzflaeche'] = floatval($item->flaechen->nutzflaeche);
                                break;
                            case 'gesamtflaeche':
                                $temp['gesamtflaeche'] = floatval($item->flaechen->gesamtflaeche);
                                break;
                            case 'ladenflaeche':
                                $temp['ladenflaeche'] = floatval($item->flaechen->ladenflaeche);
                                break;
                            case 'lagerflaeche':
                                $temp['lagerflaeche'] = floatval($item->flaechen->lagerflaeche);
                                break;
                            case 'verkaufsflaeche':
                                $temp['verkaufsflaeche'] = floatval($item->flaechen->verkaufsflaeche);
                                break;
                            case 'freiflaeche':
                                $temp['freiflaeche'] = floatval($item->flaechen->freiflaeche);
                                break;
                            case 'bueroflaeche':
                                $temp['bueroflaeche'] = floatval($item->flaechen->bueroflaeche);
                                break;
                            case 'bueroteilflaeche':
                                $temp['bueroteilflaeche'] = floatval($item->flaechen->bueroteilflaeche);
                                break;
                            case 'fensterfront':
                                $temp['fensterfront'] = floatval($item->flaechen->fensterfront);
                                break;
                            case 'verwaltungsflaeche':
                                $temp['verwaltungsflaeche'] = floatval($item->flaechen->verwaltungsflaeche);
                                break;
                            case 'gastroflaeche':
                                $temp['gastroflaeche'] = floatval($item->flaechen->gastroflaeche);
                                break;
                            case 'grz':
                                $temp['grundflaechenzahl'] = strval($item->flaechen->grz);
                                break;
                            case 'gfz':
                                $temp['geschossflaechenzahl'] = strval($item->flaechen->gfz);
                                break;
                            case 'bmz':
                                $temp['baumassenzahl'] = strval($item->flaechen->bmz);
                                break;
                            case 'bgf':
                                $temp['bruttogeschossflaechenzahl'] = floatval($item->flaechen->bgf);
                                break;
                            case 'grundstuecksflaeche':
                                $temp['grundstuecksflaeche'] = floatval($item->flaechen->grundstuecksflaeche);
                                break;
                            case 'sonstflaeche':
                                $temp['sonstflaeche'] = floatval($item->flaechen->sonstflaeche);
                                break;
                            case 'anzahl_zimmer':
                                $temp['anzahl_zimmer'] = floatval($item->flaechen->anzahl_zimmer);
                                break;
                            case 'anzahl_schlafzimmer':
                                $temp['anzahl_schlafzimmer'] = floatval($item->flaechen->anzahl_schlafzimmer);
                                break;
                            case 'anzahl_badezimmer':
                                $temp['anzahl_badezimmer'] = floatval($item->flaechen->anzahl_badezimmer);
                                break;
                            case 'anzahl_sep_wc':
                                $temp['anzahl_separate_wc'] = floatval($item->flaechen->anzahl_sep_wc);
                                break;
                            case 'balkon_terrasse_flaeche':
                                $temp['balkon_terasse_flaeche'] = floatval($item->flaechen->balkon_terrasse_flaeche);
                                break;
                            case 'anzahl_wohn_schlafzimmer':
                                $temp['anzahl_wohn_schlafzimmer'] = floatval($item->flaechen->anzahl_wohn_schlafzimmer);
                                break;
                            case 'gartenflaeche':
                                $temp['gartenflaeche'] = floatval($item->flaechen->gartenflaeche);
                                break;
                            case 'anzahl_balkone':
                                $temp['anzahl_balkone'] = floatval($item->flaechen->anzahl_balkone);
                                break;
                            case 'anzahl_terrassen':
                                $temp['anzahl_terassen'] = floatval($item->flaechen->anzahl_terrassen);
                                break;
                            case 'anzahl_logia':
                                $temp['anzahl_logia'] = floatval($item->flaechen->anzahl_logia);
                                break;
                            case 'fensterfront_qm':
                                $temp['fensterfront_m2'] = floatval($item->flaechen->fensterfront_qm);
                                break;
                            case 'grundstuecksfront':
                                $temp['grundstuecksfront'] = floatval($item->flaechen->grundstuecksfront);
                                break;
                            case 'dachbodenflaeche':
                                $temp['dachbodenflaeche'] = floatval($item->flaechen->dachbodenflaeche);
                                break;
                            case 'teilbar_ab':
                                $temp['teilbar_ab'] = floatval($item->flaechen->teilbar_ab);
                                break;
                            case 'kellerflaeche':
                                $temp['kellerflaeche'] = floatval($item->flaechen->kellerflaeche);
                                break;
                            case 'beheizbare_flaeche':
                                $temp['beheizbare_flaeche'] = floatval($item->flaechen->beheizbare_flaeche);
                                break;
                            case 'anzahl_stellplaetze':
                                $temp['anzahl_stellplaetze'] = floatval($item->flaechen->anzahl_stellplaetze);
                                break;
                            case 'plaetze_gastraum':
                                $temp['plaetze_gastraum'] = floatval($item->flaechen->plaetze_gastraum);
                                break;
                            case 'anzahl_betten':
                                $temp['anzahl_betten'] = floatval($item->flaechen->anzahl_betten);
                                break;
                            case 'anzahl_tagungsraeume':
                                $temp['anzahl_tagungsraeume'] = floatval($item->flaechen->anzahl_tagungsraeume);
                                break;
                            case 'vermietbare_flaeche':
                                $temp['vermietbare_flaeche'] = floatval($item->flaechen->vermietbare_flaeche);
                                break;
                            case 'anzahl_wohneinheiten':
                                $temp['anzahl_wohneinheiten'] = floatval($item->flaechen->anzahl_wohneinheiten);
                                break;
                            case 'anzahl_gewerbeeinheiten':
                                $temp['anzahl_gewerbeeinheiten'] = floatval($item->flaechen->anzahl_gewerbeeinheiten);
                                break;
                            case 'einliegerwohnung':
                                $temp['einliegerwohnung'] = $this->getBoolValue($item->flaechen->einliegerwohnung);
                                break;
                            case 'kubatur':
                                $temp['kubatur'] = floatval($item->flaechen->kubatur);
                                break;
                            case 'ausnuetzungsziffer':
                                $temp['ausnuetzungsziffer'] = floatval($item->flaechen->ausnuetzungsziffer);
                                break;
                            case 'flaechevon':
                                $temp['flaechevon'] = floatval($item->flaechen->flaechevon);
                                break;
                            case 'flaechebis':
                                $temp['flaechebis'] = floatval($item->flaechen->flaechebis);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'ausstattung':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'wg_geeignet':
                                $temp['wg_geeignet'] = $this->getBoolValue($item->ausstattung->wg_geeignet);
                                break;
                            case 'raeume_veraenderbar':
                                $temp['raeume_veraenderbar'] = $this->getBoolValue($item->ausstattung->raeume_veraenderbar);
                                break;
                            case 'kamin':
                                $temp['kamin'] = $this->getBoolValue($item->ausstattung->kamin);
                                break;
                            case 'klimatisiert':
                                $temp['klimatisiert'] = $this->getBoolValue($item->ausstattung->klimatisiert);
                                break;
                            case 'fahrstuhl':
                                $temp['aufzug_personen'] = $this->getBoolValue($item->ausstattung->fahrstuhl->attributes()->PERSONEN);
                                //$temp['aufzug_personen'] = $this->getBoolValue($item->ausstattung->fahrstuhl->attributes()->LASTEN);
                                break;
                            case 'gartennutzung':
                                $temp['gartennutzung'] = $this->getBoolValue($item->ausstattung->gartennutzung);
                                break;
                            case 'moebliert':
                                $temp['moebliert'] = strval($item->ausstattung->moebliert->attributes()->moeb);
                                break;
                            case 'rollstuhlgerecht':
                                $temp['rollstuhlgerecht'] = $this->getBoolValue($item->ausstattung->rollstuhlgerecht);
                                break;
                            case 'kabel_sat_tv':
                                $temp['kabel_sat_tv'] = $this->getBoolValue($item->ausstattung->kabel_sat_tv);
                                $temp['kabelfernsehen'] = $this->getBoolValue($item->ausstattung->kabel_sat_tv);
                                break;
                            case 'dvbt':
                                $temp['dvbt'] = $this->getBoolValue($item->ausstattung->dvbt);
                                break;
                            case 'barrierefrei':
                                $temp['barrierefrei'] = $this->getBoolValue($item->ausstattung->barrierefrei);
                                break;
                            case 'sauna':
                                $temp['sauna'] = $this->getBoolValue($item->ausstattung->sauna);
                                break;
                            case 'swimmingpool':
                                $temp['swimmingpool'] = $this->getBoolValue($item->ausstattung->swimmingpool);
                                break;
                            case 'wasch_trockenraum':
                                $temp['wasch_trockenraum'] = $this->getBoolValue($item->ausstattung->wasch_trockenraum);
                                break;
                            case 'wintergarten':
                                $temp['wintergarten'] = $this->getBoolValue($item->ausstattung->wintergarten);
                                break;
                            case 'dv_verkabelung':
                                $temp['dv_verkabelung'] = $this->getBoolValue($item->ausstattung->dv_verkabelung);
                                break;
                            case 'rampe':
                                $temp['rampe'] = $this->getBoolValue($item->ausstattung->rampe);
                                break;
                            case 'hebebuehne':
                                $temp['hebebuehne'] = $this->getBoolValue($item->ausstattung->hebebuehne);
                                break;
                            case 'kran':
                                $temp['kran'] = $this->getBoolValue($item->ausstattung->kran);
                                break;
                            case 'gastterrasse':
                                $temp['gastterrasse'] = $this->getBoolValue($item->ausstattung->gastterrasse);
                                break;
                            case 'stromanschlusswert':
                                $temp['stromanschlusswert'] = floatval($item->ausstattung->stromanschlusswert);
                                break;
                            case 'kantine_cafeteria':
                                $temp['katine_cafeteria'] = $this->getBoolValue($item->ausstattung->kantine_cafeteria);
                                break;
                            case 'teekueche':
                                $temp['teekueche'] = $this->getBoolValue($item->ausstattung->teekueche);
                                break;
                            case 'hallenhoehe':
                                $temp['hallenhoehe'] = floatval($item->ausstattung->hallenhoehe);
                                break;
                            case 'angeschl_gastronomie':
                                $temp['hotelrestaurant_angeschlossen'] = $this->getBoolValue($item->ausstattung->angeschl_gastronomie->attributes()->HOTELRESTAURANT);
                                $temp['bar_angeschlossen'] = $this->getBoolValue($item->ausstattung->angeschl_gastronomie->attributes()->BAR);
                                break;
                            case 'brauereibindung':
                                $temp['brauereibindung'] = $this->getBoolValue($item->ausstattung->brauereibindung);
                                break;
                            case 'sporteinrichtungen':
                                $temp['sporteinrichtungen'] = $this->getBoolValue($item->ausstattung->sporteinrichtungen);
                                break;
                            case 'wellnessbereich':
                                $temp['wellnessbereich'] = $this->getBoolValue($item->ausstattung->wellnessbereich);
                                break;
                            case 'serviceleistungen':
                                $temp['service_betreutes_wohnen'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->BETREUTES_WOHNEN);
                                $temp['service_catering'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->CATERING);
                                $temp['service_reinigung'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->REINIGUNG);
                                $temp['service_einkauf'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->EINKAUF);
                                $temp['service_wachdienst'] = $this->getBoolValue($item->ausstattung->serviceleistungen->attributes()->WACHDIENST);
                                break;
                            case 'telefon_ferienimmobilie':
                                $temp['telefon_ferienimmobilie'] = $this->getBoolValue($item->ausstattung->telefon_ferienimmobilie);
                                break;
                            case 'breitband_zugang':
                                $temp['breitband_zugang_art'] = strval($item->ausstattung->breitband_zugang->attributes()->art);
                                $temp['breitband_zugang_speed'] = floatval($item->ausstattung->breitband_zugang->attributes()->speed);
                                break;
                            case 'umts_empfang':
                                $temp['umts_empfang'] = $this->getBoolValue($item->ausstattung->umts_empfang);
                                break;
                            case 'sicherheitstechnik':
                                $temp['sicherheit_alarmanlage'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->ALARMANLAGE);
                                $temp['sicherheit_kamera'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->KAMERA);
                                $temp['sicherheit_polizeiruf'] = $this->getBoolValue($item->ausstattung->sicherheitstechnik->attributes()->POLIZEIRUF);
                                break;
                            case 'abstellraum':
                                $temp['abstellraum'] = $this->getBoolValue($item->ausstattung->abstellraum);
                                break;
                            case 'fahrradraum':
                                $temp['fahrradraum'] = $this->getBoolValue($item->ausstattung->fahrradraum);
                                break;
                            case 'rolladen':
                                $temp['rolladen'] = $this->getBoolValue($item->ausstattung->rolladen);
                                break;
                            case 'bibliothek':
                                $temp['bibliothek'] = $this->getBoolValue($item->ausstattung->bibliothek);
                                break;
                            case 'dachboden':
                                $temp['dachboden'] = $this->getBoolValue($item->ausstattung->dachboden);
                                break;
                            case 'gaestewc':
                                $temp['gaestewc'] = $this->getBoolValue($item->ausstattung->gaestewc);
                                break;
                            case 'kabelkanaele':
                                $temp['kabelkanaele'] = $this->getBoolValue($item->ausstattung->kabelkanaele);
                                break;
                            case 'seniorengerecht':
                                $temp['seniorengerecht'] = $this->getBoolValue($item->ausstattung->seniorengerecht);
                                break;
                            case 'unterkellert':
                                $temp['unterkellert'] = $this->getBoolValue($item->ausstattung->unterkellert->attributes()->keller);
                                break;
                            case 'user_defined_simplefield':
                                foreach ($child2->attributes() as $attribute) {
                                    switch (strval($attribute)) {
                                        case 'gegensprechanlage':
                                            $temp['gegensprechanlage'] = $this->getBoolValue($child2);
                                            break;
                                        case 'leicht_erreichbar':
                                            $temp['leicht_erreichbar'] = $this->getBoolValue($child2);
                                            break;
                                        case 'bad_mit_fenster':
                                            if ($child1->bad) {
                                                if ($child2 == 'true') {
                                                    $child1->bad->addAttribute('FENSTER', 'true');
                                                }
                                            } else {
                                                $child1->addChild('bad');
                                                if ($child2 == 'true') {
                                                    $child1->bad->addAttribute('FENSTER', 'true');
                                                }
                                            }
                                            break;
                                        case 'wunschwohnung':
                                            if (!($child1->kampagne)) {
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true') {
                                                $child1->kampagne->addAttribute('WUNSCHWOHNUNG', 'true');
                                            }
                                            break;
                                        case 'junges_wohnen':
                                            if (!($child1->kampagne)) {
                                                $child1->addChild('kampagne');
                                            }

                                            if ($child2 == 'true') {
                                                $child1->kampagne->addAttribute('JUNGESWOHNEN', 'true');
                                            }
                                            break;
                                        case 'bad_modernisiert':
                                            $temp['bad_modernisiert'] = $this->getBoolValue($child2);
                                            break;
                                        case 'hd_satellitenfernsehen':
                                            $temp['satellitenfernsehen'] = $this->getBoolValue($child2);
                                            break;
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'zustand_angaben':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'baujahr':
                                $temp['baujahr'] = strval($item->zustand_angaben->baujahr);
                                break;
                            case 'letztemodernisierung':
                                $temp['letztemodernisierung'] = strval($item->zustand_angaben->letztemodernisierung);
                                break;
                            case 'bauzone':
                                $temp['bauzone'] = strval($item->zustand_angaben->bauzone);
                                break;
                            case 'altlasten':
                                $temp['altlasten'] = strval($item->zustand_angaben->altlasten);
                                break;
                            case 'energiepass':
                                $temp['energiepass_energieverbrauchkennwert'] = strval($item->zustand_angaben->energiepass->energieverbrauchkennwert);
                                $temp['energiepass_mitwarmwasser'] = $this->getBoolValue($item->zustand_angaben->energiepass->mitwarmwasser);
                                $temp['energiepass_endenergiebedarf'] = strval($item->zustand_angaben->energiepass->endenergiebedarf);
                                $temp['energiepass_baujahr'] = strval($item->zustand_angaben->energiepass->baujahr);
                                $temp['energiepass_wertklasse'] = strval($item->zustand_angaben->energiepass->wertklasse);
                                $temp['energiepass_ausstelldatum'] = $this->getDateValue($item->zustand_angaben->energiepass->ausstelldatum);

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassart')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->energiepass->epart))))
                                    ->execute();

                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['energiepass_art'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_befeuerungsart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_befeuerungsart')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->energiepass->primaerenergietraeger))))
                                    ->execute();

                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['energiepass_primaerenergietraeger'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassjahrgang');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassjahrgang')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->energiepass->jahrgang))))
                                    ->execute();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['energiepass_jahrgang'] = $resultItemUid;
                                }
                                $result = null;

                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_energiepassgebaeudeart');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_energiepassgebaeudeart')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->energiepass->gebaeudeart))))
                                    ->execute();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['energiepass_gebaeudeart'] = $resultItemUid;
                                }
                                $result = null;

                                break;
                            case 'alter':
                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_altertyp');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_altertyp')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->alter->attributes()->alter_attr))))
                                    ->execute();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['altertyp'] = $resultItemUid;
                                }
                                $result = null;
                                break;
                            case 'verkaufstatus':
                                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_verkaufstatus');
                                $result = $queryBuilder
                                    ->select('uid')
                                    ->from('tx_sgestatecore_domain_model_verkaufstatus')
                                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->verkaufstatus))))
                                    ->execute();
                                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                                    $temp['verkaufstatus'] = $resultItemUid;
                                }
                                $result = null;
                                break;
                            case 'user_defined_simplefield':
                            default:
                                break;
                        }
                    }
                    break;
                case 'freitexte':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'objekttitel':
                                $temp['objekttitel'] = nl2br($item->freitexte->objekttitel);
                                break;
                            case 'dreizeiler':
                                $temp['kurzbeschreibung'] = nl2br($item->freitexte->dreizeiler);
                                break;
                            case 'lage':
                                $temp['lage'] = nl2br($item->freitexte->lage);
                                break;
                            case 'ausstatt_beschr':
                                $temp['austattungsbeschreibung'] = nl2br($item->freitexte->ausstatt_beschr);
                                break;
                            case 'objektbeschreibung':
                                $temp['objektbeschreibung'] = nl2br($item->freitexte->objektbeschreibung);
                                break;
                            case 'sonstige_angaben':
                                $temp['sonstige_angaben'] = nl2br($item->freitexte->sonstige_angaben);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_objekt':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'objektadresse_freigeben':
                                $temp['objektadresse_freigeben'] = $this->getBoolValue($item->verwaltung_objekt->objektadresse_freigeben);
                                break;
                            case 'verfuegbar_ab':
                                $temp['verfuegbar_ab'] = strval($item->verwaltung_objekt->verfuegbar_ab);
                                break;
                            case 'abdatum':
                                $temp['verfuegbar_ab_datum'] = $this->getDateValue($item->verwaltung_objekt->abdatum);
                                break;
                            case 'bisdatum':
                                $temp['verfuegbar_bis_datum'] = $this->getDateValue($item->verwaltung_objekt->bisdatum);
                                break;
                            case 'versteigerungstermin':
                                $temp['versteigerungstermin'] = $this->getDateValue($item->verwaltung_objekt->versteigerungstermin);
                                break;
                            case 'wbs_sozialwohnung':
                                $temp['wbs_erforderlich'] = $this->getBoolValue($item->verwaltung_objekt->wbs_sozialwohnung);
                                break;
                            case 'vermietet':
                                $temp['vermietet'] = $this->getBoolValue($item->verwaltung_objekt->vermietet);
                                break;
                            case 'gruppennummer':
                                $temp['gruppennummer'] = strval($item->verwaltung_objekt->gruppennummer);
                                break;
                            case 'zugang':
                                $temp['zugang'] = strval($item->verwaltung_objekt->zugang);
                                break;
                            case 'laufzeit':
                                $temp['laufzeit'] = floatval($item->verwaltung_objekt->laufzeit);
                                break;
                            case 'max_personen':
                                $temp['maximale_personenanzahl'] = intval($item->verwaltung_objekt->max_personen);
                                break;
                            case 'nichtraucher':
                                $temp['nichtraucher'] = $this->getBoolValue($item->verwaltung_objekt->nichtraucher);
                                break;
                            case 'haustiere':
                                $temp['haustiere'] = $this->getBoolValue($item->verwaltung_objekt->haustiere);
                                break;
                            case 'denkmalgeschuetzt':
                                $temp['denkmalgeschuetzt'] = $this->getBoolValue($item->verwaltung_objekt->denkmalgeschuetzt);
                                break;
                            case 'als_ferien':
                                $temp['als_ferien'] = $this->getBoolValue($item->verwaltung_objekt->als_ferien);
                                break;
                            case 'gewerbliche_nutzung':
                                $temp['gewerbliche_nutzung'] = $this->getBoolValue($item->verwaltung_objekt->gewerbliche_nutzung);
                                break;
                            case 'branchen':
                                $temp['branchen'] = strval($item->verwaltung_objekt->branchen);
                                break;
                            case 'hochhaus':
                                $temp['hochhaus'] = $this->getBoolValue($item->verwaltung_objekt->hochhaus);
                                break;
                            case 'aktiv_von':
                                $temp['anzeige_ab'] = $this->getDateValue($item->verwaltung_objekt->aktiv_von);
                                break;
                            case 'aktiv_bis':
                                $temp['anzeige_bis'] = $this->getDateValue($item->verwaltung_objekt->aktiv_bis);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'verwaltung_techn':
                    foreach ($child1->children() as $child2) {
                        switch ($child2->getName()) {
                            case 'openimmo_obid':
                                $temp['openimmo_objid'] = strval($item->verwaltung_techn->openimmo_obid);
                                break;
                            case 'objektnr_intern':
                                $temp['objektnr_intern'] = strval($item->verwaltung_techn->objektnr_intern);
                                break;
                            case 'objektnr_extern':
                                $temp['objektnr_extern'] = strval($item->verwaltung_techn->objektnr_extern);
                                break;
                            case 'kennung_ursprung':
                                $temp['kennung_ursprung'] = strval($item->verwaltung_techn->kennung_ursprung);
                                break;
                            case 'stand_vom':
                                $temp['stand_vom'] = $this->getDateValue($item->verwaltung_techn->stand_vom);
                                break;
                            case 'weitergabe_generell':
                                $temp['weitergabe_generell'] = $this->getBoolValue($item->verwaltung_techn->weitergabe_generell);
                                break;
                            case 'weitergabe_positivliste':
                                $temp['weitergabe_positivliste'] = strval($item->verwaltung_techn->weitergabe_positivliste);
                                break;
                            case 'weitergabe_negativliste':
                                $temp['weitergabe_negativliste'] = strval($item->verwaltung_techn->weitergabe_negativliste);
                                break;
                            case 'gruppen_kennung':
                                $temp['gruppen_kennung'] = strval($item->verwaltung_techn->gruppen_kennung);
                                break;
                            case 'master':
                                $temp['master_objekt'] = strval($item->verwaltung_techn->master);
                                $temp['master_visible'] = $this->getBoolValue($item->verwaltung_techn->master->attributes()->visible);
                                break;
                            case 'sprache':
                                $temp['sprache'] = strval($item->verwaltung_techn->sprache);
                                break;
                            default:
                                break;
                        }
                    }
                    break;
                case 'kontaktperson':
                case 'anhaenge':
                case 'objektkategorie':
                    break;
            }
        }
        return $temp;
    }

    private function addN1Connections($item, $propertyUid)
    {
        $updateArray = [];

        if (($item->infrastruktur->ausblick)&&(strval($item->infrastruktur->ausblick->attributes()->blick) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ausblick');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_ausblick')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->infrastruktur->ausblick->attributes()->blick))))
                ->execute();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['ausblick'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->geschlecht)&&(strval($item->verwaltung_objekt->geschlecht->attributes()->geschl_attr) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_geschlecht');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_geschlecht')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->verwaltung_objekt->geschlecht->attributes()->geschl_attr))))
                ->execute();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['geschlecht'] = $resultItemUid;
            }
            $result = null;
        }
        if ($item->objektkategorie->objektart) {
            foreach ($item->objektkategorie->objektart->children() as $child) {
                /**
                 * @var $child SimpleXMLElement
                 */
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektart');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_objektart')
                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($child->getName()))))
                    ->execute();
                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                    $updateArray['objektart'] = $resultItemUid;
                }

                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektarttyp');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_objektarttyp')
                    ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($child[$child->getName() . 'typ']))))
                    ->execute();

                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                    $updateArray['objektarttyp'] = $resultItemUid;
                } else {
                    $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_objektarttyp');
                    $result = $queryBuilder
                        ->select('uid')
                        ->from('tx_sgestatecore_domain_model_objektarttyp')
                        ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($child[$child->getName() . '_typ']))))
                        ->execute();
                    if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                        $updateArray['objektarttyp'] = $resultItemUid;
                    }
                }
            }
        }

        $updateArray['objekt_ort'] = $this->addCity(strval($item->geo->ort), true);
        if (strval($item->geo->regionaler_zusatz) == '') {
            foreach ($item->geo->children() as $subnode) {
                /**
                 * @var $subnode SimpleXMLElement
                 */
                switch ($subnode->getName()) {
                    case 'user_defined_simplefield':
                        foreach ($subnode->attributes() as $attribute) {
                            switch (strval($attribute)) {
                                case 'SearchRegion': // Immosolve District
                                case 'ZONE': //Immoblue District
                                    if (strval($subnode) != '') {
                                        $updateArray['objekt_stadtteil'] = $this->addDistrict(strval($subnode), $updateArray['objekt_ort'], true);
                                    }
                                    break;
                            }
                        }
                        break;
                }
            }
        } else {
            $updateArray['objekt_stadtteil'] = $this->addDistrict($item->geo->regionaler_zusatz, $updateArray['objekt_ort'], true);
        }

        if (($item->geo->land)&&(strval($item->geo->land->attributes()->iso_land) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_land');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_land')
                ->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter(strval($item->geo->land->attributes()->iso_land))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['objekt_land'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->geo->lage_gebiet)&&(strval($item->geo->lage_gebiet->attributes()->gebiete) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_lage');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_lage')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->geo->lage_gebiet->attributes()->gebiete))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['objekt_lage'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->preis_zeiteinheit)&&(strval($item->preise->preis_zeiteinheit->attributes()->zeiteinheit) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->preise->preis_zeiteinheit->attributes()->zeiteinheit))))
                ->execute();

            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['preis_zeiteinheit'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_ist)&&(strval($item->preise->mieteinnahmen_ist->attributes()->periode) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->preise->mieteinnahmen_ist->attributes()->periode))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['mieteinnahmen_ist_periode'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->mieteinnahmen_soll)&&(strval($item->preise->mieteinnahmen_soll->attributes()->periode) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->preise->mieteinnahmen_soll->attributes()->periode))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['mieteinnahmen_soll_periode'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->preise->waehrung)&&(strval($item->preise->waehrung->attributes()->iso_waehrung) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_waehrung');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_waehrung')
                ->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter(strval($item->preise->waehrung->attributes()->iso_waehrung))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['waehrung'] = $resultItemUid;
            }
            $result = null;
        } else {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_waehrung');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_waehrung')
                ->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter('EUR')))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['waehrung'] = $resultItemUid;
            } else {
                die('Keine Währung angelegt.');
            }
        }
        if (strval($item->ausstattung->ausstatt_kategorie) != '') {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_austattungskategorie');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_austattungskategorie')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->ausstattung->ausstatt_kategorie))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['ausstattungs_kategorie'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->zustand)&&(strval($item->zustand_angaben->zustand->attributes()->zustand_art) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_zustand');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_zustand')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->zustand->attributes()->zustand_art))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['zustand'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->bebaubar_nach)&&(strval($item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_bebaubarnach');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_bebaubarnach')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->bebaubar_nach->attributes()->bebaubar_attr))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['bebaubar_nach'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung)&&(strval($item->zustand_angaben->erschliessung->attributes()->erschl_attr) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_erschliessung');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_erschliessung')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->erschliessung->attributes()->erschl_attr))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['erschliessung'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->zustand_angaben->erschliessung_umfang)&&(strval($item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_erschliessungdetails');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_erschliessungdetails')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->zustand_angaben->erschliessung_umfang->attributes()->erschl_attr))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['erschliessung_umfang'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->min_mietdauer)&&(strval($item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer) != '')) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_periode');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->verwaltung_objekt->min_mietdauer->attributes()->min_dauer))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['minimale_mietdauer'] = $resultItemUid;
            }
            $result = null;
        }
        if (($item->verwaltung_objekt->max_mietdauer)&&(strval($item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer) != '')) {
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_periode')
                ->where($queryBuilder->expr()->eq('kuerzel', $queryBuilder->createNamedParameter(strval($item->verwaltung_objekt->max_mietdauer->attributes()->max_dauer))))
                ->execute();
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $updateArray['maximale_mietdauer'] = $resultItemUid;
            }
            $result = null;
        }
        $updateArray['kontaktperson'] = $this->addContactperson($item);

        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
        $connection->update(
            'tx_sgestatecore_domain_model_immobilie',
            $updateArray,
            ['uid' => (int)$propertyUid]
        );
    }

    private function addMnConnections($item, $propertyUid)
    {
        if (is_object($item->objektkategorie->nutzungsart)) {
            $this->setMM($propertyUid, $item->objektkategorie->nutzungsart, 'tx_sgestatecore_domain_model_nutzungsart', 'tx_sgestatecore_immobilie_nutzungsart_mm');
        }

        if (is_object($item->objektkategorie->vermarktungsart)) {
            $this->setMM($propertyUid, $item->objektkategorie->vermarktungsart, 'tx_sgestatecore_domain_model_vermarktungsart', 'tx_sgestatecore_immobilie_vermarktungsart_mm');
        }

        if (is_object($item->ausstattung->bad)) {
            $this->setMM($propertyUid, $item->ausstattung->bad, 'tx_sgestatecore_domain_model_badausstattung', 'tx_sgestatecore_immobilie_badausstattung_mm');
        }

        if (is_object($item->ausstattung->kueche)) {
            $this->setMM($propertyUid, $item->ausstattung->kueche, 'tx_sgestatecore_domain_model_kuechenausstattung', 'tx_sgestatecore_immobilie_kuechenausstattung_mm');
        }

        if (is_object($item->ausstattung->boden)) {
            $this->setMM($propertyUid, $item->ausstattung->boden, 'tx_sgestatecore_domain_model_bodenbelag', 'tx_sgestatecore_immobilie_bodenbelag_mm');
        }

        if (is_object($item->ausstattung->heizungsart)) {
            $this->setMM($propertyUid, $item->ausstattung->heizungsart, 'tx_sgestatecore_domain_model_heizungsart', 'tx_sgestatecore_immobilie_heizungsart_mm');
        }

        if (is_object($item->ausstattung->befeuerung)) {
            $this->setMM($propertyUid, $item->ausstattung->befeuerung, 'tx_sgestatecore_domain_model_befeuerungsart', 'tx_sgestatecore_immobilie_befeuerungsart_mm');
        }

        // @todo: ausricht_balkon_terasse int(11) unsigned DEFAULT '0' NOT NULL, Anscheinend Himmelrichtung falsch verMMt
        // $this->setMM($propertyUid, $item->ausstattung->ausricht_balkon_terrasse, '', '');

        if (is_object($item->ausstattung->dachform)) {
            $this->setMM($propertyUid, $item->ausstattung->dachform, 'tx_sgestatecore_domain_model_dachform', 'tx_sgestatecore_immobilie_dachform_mm');
        }

        if (is_object($item->ausstattung->bauweise)) {
            $this->setMM($propertyUid, $item->ausstattung->bauweise, 'tx_sgestatecore_domain_model_bauweise', 'tx_sgestatecore_immobilie_bauweise_mm');
        }

        if (is_object($item->ausstattung->ausbaustufe)) {
            $this->setMM($propertyUid, $item->ausstattung->ausbaustufe, 'tx_sgestatecore_domain_model_ausbaustufe', 'tx_sgestatecore_immobilie_ausbaustufe_mm');
        }

        if (is_object($item->ausstattung->kampagne)) {
            $this->setMM($propertyUid, $item->ausstattung->kampagne, 'tx_sgestatecore_domain_model_kampagne', 'tx_sgestatecore_immobilie_kampagne_mm');
        }

        // @todo: energietyp int(11) unsigned DEFAULT '0', n:1 verknüpft auf M:n umstellen
        // $this->setMM($propertyUid, $item->ausstattung->energietyp, 'tx_sgestatecore_domain_model_energietyp', '');

        // @todo: weitere_adressen int(11) unsigned DEFAULT '0' NOT NULL,
    }

    private function setMM($propertyUid, \SimpleXMLElement $xmlNode, $table, $tableMM)
    {
        if ($xmlNode) {
            $kuerzel = $rows = [];
            $item = null;

            foreach ($xmlNode->attributes() as $key => $value) {
                if ((strval($value) == '1')||(strtolower(strval($value)) == 'true')) {
                    array_push($kuerzel, '"' . $key . '"');
                }
            }
            if (count($kuerzel)>0) {
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable($table);
                $result = $queryBuilder
                    ->select('uid')
                    ->from($table)
                    ->where($queryBuilder->expr()->in('kuerzel', implode(',', $kuerzel)))
                    ->execute();
                $rows = $result->fetchAll(FetchMode::ASSOCIATIVE);

                $queryBuilder = $this->connectionPool->getQueryBuilderForTable($tableMM);
                $queryBuilder
                    ->delete($tableMM)
                    ->where($queryBuilder->expr()->eq('uid_local', $queryBuilder->createNamedParameter($propertyUid)))
                    ->execute();
                $connection = $this->connectionPool->getConnectionForTable($tableMM);
                foreach ($rows as $item) {
                    $connection->insert(
                        $tableMM,
                        [
                            'uid_local' => $propertyUid,
                            'uid_foreign' => $item['uid']
                        ]
                    );
                }
            }
        }
    }

    private function addAttachments($propertyUid, $xmlAttachments)
    {
        if (is_numeric($propertyUid)) {
            // Remove old Attachments
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhang');
            $result = $queryBuilder
                ->select('uid', 'datei')
                ->from('tx_sgestatecore_domain_model_anhang')
                ->where($queryBuilder->expr()->eq('immobilie', $queryBuilder->createNamedParameter(intval($propertyUid))))
                ->execute();
            $rows = $result->fetchAll(FetchMode::ASSOCIATIVE);

            foreach ($rows as $item) {
                @unlink($this->mediaPath . $item['datei']);
                $queryBuilder
                    ->delete('tx_sgestatecore_domain_model_anhang')
                    ->where($queryBuilder->expr()->eq('uid', $queryBuilder->createNamedParameter($item['uid'])))
                    ->execute();
            }

            if ($xmlAttachments) {
                foreach ($xmlAttachments as $attachment) {
                    /**
                     * @var $attachment \SimpleXMLElement
                     */
                    $source = strval($attachment->attributes()->location);
                    switch ($source) {
                        case 'EXTERN':
                            $title = strval($attachment->anhangtitel);
                            $format = strval($attachment->format);
                            $tempFile = strval($attachment->daten->pfad);
                            $type = strval($attachment->attributes()->gruppe);
                            break;
                        case 'INTERN':
                        case 'REMOTE':
                            // @todo: Implementieren
                            break;
                    }
                    $this->addAttachment($propertyUid, $title, $format, $tempFile, $type);
                    $title = $format = $tempFile = $type = null;
                }
            }
            return true;
        } else {
            return false;
        }
    }

    private function addAttachment($propertyUid, $title, $format, $tempFile, $type)
    {
        if ((strval($tempFile)!= '')&&(is_numeric($propertyUid))) {
            if (in_array(strtoupper($format), ['JPG', 'GIF', 'PNG', 'PDF', 'IMAGE/JPEG', 'IMAGE/GIF', 'IMAGE/PNG', 'APPLICATION/PDF'])) {
                // @todo: Standardtyp
                if ($type == '') {
                    $type = 'DOKUMENTE';
                }
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhangtyp');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_anhangtyp')
                    ->where($queryBuilder->expr()->like('kuerzel', $queryBuilder->createNamedParameter($type)))
                    ->execute();

                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                    $insertArray = [];
                    $insertArray['pid']         = $this->storagePid;
                    $insertArray['typ']         = $resultItemUid;
                    $insertArray['format']      = strtoupper($format);
                    $insertArray['titel']       = $title;
                    $insertArray['immobilie']   = $propertyUid;

                    $path_info = pathinfo($tempFile);
                    $new_filename = $path_info['filename'].'_'.uniqid().'.'.$path_info['extension'];
                    
                    if ((is_file($this->tempPath . $tempFile))&&(rename($this->tempPath . $tempFile, $this->mediaPath.$new_filename))) {
                        $insertArray['datei']       = $new_filename;
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                        return (int)$connection->lastInsertId('tx_sgestatecore_domain_model_anhang');
                    } elseif (is_file($this->mediaPath . $tempFile)) {
                        $insertArray['datei']       = $tempFile;
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_anhang');
                        $connection->insert(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );
                        return (int)$connection->lastInsertId('tx_sgestatecore_domain_model_anhang');
                    } else {
                        $this->addAlert('Fehler bei der Anhangzuordnung', 'Datei konnte nicht kopiert werden');
                    }
                } else {
                    $this->addAlert('Fehler bei der Anhangzuordnung', 'Es wurde ein ungültiger Typ übergeben: ' . $type);
                }
            } else {
                $this->addAlert('Fehler bei der Anhangzuordnung', 'Der Anhang ist in keinem zulässigen Format');
            }
        } else {
            $this->addAlert('Fehler bei der Anhangzuordnung', 'Es wurde keine Immobilie übergeben');
        }
        return null;
    }

    private function addCity($cityName, $useForSearch = null)
    {
        if ($cityName != '') {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ort');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_ort')
                ->where($queryBuilder->expr()->like('bezeichner', $queryBuilder->createNamedParameter($cityName)))
                ->execute();

            $data = [];
            $data['pid']        = $this->storagePid;
            $data['bezeichner'] = $cityName;
            if ($useForSearch === true) {
                $data['fuer_suche_verwenden'] = 1;
            }
            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_ort');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_ort',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_ort',
                    $data
                );
                $uid = (int)$connection->lastInsertId('tx_sgestatecore_domain_model_ort');
            }
            return $uid;
        } else {
            return null;
        }
    }

    private function addDistrict($districtName, $cityUid, $useForSearch = null)
    {
        if (($districtName != '')&&(is_numeric($cityUid))) {
            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_stadtteil');
            $result = $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_stadtteil')
                ->where($queryBuilder->expr()->like('bezeichner', $queryBuilder->createNamedParameter($districtName)))
                ->andWhere($queryBuilder->expr()->eq('ort', $queryBuilder->createNamedParameter($cityUid)))
                ->execute();

            $data = [];
            $data['pid']        = $this->storagePid;
            $data['ort']        = $cityUid;
            $data['bezeichner'] = $districtName;
            if ($useForSearch === true) {
                $data['fuer_suche_verwenden'] = 1;
            }

            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_stadtteil');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_stadtteil',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_stadtteil',
                    $data
                );
                $uid = (int)$connection->lastInsertId('tx_sgestatecore_domain_model_stadtteil');
            }
            return $uid;
        } else {
            return null;
        }
    }

    private function addContactperson(\SimpleXMLElement $item)
    {
        $kontaktpersonDaten = $item->kontaktperson;
        if (is_object($kontaktpersonDaten)) {
            $data['pid']                            = $this->storagePid;
            $data['name']                           = strval($kontaktpersonDaten->name);
            $data['vorname']                        = strval($kontaktpersonDaten->vorname);
            $data['titel']                          = strval($kontaktpersonDaten->titel);
            $data['anrede']                         = strval($kontaktpersonDaten->anrede);
            $data['position']                       = strval($kontaktpersonDaten->position);
            $data['anrede_brief']                   = strval($kontaktpersonDaten->anrede_brief);
            $data['firma']                          = strval($kontaktpersonDaten->firma);
            $data['zusatzfeld']                     = strval($kontaktpersonDaten->zusatzfeld);
            $data['strasse']                        = strval($kontaktpersonDaten->strasse);
            $data['hausnummer']                     = strval($kontaktpersonDaten->hausnummer);
            $data['plz']                            = strval($kontaktpersonDaten->plz);
            $data['postfach']                       = strval($kontaktpersonDaten->postfach);
            $data['postfach_plz']                   = strval($kontaktpersonDaten->postf_plz);
            $data['email_zentrale']                 = strval($kontaktpersonDaten->email_zentrale);
            $data['email_direkt']                   = strval($kontaktpersonDaten->email_direkt);
            $data['email_privat']                   = strval($kontaktpersonDaten->email_privat);
            $data['email_sonstige']                 = strval($kontaktpersonDaten->email_sonstige);
            $data['email_feedback']                 = strval($kontaktpersonDaten->email_feedback);
            $data['telefon_zentrale']               = strval($kontaktpersonDaten->tel_zentrale);
            $data['telefon_durchwahl']              = strval($kontaktpersonDaten->tel_durchw);
            $data['telefon_fax']                    = strval($kontaktpersonDaten->tel_fax);
            $data['telefon_handy']                  = strval($kontaktpersonDaten->tel_handy);
            $data['telefon_privat']                 = strval($kontaktpersonDaten->tel_privat);
            $data['telefon_sonstige']               = strval($kontaktpersonDaten->tel_sonstige);
            $data['url']                            = strval($kontaktpersonDaten->url);
            $data['adressfreigabe']                 = 0;
            $data['personennummer']                 = strval($kontaktpersonDaten->personennummer);
            $data['immobilientreuhaenderid']        = strval($kontaktpersonDaten->immobilientreuhaenderid);
            $data['freitextfeld']                   = strval($kontaktpersonDaten->freitextfeld);
            $data['ort']                            = $this->addCity(strval($kontaktpersonDaten->ort));
            if ((isset($kontaktpersonDaten) && !empty($kontaktpersonDaten->xpath('(land)[1]')))&&(strval($kontaktpersonDaten->land->attributes()->iso_land) != '')) {
                $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_land');
                $result = $queryBuilder
                    ->select('uid')
                    ->from('tx_sgestatecore_domain_model_land')
                    ->where($queryBuilder->expr()->eq('iso_kennzeichen', $queryBuilder->createNamedParameter(strval($item->geo->land->attributes()->iso_land))))
                    ->execute();
                if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                    $data['land'] = $resultItemUid;
                }
                $result = null;
            }

            $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_kontaktperson');
            $queryBuilder
                ->select('uid')
                ->from('tx_sgestatecore_domain_model_kontaktperson');

            // If a personnumber is provided this will be used for check if a contactperson is already present
            // if not firstname and lastname will be used
            if (strval($kontaktpersonDaten->personennummer) != '') {
                $queryBuilder->where($queryBuilder->expr()->like('personennummer', $queryBuilder->createNamedParameter(strval($kontaktpersonDaten->personennummer))));
            } else {
                if (strval($kontaktpersonDaten->vorname) != '') {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('vorname', $queryBuilder->createNamedParameter(strval($kontaktpersonDaten->vorname))));
                }
                if (strval($kontaktpersonDaten->name) != '') {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('name', $queryBuilder->createNamedParameter(strval($kontaktpersonDaten->name))));
                }
            }
            $result = $queryBuilder->execute();

            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_kontaktperson');
            if (($result->rowCount() > 0)&&(is_numeric($resultItemUid = $result->fetch()['uid']))) {
                $connection->update(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    $data,
                    ['uid' => $resultItemUid]
                );
                $uid = $resultItemUid;
            } else {
                $connection->insert(
                    'tx_sgestatecore_domain_model_kontaktperson',
                    $data
                );
                $uid = (int)$connection->lastInsertId('tx_sgestatecore_domain_model_kontaktperson');
            }
            return $uid;
        } else {
            return null;
        }
    }

    private function getBoolValue($value)
    {
        if (strval($value) != '') {
            if ((strtolower(strval($value)) == 'true')||(strtolower(strval($value)) == 'ja')||(strtolower(strval($value)) == 'j')) {
                return 1;
            } elseif ((strtolower(strval($value)) == 'false')||(strtolower(strval($value)) == 'n')) {
                return -1;
            } else {
                return 0;
            }
        } else {
            return 0;
        }
    }

    /**
     * @param $value
     * @return string
     * @throws InvalidTypeException
     */
    private function getDateValue($value)
    {
        try {
            $temp = new \DateTime($value);
            return $temp->format('U');
        } catch (Exception $e) {
            throw new InvalidTypeException();
        }
    }

    /**
     * Unzips filenameZip to pathExtract
     *
     * @param string $filenameZip
     * @param string $pathExtract
     * @return bool
     */
    private function unzip($filenameZip, $pathExtract)
    {
        $zip = new \ZipArchive;
        if ($zip->open($filenameZip) === true) {
            $zip->extractTo($pathExtract);
            $zip->close();
            return true;
        } else {
            return false;
        }
    }

    private function cleanUpTempDirectory()
    {
        $directoryContent = scandir($this->tempPath);
        foreach ($directoryContent as $entry) {
            if (!is_dir($entry)) {
                @unlink($this->tempPath . $entry);
            }
        }
    }

    /**
     * Adds an Entry to the internal Log Array
     *
     * @param $titel
     * @param $text
     */
    private function addAlert($title, $content)
    {
        $this->alerts[] = [
            'titel' => $title,
            'text' => $content
        ];
    }
}
