<?php
namespace Daho\SgEstateImportIs24\Tasks;

require_once(dirname(__FILE__).'/../Util/Immocaster/Sdk.php');

class Import extends \TYPO3\CMS\Scheduler\Task\AbstractTask {

    /**
     * @var $sgEstateCoreServices \Daho\SgEstateCore\Util\Services
     */
    private $sgEstateCoreServices;

  /**
   * @var string
   */
  protected $title;

    /**
     * An email address to be used during the process
     *
     * @var integer $storagePid
     */
    protected $storagePid;

  /**
   * @var string
   */
  protected $reportSenderEmail = '';


    /**
     * @var string $username
     */
    protected $username;

    /**
     * @var string
     */
    protected $tempPfad = '';

    /**
     * @var string
     */
    protected $archivPfad = '';

    protected $mediaPfad = '';

    /**
     * @var array $reportVersandAn
     */
    protected $reportVersandAn = array();

    /**
     * @var array $alertVersandAn
     */
    protected $alertVersandAn = array();

    /**
     * @var array
     */
    private $reportDaten = array();

    /**
     * @var array
     */
    private $fehlerBeiAusfuehrung = array();

    /**
     * Array zur Speicherung von Warnungen während der Ausführung um diese nachher gesammelt zu Versenden
     *
     * Aufbau
     * [] = array(
     *      'titel' => '',
     *      'text'  => '',
     * )
     *
     * @var array
     */
    private $warnungen = array();

    /**
     * @var \TYPO3\CMS\Extbase\Object\ObjectManager
     */
    private $objectManager;

    /**
     * @var array
     */
    private $statsImportLog = array();

    /**
     * @var \SimpleXMLElement
     */
    private $dataToImport;

    /**
     * @var \Immocaster_Sdk
     */
    private $oImmocaster;

    /**
     * @var string
     */
    private $apiKey;

    /**
     * @var array
     */
    private $typesForImport = array();

    /**
     * @var string
     */
    private $apiSecret;

    private $allTypesForImport = array(
        'offerlistelement:OfferApartmentRent'           => 'Wohnung Miete',
        'offerlistelement:OfferHouseRent'               => 'Haus Miete',
        'offerlistelement:OfferApartmentBuy'            => 'Wohnung Kauf',
        'offerlistelement:OfferHouseBuy'                => 'Haus Kauf',
        'offerlistelement:OfferShortTermAccommodation'  => 'Wohnen auf Zeit',
        'offerlistelement:OfferGarageBuy'               => 'Garage Kauf',
        'offerlistelement:OfferGarageRent'              => 'Garage Miete',
        'offerlistelement:OfferOffice'                  => 'Büro',
        'offerlistelement:OfferStore'                   => 'Geschäft',
        'offerlistelement:OfferGastronomy'              => 'Gastronomie',
        'offerlistelement:OfferIndustry'                => 'Industrie',
        'offerlistelement:OfferInvestment'              => 'Investment',
        'offerlistelement:OfferLivingBuySite'           => 'LivingBuySite',
        'offerlistelement:OfferLivingRentSite'          => 'LivingRentSite',
        'offerlistelement:OfferTradeSite'               => 'TradeSite',
        'offerlistelement:OfferSpecialPurpose'          => 'SpecialPurpose',
        'offerlistelement:OfferHouseType'               => 'OfferHouseType',
        'offerlistelement:OfferFlatShareRoom'           => 'FlatShareRoom',
        'offerlistelement:OfferAssistedLiving'          => 'AssistedLiving',
        'offerlistelement:OfferSeniorCare'              => 'SeniorCar',
        'offerlistelement:OfferCompulsoryAuction'       => 'Zwangsversteigerung',
    );

    private $imageWidth;

    private $imageHeight;

    private $uselive;

    private $publishChannel;

    public function execute(){
        libxml_use_internal_errors(true);

        $this->reportDaten['importStart'] = date('d.m.Y H:i:s');
        $this->initialize();

        $prepocessState = $this->preprocess();

        if ($prepocessState){
            $this->import();
        }

        $this->reportDaten['importEnde'] = date('d.m.Y H:i:s');

        $this->postprocess();

        if (count($this->fehlerBeiAusfuehrung) > 0){
            throw new \RuntimeException(implode(', ', $this->fehlerBeiAusfuehrung), 4711);
        }else{
            return true;
        }
    }

    private function initialize(){
        $this->fehlerBeiAusfuehrung = array();
        $this->objectManager = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Object\ObjectManager');
        $this->sgEstateCoreServices = $this->objectManager->get('Daho\SgEstateCore\Util\Services');

        $this->tempPfad = PATH_site . 'uploads/tx_sgestateimportis24/temp/';
        $this->archivPfad = PATH_site . 'uploads/tx_sgestateimportis24/archiv/';

        $this->mediaPfad = PATH_site . 'uploads/tx_sgestatecore/media/';

        $this->oImmocaster = \Immocaster_Sdk::getInstance('is24',$this->getApiKey(),$this->getApiSecret());

        $this->oImmocaster->setDataStorage(array(
            'mysql',
            $GLOBALS['TYPO3_CONF_VARS']['DB']['host'],
            $GLOBALS['TYPO3_CONF_VARS']['DB']['username'],
            $GLOBALS['TYPO3_CONF_VARS']['DB']['password'],
            $GLOBALS['TYPO3_CONF_VARS']['DB']['database']
        ),null,'tx_sgestateimportis24_immocaster');

        $this->oImmocaster->authenticateWithoutDB(false);

        if ($this->getUselive() == 1){
            $this->oImmocaster->setRequestUrl('live');
        }



    }

    private function preprocess(){
        $pagesize = 100;
        $actualPage = 1;

        do{

            $aParameter = array(
                'username'      => $this->getUsername(),
                'pagesize'      => $pagesize,
                'pagenumber'    => $actualPage,
                'publishchannel' => $this->getPublishChannel()
            );

            $temp = $this->oImmocaster->fullUserSearch($aParameter);

            $result = simplexml_load_string($temp);

            if (!$result){
                $this->addWarnung('XML-Object (Listing) could not be generated','Received String: "'.$temp.'"');
                array_push($this->fehlerBeiAusfuehrung,'XML-Object (Listing) could not be generated');
                return false;
            }

            $pagination = array(
                'pageNumber'    => intval($result->Paging->pageNumber),
                'pageSize'      => intval($result->Paging->pageSize),
                'numberOfPages' => intval($result->Paging->numberOfPages),
                'numberOfHits'  => intval($result->Paging->numberOfHits),
            );

            foreach ($result->realEstateList->children() as $item){
                //if (in_array(strval($item->attributes('xsi',true)->type), $this->getTypesForImport())){
                if ((strval($item->realEstateState) == 'ACTIVE')&&(in_array(strval($item->attributes('xsi',true)->type), $this->getTypesForImport()))){
                    $this->dataToImport[] = strval($item->attributes()->id);
                }
            }
            $actualPage++;
        }while($pagination['pageNumber']<$pagination['numberOfPages']);


        return true;
    }

    private function import(){
        /**
         * @var $item \SimpleXMLElement
         */
        $this->resetAllUpdatableImmobilien();
        if (count($this->dataToImport) > 0){
            foreach ($this->dataToImport as $exposeId){
                $exposeData = $this->oImmocaster->getExpose(array('exposeid'=>strval($exposeId)));
                $expose= simplexml_load_string($exposeData);
                if (!$expose){
                    $this->addWarnung('XML-Object (Expose) could not be generated','Received String: "'.$exposeData.'"');
                    array_push($this->fehlerBeiAusfuehrung,'XML-Object (Expose) could not be generated');
                    return false;
                }

                $importData = array();

                // Generelle Importdaten erzeugen
                $importData = array_merge(
                    $importData,
                    $this->processGeneralImportData($expose)
                );

                // Vom Typ abhängige Daten erzeugen
                $xsiType = strval($expose->realEstate->attributes('xsi',true)->type);
                $xsiType = str_replace('expose:','',$xsiType);
                switch ($xsiType){
                    case 'ApartmentRent':
                        $importData = array_merge(
                            $importData,
                            $this->processApartmentRentImportData($expose)
                        );
                        break;
                    case 'HouseRent':
                        $importData = array_merge(
                            $importData,
                            $this->processHouseRentImportData($expose)
                        );
                        break;
                    case 'GarageRent':
                        $importData = array_merge(
                            $importData,
                            $this->processGarageRentImportData($expose)
                        );
                        break;
                    case 'ApartmentBuy':
                        $importData = array_merge(
                            $importData,
                            $this->processApartmentBuyImportData($expose)
                        );
                        break;
                    case 'HouseBuy':
                        $importData = array_merge(
                            $importData,
                            $this->processHouseBuyImportData($expose)
                        );
                        break;
                    case 'ShortTermAccommodation':
                    case 'HouseType':
                    case 'Office':
                        $importData = array_merge(
                            $importData,
                            $this->processOfficeImportData($expose)
                        );
                        break;
                    case 'Store':
                        $importData = array_merge(
                            $importData,
                            $this->processStoreImportData($expose)
                        );
                        break;
                    case 'Gastronomy':
                    case 'Industry':
                    case 'SpecialPurpose':
                    case 'Investment':
                    case 'CompulsoryAuction':
                    case 'LivingBuySite':
                    case 'LivingRentSite':
                    case 'TradeSite':
                    case 'GarageBuy':
                    case 'FlatShareRoom':
                    case 'AssistedLiving':
                    case 'SeniorCare':
                        array_push($this->fehlerBeiAusfuehrung, 'This Estate-Type is not implemented');
                        $this->addWarnung('Type not implemented',$xsiType);
                        break;
                }

                if ($this->isValidEstate($importData)){
                    $uid = $this->saveDataToDB($importData);
                    $this->processAttachments($uid, $expose);
                }else{
                    array_push($this->fehlerBeiAusfuehrung, 'The generated data for Import is not valid.');
                    $this->addWarnung('Data invalid',print_r($importData, true));
                }
            }
        }
    }

    private function processGeneralImportData(\SimpleXMLElement $expose){
        $importData = array();

        // Name der Stadt ermitteln
        $cityParameter = array(
            'country-id'=>intval($expose->realEstate->address->geoHierarchy->country->geoCodeId),
            'region-id'=>intval($expose->realEstate->address->geoHierarchy->region->geoCodeId),
            'city-id'=>intval($expose->realEstate->address->geoHierarchy->city->geoCodeId),
            'list'=>false
        );
        $cityres = simplexml_load_string($this->oImmocaster->geoService($cityParameter));

        // Name des Stadtteilsermitteln
        $quarterParameter = array(
            'country-id'=>intval($expose->realEstate->address->geoHierarchy->country->geoCodeId),
            'region-id'=>intval($expose->realEstate->address->geoHierarchy->region->geoCodeId),
            'city-id'=>intval($expose->realEstate->address->geoHierarchy->city->geoCodeId),
            'quarter-id'=> intval($expose->realEstate->address->geoHierarchy->quarter->geoCodeId),
            'list'=>false
        );
        $quarterres = simplexml_load_string($this->oImmocaster->geoService($quarterParameter));


        $importData['pid']                              = $this->getStoragePid();
        $importData['objekttitel']                      = strval($expose->realEstate->title);
        $importData['objektnr_extern']                  = strval($expose->realEstate->externalId);
        $importData['openimmo_objid']                   = strval($expose->attributes()->id);
        $importData['objekt_strasse']                   = strval($expose->realEstate->address->street);
        $importData['objekt_hausnummer']                = strval($expose->realEstate->address->houseNumber);
        $importData['objekt_plz']                       = strval($expose->realEstate->address->postcode);
        $importData['objekt_ort']                       = $this->processCity($cityres->name); //$this->processCity(strval($expose->realEstate->address->city));
        $importData['objekt_stadtteil']                 = $this->processQuarter($quarterres->name, $importData['objekt_ort']); //$this->processQuarter(strval($expose->realEstate->address->quarter), $importData['objekt_ort']);
        $importData['objekt_breitengrad']               = strval($expose->realEstate->address->wgs84Coordinate->latitude);
        $importData['objekt_laengengrad']               = strval($expose->realEstate->address->wgs84Coordinate->longitude);
        $importData['waehrung']                         = $this->processCurrency('EUR');
        $importData['updatable']                        = true;
        $importData['import_number']                    = $this->taskUid;

        return $importData;

    }

    private function processApartmentRentImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('WOHNUNG');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->apartmentType),$importData['objektart']);
        $importData['kaltmiete']                = $this->getDecimalWert(strval($expose->realEstate->baseRent));
        $importData['wohnflaeche']              = $this->getDecimalWert(strval($expose->realEstate->livingSpace));
        $importData['anzahl_zimmer']            = $this->getDecimalWert(strval($expose->realEstate->numberOfRooms));
        $importData['objekt_etage']             = strval($expose->realEstate->floor);
        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);
        $importData['gesamtmiete_brutto']       = $this->getDecimalWert(strval($expose->realEstate->totalRent));
        $importData['nebenkosten']              = $this->getDecimalWert(strval($expose->realEstate->serviceCharge));
        $importData['heizkosten']               = $this->getDecimalWert(strval($expose->realEstate->heatingCosts));
        $importData['heizkosten_enthalten']     = $this->getBoolWert(strval($expose->realEstate->heatingCostsInServiceCharge));
        $importData['nutzflaeche']              = strval($expose->realEstate->usableFloorSpace);
        $importData['anzahl_schlafzimmer']      = strval($expose->realEstate->numberOfBedRooms);
        $importData['anzahl_badezimmer']        = strval($expose->realEstate->numberOfBathRooms);
        $importData['haustiere']                = $this->getBoolWert(strval($expose->realEstate->petsAllowed));
        $importData['gaestewc']                 = $this->getBoolWert(strval($expose->realEstate->guestToilet));
        $importData['aufzug_personen']          = $this->getBoolWert(strval($expose->realEstate->lift));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['wg_geeignet']              = $this->getBoolWert(strval($expose->realEstate->useAsFlatshareRoom));
        $importData['leicht_erreichbar']        = $this->getBoolWert(strval($expose->realEstate->handicappedAccessible));
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['wbs_erforderlich']         = $this->getBoolWert(strval($expose->realEstate->certificateOfEligibilityNeeded));
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['energiepass_baujahr']      = strval($expose->realEstate->constructionYear);
        $importData['anzahl_balkone']           = (strval($expose->realEstate->balcony) == 'true') ? 1 : 0 ;

        $importData['gartennutzung']            = $this->getBoolWert(strval($expose->realEstate->garden));
        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);
        $temp = strval($expose->realEstate->deposit);
        $temp = str_replace('.','',$temp);
        $temp = str_replace(',','.',$temp);
        $importData['kaution']                  = $temp;
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);

        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
            $importData['energiepass_endenergiebedarf'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
          $importData['energiepass_wertklasse'] = strval($expose->realEstate->energyCertificate->energyEfficiencyClass);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }

        if (($expose->realEstate->energySourcesEnev2014)&&(count($expose->realEstate->energySourcesEnev2014->children())>0)){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }

        $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('MIETE_PACHT'));
        $importData['mm']['tx_sgestatecore_immobilie_nutzungsart_mm'] = array($this->processNutzungsart('WOHNEN'));
        if (strval($expose->realEstate->builtInKitchen) == 'true'){
            $importData['mm']['tx_sgestatecore_immobilie_kuechenausstattung_mm'] = array($this->processKuechenausstattung('EBK'));
        }
        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingTypeEnev2014)));
        }

        return $importData;
    }

    private function processApartmentBuyImportData(\SimpleXMLElement $expose){
        $importData = array();

        $importData['objektart']                = $this->processObjektart('WOHNUNG');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->apartmentType),$importData['objektart']);
        $importData['wohnflaeche']              = $this->getDecimalWert(strval($expose->realEstate->livingSpace));
        $importData['anzahl_zimmer']            = $this->getDecimalWert(strval($expose->realEstate->numberOfRooms));
        $importData['objekt_etage']             = strval($expose->realEstate->floor);
        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);

        $importData['kaufpreis']                = $this->getDecimalWert(strval($expose->realEstate->price->value));
        $importData['hausgeld']                 = $this->getDecimalWert(strval($expose->realEstate->rentSubsidy));

        $importData['nutzflaeche']              = strval($expose->realEstate->usableFloorSpace);
        $importData['anzahl_schlafzimmer']      = strval($expose->realEstate->numberOfBedRooms);
        $importData['anzahl_badezimmer']        = strval($expose->realEstate->numberOfBathRooms);
        $importData['haustiere']                = $this->getBoolWert(strval($expose->realEstate->petsAllowed));
        $importData['gaestewc']                 = $this->getBoolWert(strval($expose->realEstate->guestToilet));
        $importData['aufzug_personen']          = $this->getBoolWert(strval($expose->realEstate->lift));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['wg_geeignet']              = $this->getBoolWert(strval($expose->realEstate->useAsFlatshareRoom));
        $importData['leicht_erreichbar']        = $this->getBoolWert(strval($expose->realEstate->handicappedAccessible));
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['energiepass_baujahr']      = strval($expose->realEstate->constructionYear);
        $importData['anzahl_balkone']           = (strval($expose->realEstate->balcony) == 'true') ? 1 : 0 ;
        $importData['gartennutzung']            = $this->getBoolWert(strval($expose->realEstate->garden));
        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);
        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
            $importData['energiepass_endenergiebedarf'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }

        if (($expose->realEstate->energySourcesEnev2014)&&(count($expose->realEstate->energySourcesEnev2014->children())>0)){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }


        $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('KAUF'));
        $importData['mm']['tx_sgestatecore_immobilie_nutzungsart_mm'] = array($this->processNutzungsart('WOHNEN'));



        if (strval($expose->realEstate->builtInKitchen) == 'true'){
            $importData['mm']['tx_sgestatecore_immobilie_kuechenausstattung_mm'] = array($this->processKuechenausstattung('EBK'));
        }
        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingType)));
        }


        return $importData;
    }

    private function processHouseRentImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('HAUS');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->buildingType),$importData['objektart']);
        $importData['kaltmiete']                = $this->getDecimalWert(strval($expose->realEstate->baseRent));
        $importData['wohnflaeche']              = $this->getDecimalWert(strval($expose->realEstate->livingSpace));
        $importData['anzahl_zimmer']            = $this->getDecimalWert(strval($expose->realEstate->numberOfRooms));
        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);
        $importData['gesamtmiete_brutto']       = $this->getDecimalWert(strval($expose->realEstate->totalRent));
        $importData['nebenkosten']              = $this->getDecimalWert(strval($expose->realEstate->serviceCharge));
        $importData['heizkosten']               = $this->getDecimalWert(strval($expose->realEstate->heatingCosts));
        $importData['heizkosten_enthalten']     = $this->getBoolWert(strval($expose->realEstate->heatingCostsInServiceCharge));
        $importData['nutzflaeche']              = strval($expose->realEstate->usableFloorSpace);
        $importData['anzahl_schlafzimmer']      = strval($expose->realEstate->numberOfBedRooms);
        $importData['anzahl_badezimmer']        = strval($expose->realEstate->numberOfBathRooms);
        $importData['haustiere']                = $this->getBoolWert(strval($expose->realEstate->petsAllowed));
        $importData['gaestewc']                 = $this->getBoolWert(strval($expose->realEstate->guestToilet));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['wg_geeignet']              = $this->getBoolWert(strval($expose->realEstate->useAsFlatshareRoom));
        $importData['leicht_erreichbar']        = $this->getBoolWert(strval($expose->realEstate->handicappedAccessible));
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['energiepass_baujahr']      = strval($expose->realEstate->constructionYear);


        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);
        $importData['kaution']                  = strval(str_replace(',','.',$expose->realEstate->deposit));
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);


        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }


        if (count($expose->realEstate->energySourcesEnev2014->children())>0){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }


        $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('MIETE_PACHT'));

        if (strval($expose->realEstate->builtInKitchen) == 'true'){
            $importData['mm']['tx_sgestatecore_immobilie_kuechenausstattung_mm'] = array($this->processKuechenausstattung('EBK'));
        }


        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingTypeEnev2014)));
        }


        return $importData;
    }

    private function processHouseBuyImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('HAUS');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->buildingType),$importData['objektart']);

        $importData['wohnflaeche']              = $this->getDecimalWert(strval($expose->realEstate->livingSpace));
        $importData['anzahl_zimmer']            = $this->getDecimalWert(strval($expose->realEstate->numberOfRooms));
        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);
        $importData['kaufpreis']                = $this->getDecimalWert(strval($expose->realEstate->price->value));


        $importData['nutzflaeche']              = strval($expose->realEstate->usableFloorSpace);
        $importData['anzahl_schlafzimmer']      = strval($expose->realEstate->numberOfBedRooms);
        $importData['anzahl_badezimmer']        = strval($expose->realEstate->numberOfBathRooms);
        $importData['haustiere']                = $this->getBoolWert(strval($expose->realEstate->petsAllowed));
        $importData['gaestewc']                 = $this->getBoolWert(strval($expose->realEstate->guestToilet));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['wg_geeignet']              = $this->getBoolWert(strval($expose->realEstate->useAsFlatshareRoom));
        $importData['leicht_erreichbar']        = $this->getBoolWert(strval($expose->realEstate->handicappedAccessible));
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['energiepass_baujahr']      = strval($expose->realEstate->constructionYear);

        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);

        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);


        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }


        if (count($expose->realEstate->energySourcesEnev2014->children())>0){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }


        $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('KAUF'));

        if (strval($expose->realEstate->builtInKitchen) == 'true'){
            $importData['mm']['tx_sgestatecore_immobilie_kuechenausstattung_mm'] = array($this->processKuechenausstattung('EBK'));
        }


        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingType)));
        }


        return $importData;
    }

    private function processGarageRentImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('PARKEN');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->garageType),$importData['objektart']);
        $importData['kaltmiete']                = $this->getDecimalWert(strval($expose->realEstate->price->value));
        $importData['nutzflaeche']              = strval($expose->realEstate->usableFloorSpace);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['energiepass_baujahr']      = strval($expose->realEstate->constructionYear);
        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);

        $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('MIETE_PACHT'));

        return $importData;
    }

    private function processOfficeImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('BUERO_PRAXEN');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->officeType),$importData['objektart']);

        $importData['kaltmiete']                = $this->getDecimalWert($expose->realEstate->price->value);
        $importData['nebenkosten']              = $this->getDecimalWert($expose->realEstate->additionalCosts->value);
        $importData['bueroflaeche']             = $this->getDecimalWert($expose->realEstate->netFloorSpace);

        $importData['denkmalgeschuetzt']        = $this->getBoolWert($expose->realEstate->listed);

        $importData['anzahl_stellplaetze']      = $this->getDecimalWert($expose->realEstate->numberOfParkingSpaces);
        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);

        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);
        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));
        $importData['teilbar_ab']               = $this->getDecimalWert(strval($expose->realEstate->minDivisible));

        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['aufzug_personen']          = $this->getBoolWert(strval($expose->realEstate->lift));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['leicht_erreichbar']        = $this->getBoolWert(strval($expose->realEstate->handicappedAccessible));

        $importData['kaution']                  = strval(str_replace(',','.',$expose->realEstate->deposit));
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));
        $importData['bueroflaeche']             = $this->getDecimalWert(strval($expose->realEstate->netFloorSpace));

        $importData['klimatisiert']             = $this->getBoolWert($expose->realEstate->airConditioning);
        $importData['teekueche']                = $this->getBoolWert($expose->realEstate->kitchenComplete);
        $importData['katine_cafeteria']         = $this->getBoolWert($expose->realEstate->hasCanteen);
        $importData['dv_verkabelung']           = $this->getBoolWert($expose->realEstate->lanCables);
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);

        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);

        if (strval($expose->realEstate->commercializationType) == 'RENT'){
            $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('MIETE_PACHT'));
        }else{
            array_push($this->fehlerBeiAusfuehrung, 'Offices only for Rent implemented');
            $this->addWarnung('Commercialtype not implemented',strval($expose->realEstate->commercializationType));
        }

        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
            $importData['energiepass_endenergiebedarf'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }

        if (($expose->realEstate->energySourcesEnev2014)&&(count($expose->realEstate->energySourcesEnev2014->children())>0)){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }
        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingTypeEnev2014)));
        }

        return $importData;
    }

    private function processStoreImportData(\SimpleXMLElement $expose){
        $importData = array();
        $importData['objektart']                = $this->processObjektart('EINZELHANDEL');
        $importData['objektarttyp']             = $this->processObjektartTyp(strval($expose->realEstate->storeType),$importData['objektart']);

        $importData['kaltmiete']                = $this->getDecimalWert($expose->realEstate->price->value);
        $importData['nebenkosten']              = $this->getDecimalWert($expose->realEstate->additionalCosts->value);

        $importData['gesamtflaeche']            = $this->getDecimalWert($expose->realEstate->totalFloorSpace);
        $importData['verkaufsflaeche']          = $this->getDecimalWert($expose->realEstate->netFloorSpace);
        $importData['sonstflaeche']             = $this->getDecimalWert($expose->realEstate->additionalArea);
        $importData['anzahl_stellplaetze']      = $this->getDecimalWert($expose->realEstate->numberOfParkingSpaces);
        $importData['ausstattungs_kategorie']   = $this->processAusstattungskategorie(strval($expose->realEstate->interiorQuality));

        $importData['objekt_anzahl_etagen']     = strval($expose->realEstate->numberOfFloors);

        $importData['objektbeschreibung']       = strval($expose->realEstate->descriptionNote);
        $importData['austattungsbeschreibung']  = strval($expose->realEstate->furnishingNote);
        $importData['lage']                     = strval($expose->realEstate->locationNote);
        $importData['sonstige_angaben']         = strval($expose->realEstate->otherNote);

        $importData['verfuegbar_ab']            = strval($expose->realEstate->freeFrom);
        $importData['baujahr']                  = strval($expose->realEstate->constructionYear);
        $importData['zustand']                  = $this->processZustand(strval($expose->realEstate->condition));
        $importData['aufzug_personen']          = $this->getBoolWert(strval($expose->realEstate->lift));
        $importData['aufzug_lasten']            = $this->getBoolWert(strval($expose->realEstate->goodsLift));
        $importData['unterkellert']             = $this->getBoolWert(strval($expose->realEstate->cellar));
        $importData['rampe']                    = $this->getBoolWert(strval($expose->realEstate->ramp));
        $importData['teilbar_ab']               = $this->getDecimalWert(strval($expose->realEstate->areaDivisibleFrom));
        $importData['letztemodernisierung']     = strval($expose->realEstate->lastRefurbishment);
        $importData['denkmalgeschuetzt']        = $this->getBoolWert($expose->realEstate->listed);

        $importData['kaution']                  = strval(str_replace(',','.',$expose->realEstate->deposit));
        $importData['provisionspflichtig']      = $this->getBoolWert($expose->realEstate->courtage->hasCourtage);
        $importData['provision_brutto']         = $this->getDecimalWert(strval($expose->realEstate->courtage->courtage));

        $importData['fensterfront']             = $this->getDecimalWert(strval($expose->realEstate->shopWindowLength));

        $importData['kontaktperson']            = $this->processKontaktperson($expose->contactDetails);

        if (strval($expose->realEstate->commercializationType) == 'RENT'){
            $importData['mm']['tx_sgestatecore_immobilie_vermarktungsart_mm'] = array($this->processVermarktungsart('MIETE_PACHT'));
        }else{
            array_push($this->fehlerBeiAusfuehrung, 'Offices only for Rent implemented');
            $this->addWarnung('Commercialtype not implemented',strval($expose->realEstate->commercializationType));
        }

        if (strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'AVAILABLE'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang(strval($expose->realEstate->energyCertificate->energyCertificateCreationDate));
            $importData['energiepass_art'] = $this->processEnergiepassArt(strval($expose->realEstate->buildingEnergyRatingType));
            $importData['energiepass_mitwarmwasser'] = $this->getBoolWert(strval($expose->realEstate->energyConsumptionContainsWarmWater));
            $importData['energiepass_energieverbrauchkennwert'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
            $importData['energiepass_endenergiebedarf'] = $this->getDecimalWert($expose->realEstate->thermalCharacteristic);
        }elseif(strval($expose->realEstate->energyCertificate->energyCertificateAvailability) == 'NOT_REQUIRED'){
            $importData['energiepass_jahrgang'] = $this->processEnergiepassJahrgang('NOT_REQUIRED');
        }
        if (($expose->realEstate->energySourcesEnev2014)&&(count($expose->realEstate->energySourcesEnev2014->children())>0)){
            foreach ($expose->realEstate->energySourcesEnev2014->children() as $item){
                $importData['mm']['tx_sgestatecore_immobilie_befeuerungsart_mm'][] = $this->processBefeuerungsart(strval($item));
            }
        }
        if (strval($expose->realEstate->heatingType) != ''){
            $importData['mm']['tx_sgestatecore_immobilie_heizungsart_mm'] = array($this->processHeizungsart(strval($expose->realEstate->heatingTypeEnev2014)));
        }


        return $importData;
    }

    /**
     * Checks if the Data is valid for inserting in DB
     *
     * @param array $data
     * @return bool
     */
    private function isValidEstate($data){
        $valid = true;
        if ($data['objekt_ort'] == null){
            $valid = false;
        }
        if ($data['objekt_stadtteil'] == null){
            $valid = false;
        }
        if ($data['waehrung'] == null){
            $valid = false;
        }
        if ($data['objektart'] == null){
            $valid = false;
        }

        return $valid;
    }

    private function processAttachments($uidEstate, $expose){
        if ($expose->realEstate->attachments){
            $attachments = $expose->realEstate->attachments->children();
            foreach($attachments as $attachment){
                switch(strval($attachment->attributes('xsi', true)->type)){
                    case 'common:Picture':
                        $insertArray = array();
                        $insertArray['pid'] = $this->getStoragePid();
                        $insertArray['immobilie'] = $uidEstate;

                        $title = strval($attachment->title);
                        $floorplan = strval($attachment->floorplan);
                        $titlePicture = strval($attachment->titlePicture);
                        $attachmentId = strval($attachment->attributes()->id);
                        if ($titlePicture == 'true'){
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "TITELBILD"'
                            );
                        }elseif($floorplan == 'true'){
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "GRUNDRISS"'
                            );
                        }else{
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "BILD"'
                            );
                        }
                        $insertArray['typ'] = $rows[0]['uid'];
                        $insertArray['format'] = 'JPG';
                        $insertArray['titel'] = $title;


                        foreach($attachment->urls->children() as $url){
                            if (strval($url->attributes()->scale) == 'SCALE'){
                                $pictureUrl = strval($url->attributes()->href);
                                $pictureUrl = str_replace('%WIDTH%',$this->getImageWidth(),$pictureUrl);
                                $pictureUrl = str_replace('%HEIGHT%',$this->getImageHeight(),$pictureUrl);
                                $filename = $uidEstate.'_'.$attachmentId.'.jpg';
                                $pathAndFilename = $this->mediaPfad.$filename;
                                $fileContent = @file_get_contents($pictureUrl);
                                if ($fileContent != false){
                                    file_put_contents($pathAndFilename, $fileContent);
                                    $insertArray['datei'] = $filename;
                                    $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                                        'tx_sgestatecore_domain_model_anhang',
                                        $insertArray
                                    );
                                }
                            }
                        }
                        break;
                    case 'common:PDFDocument':
                        $insertArray = array();
                        $insertArray['pid'] = $this->getStoragePid();
                        $insertArray['immobilie'] = $uidEstate;

                        $title = strval($attachment->title);
                        $floorplan = strval($attachment->floorplan);
                        $titlePicture = strval($attachment->titlePicture);

                        if ($titlePicture == 'true'){
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "TITELBILD"'
                            );
                        }elseif($floorplan == 'true'){
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "GRUNDRISS"'
                            );
                        }else{
                            $rows = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                                'uid',
                                'tx_sgestatecore_domain_model_anhangtyp',
                                'sys_language_uid = 0 AND kuerzel LIKE "DOKUMENTE"'
                            );
                        }
                        $insertArray['typ'] = $rows[0]['uid'];
                        $insertArray['format'] = 'PDF';
                        $insertArray['titel'] = $title;

                        $attachmentId = strval($attachment->attributes()->id);
                        $filename = $uidEstate.'_'.$attachmentId.'.pdf';
                        $pathAndFilename = $this->mediaPfad.$filename;
                        file_put_contents($pathAndFilename, file_get_contents(strval($attachment->url)));
                        $insertArray['datei'] = $filename;
                        $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                            'tx_sgestatecore_domain_model_anhang',
                            $insertArray
                        );

                        break;
                    default:
                }
            }
        }
    }

    private function processNutzungsart($value){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_nutzungsart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$value.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processVermarktungsart($value){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_vermarktungsart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$value.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processKuechenausstattung($value){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_kuechenausstattung',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$value.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processObjektart($value){

        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_objektart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$value.'"'
        );

        //echo "<pre>";print_r($result);die();
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processObjektartTyp($value, $objektArtUid){
        $token = $this->translateToken('kindOfObjectType', $value);

        $result = $GLOBALS['TYPO3_DB']->exec_SELECT_mm_query(
            'tx_sgestatecore_domain_model_objektarttyp.uid', //$select,
            'tx_sgestatecore_domain_model_objektart', //$local_table,
            'tx_sgestatecore_objektart_objektarttyp_mm', // $mm_table,
            'tx_sgestatecore_domain_model_objektarttyp', //$foreign_table,
            ' AND tx_sgestatecore_domain_model_objektarttyp.deleted = 0 AND tx_sgestatecore_domain_model_objektarttyp.sys_language_uid = 0 AND tx_sgestatecore_domain_model_objektarttyp.kuerzel LIKE "'.$token.'" AND tx_sgestatecore_domain_model_objektart.uid = '.$objektArtUid, //'  ', //$whereClause = '',
            '',
            '',
            1
        );
        $row = $GLOBALS['TYPO3_DB']->sql_fetch_assoc($result);

        if (count($result) == 1){
            return $row['uid'];
        }else{
            return null;
        }
    }

    private function processAusstattungskategorie($value){
        $token = $this->translateToken('equipment', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_austattungskategorie',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processCurrency($value)
    {
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_waehrung',
            'deleted = 0 AND sys_language_uid = 0 AND iso_kennzeichen LIKE "'.$value.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processHeizungsart($value){
        $token = $this->translateToken('heatingType', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_heizungsart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processEnergiepassJahrgang($value){
        $token = $this->translateToken('energiepassJahrgang', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_energiepassjahrgang',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processEnergiepassArt($value){
        $token = $this->translateToken('energiepassArt', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_energiepassart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processBefeuerungsart($value){
        $token = $this->translateToken('beacoaningType', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_befeuerungsart',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processZustand($value){
        $token = $this->translateToken('condition', $value);
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_zustand',
            'deleted = 0 AND sys_language_uid = 0 AND kuerzel LIKE "'.$token.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processLand($value){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_land',
            'deleted = 0 AND sys_language_uid = 0 AND iso_kennzeichen LIKE "'.$value.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            return null;
        }
    }

    private function processKontaktperson($contactDetails){
        $importData = array();

        $importData['pid']                  = $this->getStoragePid();
        $importData['name']                 = strval($contactDetails->lastname);
        $importData['vorname']              = strval($contactDetails->firstname);
        $importData['titel']                = strval($contactDetails->title);
        $importData['anrede']               = strval($contactDetails->salutation);
        $importData['position']             = strval($contactDetails->position);
        $importData['firma']                = strval($contactDetails->company);
        $importData['strasse']              = strval($contactDetails->address->street);
        $importData['hausnummer']           = strval($contactDetails->address->houseNumber);
        $importData['plz']                  = strval($contactDetails->address->postcode);
        $importData['ort']                  = strval($contactDetails->address->city);
        $importData['email_direkt']         = strval($contactDetails->email);

        if (strval($contactDetails->faxNumberSubscriber) != ''){
            $faxNumberPrefix = strval($contactDetails->faxNumberCountryCode) == '+49' ? '0' : strval($contactDetails->faxNumberCountryCode). ' ';
            $importData['telefon_fax']          = $faxNumberPrefix.strval($contactDetails->faxNumberAreaCode).' '.strval($contactDetails->faxNumberSubscriber);
        }

        if (strval($contactDetails->phoneNumberSubscriber) != ''){
            $phoneNumberPrefix = strval($contactDetails->phoneNumberCountryCode) == '+49' ? '0' : strval($contactDetails->phoneNumberCountryCode). ' ';
            $importData['telefon_durchwahl']    = $phoneNumberPrefix.strval($contactDetails->phoneNumberAreaCode).' '.strval($contactDetails->phoneNumberSubscriber);
        }

        if (strval($contactDetails->cellPhoneNumberSubscriber) != ''){
            $cellPhoneNumberPrefix = strval($contactDetails->cellPhoneNumberCountryCode) == '+49' ? '0' : strval($contactDetails->cellPhoneNumberCountryCode). ' ';
            $importData['telefon_handy']        = $cellPhoneNumberPrefix.strval($contactDetails->cellPhoneNumberAreaCode).' '.strval($contactDetails->cellPhoneNumberSubscriber);
        }

        $importData['land']                 = $this->processLand(strval($contactDetails->countryCode));
        $importData['url']                  = strval($contactDetails->homepageUrl);


        $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
            'tx_sgestatecore_domain_model_kontaktperson',
            $importData
        );
        return $GLOBALS['TYPO3_DB']->sql_insert_id();
    }

    /**
     * Returns the uid of the city
     * Checks if City exists in Database, if not City will be inserted
     *
     * @param string $name
     * @return integer
     */
    private function processCity($name){
        $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
            'uid',
            'tx_sgestatecore_domain_model_ort',
            'deleted = 0 AND sys_language_uid = 0 AND bezeichner LIKE "'.$name.'"'
        );
        if (count($result) == 1){
            return $result[0]['uid'];
        }else{
            $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                'tx_sgestatecore_domain_model_ort',
                array(
                    'pid' => $this->getStoragePid(),
                    'bezeichner' => $name,
                    'fuer_suche_verwenden' => 1
                )
            );
            $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();
            return $uid;
        }
    }

    /**
     * Returns the uid of the Quarter
     * Checks if Quarter exists in Database, if not Quarter will be inserted
     *
     * @param string $name
     * @param integer $cityUid
     * @return mixed
     */
    private function processQuarter($name, $cityUid){
        if ($cityUid != null){
            $result = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows(
                'uid',
                'tx_sgestatecore_domain_model_stadtteil',
                'deleted=0 AND sys_language_uid = 0 AND bezeichner LIKE "'.$name.'" AND ort = '.$cityUid
            );
            if (count($result) == 1){
                return $result[0]['uid'];
            }else{
                $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                    'tx_sgestatecore_domain_model_stadtteil',
                    array(
                        'pid' => $this->getStoragePid(),
                        'ort' => $cityUid,
                        'bezeichner' => $name,
                        'fuer_suche_verwenden' => 1
                    )
                );
                $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();
                return $uid;
            }
        }else{
            return null;
        }
    }

    private function saveDataToDB($importData){
        $mmData = $importData['mm'];
        unset($importData['mm']);

        $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
            'tx_sgestatecore_domain_model_immobilie',
            $importData
        );
        $uid = $GLOBALS['TYPO3_DB']->sql_insert_id();

        foreach($mmData as $tableName => $data){
            foreach ($data as $value){
                $result = $GLOBALS['TYPO3_DB']->exec_INSERTquery(
                    $tableName,
                    array(
                        'uid_local' => $uid,
                        'uid_foreign' => $value
                    )
                );
            }
        }
        return $uid;

    }

    private function getBoolWert($value){
        if (strval($value) != ''){
            if ((strtolower(strval($value)) == 'true')||(strtolower(strval($value)) == 'ja')||(strtolower(strval($value)) == 'yes')){
                return 1;
            }elseif ((strtolower(strval($value)) == 'false')||(strtolower(strval($value)) == 'no')){
                return -1;
            }else{
                return 0;
            }
        }else{
            return 0;
        }
    }

    private function getDecimalWert($value){
        if (strval($value) != ''){
            return str_replace(',','.',$value);
        }else{
            return 0;
        }
    }

    private function getDateWert($value){
        $temp = new \DateTime($value);
        return $temp->format('U');
    }

    private function resetAllUpdatableImmobilien(){
        $this->sgEstateCoreServices->deleteAllUpdatableRealties($this->taskUid);
    }

    private function postprocess(){
        // Report
        $this->reportDaten['aktiveImmobilien'] = $GLOBALS['TYPO3_DB']->exec_SELECTgetRows('objektnr_extern','tx_sgestatecore_domain_model_immobilie','deleted=0 and hidden=0');

        if (count($this->reportDaten['aktiveImmobilien']) > 0){
            $this->sendTemplateEmail($this->reportVersandAn, array(), $this->reportSenderEmail, 'Immobilienimport Report', 'Report', $this->reportDaten, array());
        }

        // Verabeitung von erzeugten Warnungen
        $this->warnungen = array_filter($this->warnungen);
        if (count($this->warnungen) > 0){
            $this->sendTemplateEmail($this->alertVersandAn, array(), $this->reportSenderEmail,'Der ImmobilienImport hat WARNUNGEN gemeldet!','Warnung',array('warnungen' => $this->warnungen),array());
        }
    }



    /**
     * @param $object
     * @param $token
     * @return null
     */
    private function translateToken($object, $token){
        $translation = array(
            'kindOfObjectType' => array(
                'ROOF_STOREY' => 'DACHGESCHOSS',
                'MAISONETTE' => 'MAISONETTE',
                'LOFT' => 'LOFT',
                'PENTHOUSE' => 'PENTHOUSE',
                'TERRACED_FLAT' => 'TERRASSEN',
                'APARTMENT' => 'ETAGE',
                'GROUND_FLOOR' => 'ERDGESCHOSS',
                'RAISED_GROUND_FLOOR' => 'HOCHPATERRE',
                'HALF_BASEMENT' => 'SOUTERRAIN',
                'OTHER' => 'SONSTIGE',
                'NO_INFORMATION' => 'KEINE_ANGABE',
                'TERRACE_HOUSE' => 'REIHENHAUS',
                'MID_TERRACE_HOUSE' => 'REIHENMITTEL',
                'END_TERRACE_HOUSE' => 'REIHENECK',
                'SEMIDETACHED_HOUSE' => 'DOPPELHAUSHAELFTE',
                'SINGLE_FAMILY_HOUSE' => 'EINFAMILIENHAUS',
                'BUNGALOW' => 'BUNGALOW',
                'VILLA' => 'VILLA',
                'FARMHOUSE' => 'BAUERNHAUS',
                'MULTI_FAMILY_HOUSE' => 'MEHRFAMILIENHAUS',
                'CASTLE_MANOR_HOUSE' => 'BURG',
                'SPECIAL_REAL_ESTATE' => 'SPECIAL_REAL_ESTATE',
                'OFFICE' => 'BUEROFLAECHE',
                'OFFICE_FLOOR' => 'OFFICE_FLOOR',
                'OFFICE_BUILDING' => 'BUEROHAUS',
                'OFFICE_CENTRE' => 'BUEROZENTRUM',
                'OFFICE_STORAGE_BUILDING' => 'OFFICE_STORAGE_BUILDING',
                'STUDIO' => 'STUDIO',
                'SURGERY' => 'PRAXIS',
                'SURGERY_FLOOR' => 'SURGERY_FLOOR',
                'SURGERY_BUILDING' => 'PRAXISHAUS',
                'COMMERCIAL_CENTRE' => 'COMMERCIAL_CENTRE',
                'LIVING_AND_COMMERCIAL_BUILDING' => 'LIVING_AND_COMMERCIAL_BUILDING',
                'OFFICE_AND_COMMERCIAL_BUILDING' => 'OFFICE_AND_COMMERCIAL_BUILDING',
                'STORE' => 'LADENLOKAL',
                'SELF_SERVICE_MARKET' => 'VERBRAUCHERMARKT',
                'SHOPPING_CENTRE' => 'EINKAUFSZENTRUM',
                'DEPARTMENT_STORE' => 'KAUFHAUS',
                'FACTORY_OUTLET' => 'FACTORY_OUTLET',
                'KIOSK' => 'KIOSK',
                'SALES_AREA' => 'VERKAUFSFLAECHE',
                'SALES_HALL' => 'VERKAUFSHALLE',
                'SHOWROOM_SPACE' => 'AUSSTELLUNGSFLAECHE',
                'STREET_PARKING' => 'STELLPLATZ',
                'CARPORT' => 'CARPORT',
                'DUPLEX' => 'DUPLEX',
                'UNDERGROUND_GARAGE' => 'TIEFGARAGE',
                'CAR_PARK' => 'PARKHAUS',
                'GARAGE' => 'GARAGE'
            ),
            'condition' => array(
                'FIRST_TIME_USE' => 'ERSTBEZUG',
                'FIRST_TIME_USE_AFTER_REFURBISHMENT' => 'ERSTBEZUG_NACH_SANIERUNG',
                'MINT_CONDITION' => 'NEUWERTIG',
                'REFURBISHED' => 'TEIL_VOLLSANIERT',
                'FULLY_RENOVATED' => 'VOLL_SANIERT',
                'NEED_OF_RENOVATION' => 'SANIERUNGSBEDUERFTIG',
                'NEGOTIABLE' => 'NACH_VEREINBARUNG',
                'MODERNIZED' => 'MODERNISIERT',
                'WELL_KEPT' => 'GEPFLEGT',
                'RIPE_FOR_DEMOLITION' => 'ABRISSOBJEKT',
                'NO_INFORMATION' => 'NO_INFORMATION'
            ),
            'location' => array(
                'CLASSIFICATION_A' => '1A',
                'CLASSIFICATION_B' => '1B',
                'SHOPPING_CENTRE' => 'EINKAUFSZENTRUM',
                'NO_INFORMATION' => 'KEINE_ANGABE'
            ),
            'heatingType' => array(
                'STOVE_HEATING' => 'OFEN',
                'SELF_CONTAINED_CENTRAL_HEATING' => 'ETAGEN',
                'CENTRAL_HEATING' => 'ZENTRAL',
                'DISTRICT_HEATING' => 'FERNWAERME',
                'FLOOR_HEATING' => 'FUSSBODEN',
                'GAS_HEATING' => 'GAS-HEIZUNG',
                'WOOD_PELLET_HEATING' => 'HOLZ-PELLETHEIZUNG',
                'NIGHT_STORAGE_HEATER' => 'NACHTSPEICHEROFEN',
                'OIL_HEATING' => 'OELHEIZUNG',
                'SOLAR_HEATING' => 'SOLAR-HEIZUNG',
                'HEAT_PUMP' => 'WAERMEPUMPE',
                'COMBINED_HEAT_AND_POWER_PLANT' => 'BLOCKHEIZKRAFTWERK',
                'ELECTRIC_HEATING' => 'ELEKTRO-HEIZUNG',
                'NO_INFORMATION' => 'NO_INFORMATION'
            ),
            'energiepassJahrgang' => array(
                'BEFORE_01_MAY_2014' => '2008',
                'FROM_01_MAY_2014' => '2014',
                'NOT_APPLICABLE' => 'OHNE',
                'NOT_REQUIRED' => 'NICHT_NOETIG'
            ),
            'energiepassArt' => array(
                'ENERGY_REQUIRED' => 'BEDARF',
                'ENERGY_CONSUMPTION' => 'VERBRAUCH'
            ),
            'beacoaningType' => array(
                'OIL' => 'OEL',
                'GAS' => 'GAS',
                'ELECTRICITY' => 'ELEKTRO',
                'SOLAR_HEATING' => 'SOLAR',
                'GEOTHERMAL' => 'ERDWAERME',
                'DISTRICT_HEATING' => 'FERNWAERME',
                'PELLET_HEATING' => 'PELLET',
                'COAL' => 'KOHLE',
                'ACID_GAS' => 'ERDGAS_LEICHT',
                'SOUR_GAS' => 'ERDGAS_SCHWER',
                'LIQUID_GAS' => 'FLUESSIGGAS',
                'STEAM_DISTRICT_HEATING' => 'FERNWAERME_DAMPF',
                'WOOD' => 'HOLZ',
                'WOOD_CHIPS' => 'HOLZ-HACKSCHNITZEL',
                'COAL_COKE' => 'KOHLE/KOKS',
                'LOCAL_HEATING' => 'NAHWAERME',
                'HEAT_SUPPLY' => 'WAERMELIEFERUNG',
                'BIO_ENERGY' => 'BIO_ENERGIE',
                'WIND_ENERGY' => 'WINDENERGIE',
                'HYDRO_ENERGY' => 'WASSERENERGIE',
                'ENVIRONMENTAL_THERMAL_ENERGY' => 'UMWELTWAERME',
                'COMBINED_HEAT_AND_POWER_FOSSIL_FUELS' => 'KWK_FOSSIL',
                'COMBINED_HEAT_AND_POWER_RENEWABLE_ENERGY' => 'KWK_ERNEUERBAR',
                'COMBINED_HEAT_AND_POWER_REGENERATIVE_ENERGY' => 'KWK_REGENERATIV',
                'COMBINED_HEAT_AND_POWER_BIO_ENERGY' => 'KWK_BIO',
                'NO_INFORMATION' => 'NO_INFORMATION'
            ),
            'equipment' => array(
                'NO_INFORMATION' => 'KEINE_ANGABE',
                'SIMPLE' => 'EINFACH',
                'NORMAL' => 'STANDARD',
                'SOPHISTICATED' => 'GEHOBEN',
                'LUXURY' => 'LUXUS'
            ),
            'marketingMethod' => array(
                '0' => 'MIETE_PACHT'
            ),
            'currency' => array(
                'EUR' => 'EUR'
            ),
            'mediaType' => array(
                'B' => 'BILD',
                'GB' => 'GRUNDRISS'
            ),
        );


        if (isset($translation[$object][$token])){
            return $translation[$object][$token];
        }else{
            return null;
        }
    }

  /**
   * @return string
   */
  public function getTitle()
  {
    return $this->title;
  }

  /**
   * @param string $title
   * @return $this
   */
  public function setTitle($title)
  {
    $this->title = $title;
    return $this;
  }



    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @param string $username
     */
    public function setUsername($username)
    {
        $this->username = $username;
    }

    /**
     * @return string
     */
    public function getApiKey()
    {
        return $this->apiKey;
    }

    /**
     * @param string $apiKey
     */
    public function setApiKey($apiKey)
    {
        $this->apiKey = $apiKey;
    }

    /**
     * @return string
     */
    public function getApiSecret()
    {
        return $this->apiSecret;
    }

    /**
     * @param string $apiSecret
     */
    public function setApiSecret($apiSecret)
    {
        $this->apiSecret = $apiSecret;
    }

    /**
     * @param int $storagePid
     */
    public function setStoragePid($storagePid)
    {
        $this->storagePid = $storagePid;
    }

    /**
     * @return int
     */
    public function getStoragePid()
    {
        return $this->storagePid;
    }

  /**
   * @return string
   */
  public function getReportSenderEmail()
  {
    return $this->reportSenderEmail;
  }

  /**
   * @param string $reportSenderEmail
   * @return $this
   */
  public function setReportSenderEmail($reportSenderEmail)
  {
    $this->reportSenderEmail = $reportSenderEmail;
    return $this;
  }

    /**
     * @param array $alertVersandAn
     */
    public function setAlertVersandAn($alertVersandAn)
    {
        $this->alertVersandAn = $alertVersandAn;
    }

    /**
     * @return array
     */
    public function getAlertVersandAn()
    {
        return $this->alertVersandAn;
    }

    /**
     * @param array $reportVersandAn
     */
    public function setReportVersandAn($reportVersandAn)
    {
        $this->reportVersandAn = $reportVersandAn;
    }

    /**
     * @return array
     */
    public function getReportVersandAn()
    {
        return $this->reportVersandAn;
    }

    /**
     * @return array
     */
    public function getTypesForImport()
    {
        return $this->typesForImport;
    }

    /**
     * @param array $typesForImport
     */
    public function setTypesForImport($typesForImport)
    {
        $this->typesForImport = $typesForImport;
    }

    /**
     * @return array
     */
    public function getAllTypesForImport()
    {
        return $this->allTypesForImport;
    }

    /**
     * @return mixed
     */
    public function getImageWidth()
    {
        return $this->imageWidth;
    }

    /**
     * @param mixed $imageWidth
     */
    public function setImageWidth($imageWidth)
    {
        $this->imageWidth = $imageWidth;
    }

    /**
     * @return mixed
     */
    public function getImageHeight()
    {
        return $this->imageHeight;
    }

    /**
     * @param mixed $imageHeight
     */
    public function setImageHeight($imageHeight)
    {
        $this->imageHeight = $imageHeight;
    }

    /**
     * @return mixed
     */
    public function getUselive()
    {
        return $this->uselive;
    }

    /**
     * @param mixed $uselive
     */
    public function setUselive($uselive)
    {
        $this->uselive = $uselive;
    }

    /**
     * @return mixed
     */
    public function getPublishChannel()
    {
        if (is_array($this->publishChannel)){
            return $this->publishChannel[0];
        }else{
            return $this->publishChannel;
        }

    }

    /**
     * @param mixed $publishChannel
     */
    public function setPublishChannel($publishChannel)
    {
        $this->publishChannel = $publishChannel;
    }



    private function addWarnung($titel, $text){
        $this->warnungen[] = array(
            'titel' => $titel,
            'text' => $text
        );
    }

    /**
     * This method returns the destination mail address as additional information
     *
     * @return string Information to display
     */
    public function getAdditionalInformation() {
        $return = 'Title:'.$this->getTitle();
        return $return;
    }

    protected function sendTemplateEmail(array $recipients = array(), array $bcc = array(), $sender, $subject, $templateName, array $variables = array(), array $attachments = array()) {
        /**
         * @var $emailView \TYPO3\CMS\Fluid\View\StandaloneView
         * @var $message \TYPO3\CMS\Core\Mail\MailMessage
         */

        $templateRootPath = \TYPO3\CMS\Core\Utility\ExtensionManagementUtility::extPath('sg_estate_import_is24') . 'Resources/Private/Templates/';
        $templatePathAndFilename = $templateRootPath . 'Email/' . $templateName . '.html';

        $emailView = $this->objectManager->get('TYPO3\CMS\Fluid\View\StandaloneView');
        $emailView->setTemplatePathAndFilename($templatePathAndFilename);
        $emailView->setFormat('html');
        $emailView->assignMultiple($variables);

        $emailBody = $emailView->render();

        $message = \TYPO3\CMS\Core\Utility\GeneralUtility::makeInstance('TYPO3\CMS\Core\Mail\MailMessage');

        if (\TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($sender)){
            $message->setFrom($sender);
        }else{
            $this->fehlerBeiAusfuehrung[] = 'Keine gültige Absenderadresse hinterlegt.';
            return false;
        }

        // Add Recipients
        $recipients = array_filter($recipients);
        if (count($recipients) > 0){
            foreach ($recipients as $email){
                if (\TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($email)){
                    $message->addTo($email);
                }
            }
        }else{
            $this->fehlerBeiAusfuehrung[] = 'Keine E-Mail Empfänger übergeben. Betreff:'. $subject .', Template:'. $templateName;
            return false;
        }

        if (count($bcc) > 0){
            foreach ($bcc as $email){
                if (\TYPO3\CMS\Core\Utility\GeneralUtility::validEmail($email)){
                    $message->addBcc($email);
                }
            }
        }

        if ($subject != ''){
            $message->setSubject($subject);
        }else{
            $this->fehlerBeiAusfuehrung[] = 'Kein Betreff hinterlegt.';
            return false;
        }

        $message->setBody($emailBody, 'text/html');

        // @todo: Dateianhang auch ohne Abspeichern im Dateisystem ermöglichen
        if (count($attachments) > 0){
            foreach ($attachments as $attachment) {
                $message->attach(\Swift_Attachment::fromPath($attachment));
            }
        }

        $message->send();
        return $message->isSent();
    }

    private function zip($filenameZip, $pathToZip){
        $zip = new \ZipArchive;
        if ($zip->open($filenameZip, \ZipArchive::CREATE) === TRUE) {

            $handle=opendir($pathToZip);
            while ($datei = readdir ($handle)) {
                if (is_file($pathToZip.$datei)){
                    $zip->addFile($pathToZip.$datei, $datei);
                }
            }
            closedir($handle);
            $zip->close();
        }
    }
}