<?php
/**
 * User: Jessica Schlierenkamp<mail@schlierenkamp.de>
 * Date: 04/2019
 */
namespace SG\SgEstateImmosolve\Service;

use SG\SgEstateImmosolve\Api\ImmosolveContactApi;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Object\ObjectManager;
use TYPO3\CMS\Extbase\Service\TypoScriptService;

class ImmosolveContactApiService
{
    const API_REQUIRED_FIELDS = [
        'realty' => [
            'salutation' => ['required' => true],
            'firstName' => ['required' => true],
            'lastName' => ['required' => true],
            'email' => ['required' => true]
        ],
        'petition' => [
            'salutation' => ['required' => true],
            'firstName' => ['required' => true],
            'lastName' => ['required' => true],
            'email' => ['required' => true],
            'minArea' => [
                'required' => true,
                'cast' => 'double'
            ],
            'minRooms' => [
                'required' => true,
                'cast' => 'int'
            ]
        ]
    ];

    /**
     * @var ImmosolveContactApi
     */
    private $immosolveContactApi;

    /**
     * @var array $settings
     */
    private $settings;

    /**
     * @var array $connectionSettings
     */
    private $connectionSettings;

    /**
     * @param array $connectionSettings
     */
    public function __construct()
    {
        $objectManager = GeneralUtility::makeInstance(ObjectManager::class);
        $configurationManager = $objectManager->get(ConfigurationManager::class);
        $typoscriptService = $objectManager->get(TypoScriptService::class);
        $extbaseFrameworkConfiguration = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT);

        $this->settings = $typoscriptService->convertTypoScriptArrayToPlainArray($extbaseFrameworkConfiguration['plugin.']['tx_sgestateimmosolve_pi1.']['settings.']);
        $this->connectionSettings = $this->settings['immosolve']['api'];
    }

    /**
     * @param array $connectionSettings
     */
    private function initializeImmosolveContactApi(array $connectionSettings)
    {
        $this->immosolveContactApi = GeneralUtility::makeInstance(ObjectManager::class)->get(ImmosolveContactApi::class);

        $this->immosolveContactApi->init();
        $this->immosolveContactApi->setConsumerKey($connectionSettings['consumerKey']);
        $this->immosolveContactApi->setConsumerSecret($connectionSettings['consumerSecret']);
        $this->immosolveContactApi->setToken($connectionSettings['token']);
        $this->immosolveContactApi->setTokenSecret($connectionSettings['tokenSecret']);
        $this->immosolveContactApi->setServer($connectionSettings['server']);
    }

    /**
     * @param array $formValues
     * @param string $type
     *
     * @return bool
     */
    public function queueInquiry(array $formValues, $type)
    {
        $queueService = GeneralUtility::makeInstance(ObjectManager::class)->get(QueueService::class);

        if ($type === 'object') {
            $mappedValues = $this->buildImmosolveContactApiArrayForRealtyRequest($formValues);
            $queueService->addItemToQueue($mappedValues, 'object', $this->settings['immosolve']['queuePid']);
        } else {
            $mappedValues = $this->buildImmosolveContactApiArrayForPetitionRequest($formValues);
            $queueService->addItemToQueue($mappedValues, 'petition', $this->settings['immosolve']['queuePid']);
        }

        return true;
    }

    /**
     * @param string $payload
     *
     * @param $type
     * @return object
     */
    public function doInquiry($payload, $type)
    {
        $this->initializeImmosolveContactApi($this->connectionSettings);

        return $this->immosolveContactApi->doInquiry($payload, $type);
    }

    /**
     * @param array $formValues
     *
     * @return array
     */
    private function buildImmosolveContactApiArrayForRealtyRequest(array $formValues)
    {
        $comments = [];
        $mappedValues = [
            'inquiryParameters' => [
                'presentationId' => $this->connectionSettings['presentationId'],
                'mandatorId' => $this->connectionSettings['mandatorId']
            ],
            'immoObjectCode' => $formValues['immoObjectCode']
        ];

        foreach (self::API_REQUIRED_FIELDS['realty'] as $field_name => $field) {
            $mappedValues['inquiryParameters'][$field_name] = $formValues[$field_name];
        }

        foreach ($this->settings['settings']['realtyRequest']['mapping'] as $field_name => $mapping_options) {
            if (!isset($mappedValues['inquiryParameters'][$field_name]) && !isset($mappedValues[$field_name])) {
                if (isset($mapping_options['addToComment']) &&
                    filter_var($mapping_options['addToComment'], FILTER_VALIDATE_BOOLEAN)) {
                    if (isset($mapping_options['commentLabel'])) {
                        $comments[] = $mapping_options['commentLabel'] . ': ' . $formValues[$field_name];
                    } else {
                        $comments[] = $field_name . ': ' . $formValues[$field_name];
                    }
                } else {
                    $mappedValues['inquiryParameters'][$field_name] = $formValues[$field_name];
                }
            }
        }
        $mappedValues['inquiryParameters']['comment'] = implode(',', $comments);

        return $mappedValues;
    }

    /**
     * @param array $formValues
     *
     * @return array
     */
    private function buildImmosolveContactApiArrayForPetitionRequest(array $formValues)
    {
        $comments = [];
        $mappedValues = [
            'presentationId' => $this->connectionSettings['presentationId'],
            'mandatorId' => $this->connectionSettings['mandatorId'],
            'objectIdentifier' => (int)$this->settings['immosolve']['api']['objectIdentifier'],
            'radius' => (int)$this->settings['immosolve']['api']['radius'],
            'latitude' => (double)$this->settings['immosolve']['api']['latitude'],
            'longitude' => (double)$this->settings['immosolve']['api']['longitude']
        ];

        foreach (self::API_REQUIRED_FIELDS['petition'] as $field_name => $field) {
            if (!$mappedValues[$field_name]) {
                $mappedValues[$field_name] = $formValues[$field_name];
            }
        }

        foreach ($this->settings['settings']['searchRequest']['mapping'] as $field_name => $mapping_options) {
            if (!isset($mappedValues[$field_name])) {
                if (isset($mapping_options['addToComment']) &&
                    filter_var($mapping_options['addToComment'], FILTER_VALIDATE_BOOLEAN)) {
                    if (isset($mapping_options['commentLabel'])) {
                        $comments[] = $mapping_options['commentLabel'] . ' ' . $formValues[$field_name]."\n";
                    } else {
                        $comments[] = $field_name . ': ' . $formValues[$field_name]."\n";
                    }
                } elseif ($formValues[$field_name]) {
                    $mappedValues[$field_name] = $formValues[$field_name];
                }
            }
        }
        $mappedValues['comment'] = implode('', $comments);

        return $mappedValues;
    }

    /**
     * @param $array_to_map
     * @param $type
     */
    public function map(&$array_to_map, $type)
    {
        $map_settings = ($type === 'petition') ? $this->settings['settings']['searchRequest']['mapping'] : $this->settings['settings']['realtyRequest']['mapping'];

        foreach ($map_settings as $api_field_name => $map_options) {
            if ($api_field_name !== $map_options['fieldName'] && isset($array_to_map[$map_options['fieldName']])) {
                $array_to_map[$api_field_name] = $array_to_map[$map_options['fieldName']];
                unset($array_to_map[$map_options['fieldName']]);
            }

            if (isset($map_options['options'])) {
                foreach ($map_options['options'] as $api_value => $form_value) {
                    if ($array_to_map[$api_field_name] === $form_value) {
                        $array_to_map[$api_field_name] = $api_value;
                        break;
                    }
                }
            }

            if (isset($map_options['concat'])) {
                $concat_value = [];
                ksort($map_options['concat']);
                foreach ($map_options['concat'] as $position => $field_name) {
                    if ($position !== 'separator' && $array_to_map[$api_field_name][$field_name]) {
                        $concat_value[] = $array_to_map[$api_field_name][$field_name];
                    }
                }
                $array_to_map[$api_field_name] = implode($map_options['concat']['separator'], $concat_value);
            }

            if ($api_field_name === 'location') {
                $districts = explode(',', $this->settings['settings']['searchRequest']['districts']);
                $district_names = [];
                foreach ($districts as $district) {
                    $single_district_composite = explode('|', $district);
                    $district_names[trim($single_district_composite[1])] = trim($single_district_composite[0]);
                }

                $concat_value = [];
                foreach ($array_to_map['location'] as $district_id) {
                    $concat_value[] = $district_names[$district_id];
                }
                $array_to_map['location'] = implode(',', $concat_value);
            }

            if (isset($map_options['cast']) || self::API_REQUIRED_FIELDS[$type][$api_field_name]['cast'] !== null) {
                $cast_type = $map_options['cast'] ?: self::API_REQUIRED_FIELDS[$type][$api_field_name]['cast'];

                switch ($cast_type) {
                    case 'int':
                    case 'integer':
                        $array_to_map[$api_field_name] = (int)$array_to_map[$api_field_name];
                        break;
                    case 'double':
                        $array_to_map[$api_field_name] = (double)$array_to_map[$api_field_name];
                        break;
                    case 'array':
                        $array_to_map[$api_field_name] = explode(',', $array_to_map[$api_field_name]);
                        break;
                }
            }

            if (!$array_to_map[$api_field_name]) {
                unset($array_to_map[$api_field_name]);
            }
        }
    }

    /**
     * @param $submitted_field_name
     * @param $type
     */
    public function getApiFieldName($submitted_field_name, $type)
    {
        $map_settings = ($type === 'petition') ? $this->settings['settings']['searchRequest']['mapping'] : $this->settings['settings']['realtyRequest']['mapping'];

        foreach ($map_settings as $api_field_name => $map_options) {
            if ($api_field_name !== $map_options['fieldName'] && $map_options['fieldName'] === $submitted_field_name) {
                return $api_field_name;
            }
        }

        return $submitted_field_name;
    }

    /**
     * @param $array_to_unmap
     * @param $type
     */
    public function unmap(&$array_to_unmap, $type)
    {
        $map_settings = ($type === 'petition') ? $this->settings['settings']['searchRequest']['mapping'] : $this->settings['settings']['realtyRequest']['mapping'];

        foreach ($map_settings as $api_field_name => $map_options) {
            if ($api_field_name !== $map_options['fieldName'] && isset($array_to_unmap[$api_field_name])) {
                $array_to_unmap[$map_options['fieldName']] = $array_to_unmap[$api_field_name];
                unset($array_to_unmap[$api_field_name]);
            }

            if (isset($map_options['options'])) {
                foreach ($map_options['options'] as $api_value => $form_value) {
                    if ($array_to_unmap[$map_options['fieldName']] === $api_value) {
                        $array_to_unmap[$map_options['fieldName']] = $form_value;
                        break;
                    }
                }
            }
        }
    }
}
