<?php
namespace SG\SgEstateImmosolve\Controller;

use SG\SgEstateImmosolve\Domain\Model\Region;
use SG\SgEstateImmosolve\Service\ImmosolveContactApiService;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;

class CommunicationController extends \TYPO3\CMS\Extbase\Mvc\Controller\ActionController
{
    /**
     * @var \SG\SgEstateImmosolve\Domain\Repository\RegionRepository
     * @TYPO3\CMS\Extbase\Annotation\Inject
     */
    protected $regionRepository;

    /**
     * @var \SG\SgEstateCore\Domain\Repository\ImmobilieRepository
     * @TYPO3\CMS\Extbase\Annotation\Inject
     */
    protected $immobilieRepository;

    /**
     * @var \SG\SgEstateImmosolve\Service\ImmosolveContactApiService
     * @TYPO3\CMS\Extbase\Annotation\Inject
     */
    protected $immosolveContactApiService;

    public function realtyRequestAction()
    {
        if ($this->request->hasArgument('realty')) {
            $realty = $this->immobilieRepository->findOneByUrlIdentifier($this->request->getArgument('realty'));
            $this->view->assign('realty', $realty);
        } elseif ($this->request->hasArgument('formValues')) {
            $formValues = $this->request->getArgument('formValues');
            $realty = $this->immobilieRepository->findByUid($formValues['realty']);
            $errors = [];
            $valid = true;

            // we need to map before validation in case api required fields have different form field names
            $this->immosolveContactApiService->map($formValues, 'realty');
            $this->validateRealtyRequest($formValues, $valid, $errors);

            if ($valid) {
                if ($this->settings['immosolve']['service'] === 'api') {
                    $sendStatus = $this->immosolveContactApiService->queueInquiry(
                        $formValues,
                        'object'
                    ) ? 'success' : 'sendfailed';
                } else {
                    $sendStatus = $this->sendTemplateEmail(
                        [$this->settings['settings']['immosolve']['email']],
                        $formValues['email'],
                        'Objektanfrage expr_' . $this->settings['settings']['immosolve']['gruppenid'],
                        'RealtyRequest',
                        ['realty' => $realty, 'formValues' => $formValues]
                    ) ? 'success' : 'sendfailed';
                }

                $this->view->assign('sendStatus', $sendStatus);
            } else {
                // we need to unmap so form values can be assigned to the correct fields & errors are shown correctly
                $this->immosolveContactApiService->unmap($formValues, 'realty');
                $this->immosolveContactApiService->unmap($errors, 'realty');
                $this->view->assignMultiple(['formValues' => $formValues, 'error' => $errors]);
            }

            $this->view->assign('realty', $realty);
        }
        //@todo: Redirect
    }

    public function searchRequestAction()
    {
        $submittedDataIsValid = true;
        $formErrors = [];
        $submittedFormData = [];
        $sendStatus = false;

        $submittedStep = $this->getSubmittedStep();
        $requestedStep = $this->getRequestedStep();

        if ($this->request->hasArgument('completeFormData')) {
            $completeFormData = json_decode($this->request->getArgument('completeFormData'), true);
        } else {
            $completeFormData = [];
        }

        if ($this->request->hasArgument('formValues')) {
            $submittedFormData = $this->request->getArgument('formValues');
            $completeFormData['step-' . $submittedStep] = $submittedFormData['step-' . $submittedStep];

            if (!empty($submittedFormData['step-' . $submittedStep])) {
                $currentFormValues = $submittedFormData['step-' . $submittedStep];
                $submittedDataIsValid = $this->validatePetitionRequest($currentFormValues, $formErrors);
            } else {
                $submittedDataIsValid = false;
            }
        } elseif ($requestedStep > 1) {
            $submittedDataIsValid = false;
        }

        if ($submittedDataIsValid) {
            $currentStep = $requestedStep;

            if ($currentStep > $this->settings['settings']['searchRequest']['formSteps']) {
                $apiData = [];
                foreach ($completeFormData as $data) {
                    $apiData = array_merge($apiData, $data);
                }

                $this->immosolveContactApiService->map($apiData, 'petition');

                if ($this->settings['immosolve']['service'] === 'api') {
                    $sendStatus = $this->immosolveContactApiService->queueInquiry(
                        $apiData,
                        'petition'
                    ) ? 'success' : 'sendfailed';
                } else {
                    $sendStatus = $this->sendTemplateEmail(
                        [$this->settings['settings']['immosolve']['searchRequest']['email']],
                        $submittedFormData['step-1']['email'],
                        'Anfrage von portal ' . $this->settings['settings']['immosolve']['searchRequest']['gruppenid'],
                        'SearchRequest',
                        ['formValues' => $submittedFormData]
                    ) ? 'success' : 'sendfailed';
                }
            }
        } else {
            $currentStep = $submittedStep;
        }

        $this->view->assignMultiple([
            'birthdayYears' => $this->getBirthdayYears(),
            'currentStep' => $currentStep,
            'errors' => $formErrors,
            'completeFormData' => json_encode($completeFormData),
            'formValues' => $completeFormData['step-' . $currentStep],
            'sendStatus' => $sendStatus ? 'success' : '',
            'regions' => $this->getRegions()
        ]);
    }

    /**
     * Action used for several AJAX Calls
     *
     * ViewHelper for URI Function "contactForm" <f:uri.action extension='SgEstateImmosolve' action='ajax' controller='Communication' arguments='{function:"contactForm"}' additionalParams='{type: 1579525392}' noCacheHash='true' />
     *
     * @throws \TYPO3\CMS\Extbase\Mvc\Exception\NoSuchArgumentException
     */
    public function ajaxAction() {
        // Bail early if required argument does not exist
        if( !$this->request->hasArgument('function') ) {
            return;
        }

        $function = $this->request->getArgument('function');
        $this->view->assign( 'function', $function );
        switch ( $function ) {
            case 'contactForm':
                $this->processContactForm();
                break;
            default:
                break;
        }

    }

    /**
     * @throws \TYPO3\CMS\Extbase\Mvc\Exception\NoSuchArgumentException
     */
    private function processContactForm() {
        if ($this->request->hasArgument('realtyUid')){
            $realty = $this->immobilieRepository->findOneByUid($this->request->getArgument('realtyUid'));
            $this->view->assign('realty', $realty);
        }

        // set default for unsent form
        $this->view->assign('showResult', false);
        $this->view->assign('success', false);

        if( $this->request->hasArgument('formValues') ) {
            $formValues = $this->request->getArgument('formValues');
            $realty = $this->immobilieRepository->findByUid($formValues['realty']);
            $this->view->assign('realty', $realty);
            $errors = [];
            $valid = true;

            // we need to map before validation in case api required fields have different form field names
            $this->immosolveContactApiService->map($formValues, 'realty');
            $this->validateRealtyRequest($formValues, $valid, $errors);

            if ($valid) {
                if ($this->settings['immosolve']['service'] === 'api') {
                    $sendStatus = $this->immosolveContactApiService->queueInquiry(
                        $formValues,
                        'object'
                    ) ? 'success' : 'sendfailed';
                } else {
                    $sendStatus = $this->sendTemplateEmail(
                        [$this->settings['settings']['immosolve']['email']],
                        $formValues['email'],
                        'Objektanfrage expr_' . $this->settings['settings']['immosolve']['gruppenid'],
                        'RealtyRequest',
                        ['realty' => $realty, 'formValues' => $formValues]
                    ) ? 'success' : 'sendfailed';
                }

                $this->view->assign('sendStatus', $sendStatus);
                $this->view->assign('showResult', true);
                if( $sendStatus === 'success' ) {
                    $this->view->assign('success', true);
                }
                else {
                    $this->view->assign('success', false);
                }
            } else {
                // we need to unmap so form values can be assigned to the correct fields & errors are shown correctly
                $this->immosolveContactApiService->unmap($formValues, 'realty');
                $this->immosolveContactApiService->unmap($errors, 'realty');
                $this->view->assignMultiple(['formValues' => $formValues, 'error' => $errors]);
            }
        }
    }

    /**
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     */
    private function getRegions()
    {
        $regions = $this->regionRepository->findByParent(0);

        foreach ($regions as $region) {
            $this->buildRegionTree($region);
        }

        return $regions;
    }

    private function buildRegionTree(Region $region)
    {
        $children = $this->regionRepository->findByParent($region);

        if ($children->count() > 0) {
            $region->setChildren($children);

            foreach ($children as $child) {
                $this->buildRegionTree($child);
            }
        }
    }

    private function getChildRegionsRecursive($region, &$regions, $index)
    {
        $children = $this->regionRepository->findByParent($region);

        if ($children->count() === 0) {
            return [
                'region' => $region,
                'children' => []
            ];

            /*$regions[$index]['children'][] = [
                'region' => $region,
                'children' => []
            ];

            return $regions;*/
        }

        foreach ($children as $child) {
            $this->getChildRegionsRecursive($child, $regions, $index);
        }

        /*$regions[$index] = [
            'region' => $region,
        ];

        foreach ($children as $child) {
            $this->getChildRegionsRecursive($child, $regions, $index);
        }*/

        $index++;
    }

    /**
     * @param $formValues
     * @param $valid
     * @param $errors
     */
    private function validateRealtyRequest(
        $formValues,
        &$valid,
        &$errors
    ) {
        $possibly_required_fields = array_merge(
            $this->settings['settings']['realtyRequest']['validation'],
            ImmosolveContactApiService::API_REQUIRED_FIELDS['realty']
        );
        foreach ($possibly_required_fields as $field_name => $field) {
            if ($field['required'] && (!isset($formValues[$field_name]) || $formValues[$field_name] === '')) {
                $valid = false;
                $errors[$field_name] = true;
            }
        }

        if (!GeneralUtility::validEmail($formValues['email'])) {
            $valid = false;
            $errors['email'] = true;
        }
    }

    /**
     * @param array $recipients
     * @param array $bcc
     * @param $sender
     * @param $subject
     * @param $templateName
     * @param array $variables
     * @param string $format
     *
     * @return bool
     */
    private function sendTemplateEmail(
        array $recipients,
        $sender,
        $subject,
        $templateName,
        array $variables = [],
        $format = 'text/plain',
        array $bcc = []
    ) {
        /**
         * @var $emailView \TYPO3\CMS\Fluid\View\StandaloneView
         * @var $message \TYPO3\CMS\Core\Mail\MailMessage
         */
        $objectManager = GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Object\ObjectManager');

        $templateRootPath = \TYPO3\CMS\Core\Utility\ExtensionManagementUtility::extPath('sg_estate_immosolve') . 'Resources/Private/Templates/';
        $templatePathAndFilename = $templateRootPath . 'Email/' . $templateName . '.html';

        $emailView = $objectManager->get('TYPO3\CMS\Fluid\View\StandaloneView');
        $emailView->setTemplatePathAndFilename($templatePathAndFilename);
        $emailView->setFormat('html');
        $emailView->assignMultiple($variables);

        $emailBody = $emailView->render();

        $message = GeneralUtility::makeInstance('TYPO3\CMS\Core\Mail\MailMessage');

        if (GeneralUtility::validEmail($sender)) {
            $message->setFrom($sender);
        } else {
            return false;
        }

        // Add Recipients
        $recipients = array_filter($recipients);
        if (count($recipients) > 0) {
            foreach ($recipients as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addTo($email);
                }
            }
        } else {
            return false;
        }

        if (count($bcc) > 0) {
            foreach ($bcc as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addBcc($email);
                }
            }
        }

        if ($subject !== '') {
            $message->setSubject($subject);
        } else {
            return false;
        }

        $message->setBody($emailBody, $format);

        $message->send();

        return $message->isSent();
    }

    /**
     * @return int
     */
    private function getRequestedStep()
    {
        if ($this->request->hasArgument('submit')) {
            $direction = $this->request->getArgument('submit');
            $submitted_step = $this->getSubmittedStep();

            switch ($direction) {
                case LocalizationUtility::translate('template.contact.next', 'sg_estate_immosolve'):
                case LocalizationUtility::translate('template.contact.submit', 'sg_estate_immosolve'):
                    $requested_step = $submitted_step + 1;
                    break;
                case LocalizationUtility::translate('template.contact.previous', 'sg_estate_immosolve'):
                    $requested_step = $submitted_step - 1;
                    break;
                default:
                    $requested_step = 1;
            }
        } else {
            $requested_step = 1;
        }

        return $requested_step;
    }

    /**
     * @return int|string
     * @throws \TYPO3\CMS\Extbase\Mvc\Exception\NoSuchArgumentException
     */
    private function getSubmittedStep()
    {
        return $this->request->hasArgument('currentStep') ? $this->request->getArgument('currentStep') : 1;
    }

    private function getBirthdayYears()
    {
        $birthday_years = [];
        $current_year = (int)date('Y');

        for ($i = $current_year - 18; $i >= $current_year - 90; $i--) {
            $birthday_years[$i] = $i;
        }

        return $birthday_years;
    }

    /**
     * @param $form_values
     * @param array $required_fields
     * @param array $errors
     * @param $data_is_valid
     * @param bool $against_api_fields
     *
     * @return bool
     */
    private function validatePetitionRequest(
        $form_values,
        array &$errors
    ) {
        $data_is_valid = true;
        $possibly_required_fields = array_merge(
            $this->settings['settings']['searchRequest']['validation'],
            ImmosolveContactApiService::API_REQUIRED_FIELDS['petition']
        );

        foreach ($form_values as $field => $value) {
            $api_fieldname = $this->immosolveContactApiService->getApiFieldName($field, 'petition');

            if (empty($value) && ($possibly_required_fields[$field]['required'] || $possibly_required_fields[$api_fieldname]['required'])) {
                $data_is_valid = false;
                $errors[$field] = true;
            }

            if (($field === 'email') && !GeneralUtility::validEmail($form_values['email'])) {
                $data_is_valid = false;
                $errors['email'] = true;
            }
        }

        return $data_is_valid;
    }
}
