<?php

namespace Daho\SgEstateImmosolve\Tasks;

use Daho\SgEstateImmosolve\Domain\Model\QueueItem;
use Daho\SgEstateImmosolve\Service\ImmosolveContactApiService;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Object\ObjectManager;
use TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager;
use Daho\SgEstateImmosolve\Domain\Repository\QueueItemRepository;
use TYPO3\CMS\Core\Mail\MailMessage;
use TYPO3\CMS\Scheduler\Task\AbstractTask;

class QueueProcessorTask extends AbstractTask
{
    /**
     * @var \TYPO3\CMS\Extbase\Object\ObjectManager
     */
    private $objectManager;

    /**
     * @var \Daho\SgEstateImmosolve\Domain\Repository\QueueItemRepository
     */
    private $queueItemRepository;

    /**
     * @var \TYPO3\CMS\Extbase\Persistence\Generic\PersistenceManager
     */
    protected $persistenceManager;

    /**
     * @var array
     */
    protected $settings = [];

    /**
     * @var int
     */
    protected $limitPerTaskExcecution = 50;

    protected $reportBody = '';

    private function initialize()
    {
        $this->objectManager = GeneralUtility::makeInstance(ObjectManager::class);
        $this->persistenceManager = $this->objectManager->get(PersistenceManager::class);
        $this->queueItemRepository = $this->objectManager->get(QueueItemRepository::class);

        /**
         * Get typoscript settings of root node
         */
        $configurationManager = $this->objectManager->get(ConfigurationManager::class);
        $extbaseFrameworkConfiguration = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT);
        $this->settings = $extbaseFrameworkConfiguration['plugin.']['tx_sgestateimmosolve_pi1.']['settings.']['immosolve.'];
    }

    /**
     * @return bool
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\IllegalObjectTypeException
     */
    public function execute()
    {
        $this->initialize();
        $start = microtime(true);

        $this->reportBody .= '<h1>Report der Abarbeitung der Immosolve-Warteschlange</h1>' . "\n";
        $this->addReportLine('QueueProcessor gestartet');

        $items = $this->queueItemRepository->findAllForQueueProcessing();

        if ($items && count($items) > 0) {
            $this->addReportLine('Insgesamt ' . count($items) . ' Elemente gefunden.');
            $this->addReportLine('Max. ' . $this->limitPerTaskExcecution . ' Elemente bei dieser Ausführung abarbeiten...');

            /** @var QueueItem $item */
            foreach ($items as $index => $item) {
                // only process the defined number of items per task excecution
                // so tasks do not overlap
                if ($index < $this->limitPerTaskExcecution) {
                    $parameter = json_decode($item->getPayload(), true);

                    if ($parameter) {
                        $immosolveContactApiService = $this->objectManager->get(ImmosolveContactApiService::class);
                        $immosolveRequestResult = $immosolveContactApiService->doInquiry($parameter, $item->getRequestType());

                        if ($immosolveRequestResult->success === true) {
                            // transmission was successful
                            $this->addReportLine('#' . $item->getUid() . ' vom Typ ' . $item->getRequestType() . ' erfolgreich abgearbeitet. Lösche Element.');

                            // remove element afterwards so we do not send it again
                            $this->queueItemRepository->remove($item);
                            $this->persistenceManager->persistAll();
                        } else {
                            // submission did not work
                            $this->addReportLine('#' . $item->getUid() . ' vom Typ ' . $item->getRequestType() . ' konnte nicht versendet werden. Fehlermeldung folgt:');
                            $this->addReportLine(json_encode($immosolveRequestResult));
                            $this->addReportLine('JSON-Parameter: ' . $item->getPayload());

                            $this->sendErrorReport($item, $immosolveRequestResult);

                            $this->addReportLine('Fehlermail gesendet.');
                        }

                        sleep(4);
                    }
                } else {
                    break;
                }
            }
            $time_elapsed_secs = microtime(true) - $start;
            $this->addReportLine('Import dauerte ' . $time_elapsed_secs . ' Sekunden. Beenden.');

            if ((int)$this->settings['sendAdminEmail'] === 1) {
                $this->sendAdminReport();
            }
        }

        return true;
    }

    /**
     * @param $string
     */
    private function addReportLine($string)
    {
        $this->reportBody .= '<p>[' . date('Y-m-d H:i') . '] ' . $string . '</p>' . "\n";
    }

    /**
     * @param QueueItem $item
     * @param $immosolveRequestResult
     *
     * @return void
     */
    private function sendErrorReport(QueueItem $item, $immosolveRequestResult)
    {
        $body = '' .
            'Das Element #' . $item->getUid() . ' mit Request Type "' . $item->getRequestType() . '" konnte nicht versendet werden.' . "\n\n" .
            'Daten:' . "\n" .
            $item->getPayload() . "\n\n" .
            'Fehlercode:' . "\n" .
            json_encode($immosolveRequestResult) . "\n\n" .
            date('Y-m-d H:i:s') .
            '';

        $message = GeneralUtility::makeInstance(MailMessage::class);
        $message
            ->setFrom($this->settings['errorEmailSender'])
            ->addTo($this->settings['errorEmail'])
            ->setSubject('[FEHLER] Warteschlangenelement konnte nicht verarbeitet werden')
            ->setBody($body, 'text/text')
            ->send();
    }

    /**
     *
     * @return void
     */
    private function sendAdminReport()
    {
        $reportMail = GeneralUtility::makeInstance(MailMessage::class);
        $reportMail->setFrom($this->settings['adminEmailSender']);
        $reportMail->addTo($this->settings['adminEmail']);
        $reportMail->setSubject('Warteschlangen-Report: Immosolve-Contact-Api');
        $reportMail->setBody($this->reportBody, 'text/html');
        $reportMail->send();
    }

}