<?php
namespace Daho\SgEstateImmosolve\Api;

// *********************************************************************************************************************
// * (C) 2016 Immosolve GmbH, Tegelbarg 43, 24576 Bad Bramstedt
// *********************************************************************************************************************

/**
 * Class to interact with Immosolve Contact API. These class helps sending inquiry of objects and inquiry
 * of petitions by using oauth authorization v1.
 *
 * @author Martin Voigt (Immosolve GmbH)
 */
class ImmosolveContactApi
{

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // constants
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    const KEY_SERVER_PILOT = "pilot";
    const KEY_SERVER_ASP1 = "asp1";
    const KEY_SERVER_ASP2 = "asp2";
    const KEY_SERVER_ASP3 = "asp3";

    const PROTOCOL = "https://";
    const HTTP_METHOD = "POST";

    const HOST_PILOT = "test.immosolve.eu";
    const HOST_ASP1 = "api1.immosolve.eu";
    const HOST_ASP2 = "api2.immosolve.eu";
    const HOST_ASP3 = "api3.immosolve.eu";

    const PATH_INQUIRY_OF_OBJECT = "/immosolve2/api/rest/contact/v1/inquiryOfObject";
    const PATH_INQUIRY_OF_PETITION = "/immosolve2/api/rest/contact/v1/inquiryOfPetition";

    const KEY_OAUTH_SIGNATUR_METHOD = "HMAC-SHA1";

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // attributes
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    private $consumerKey = "";
    private $consumerSecret = "";
    private $token = "";
    private $tokenSecret = "";
    private $host = "";
    private $curl;

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // set/get methods
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * @return string consumerKey the oauth_consumer_key
     */
    public function getConsumerKey()
    {
        return $this->consumerKey;
    }

    /**
     * @param string consumerKey the oauth_consumer_key to set
     */
    public function setConsumerKey($consumerKey)
    {
        $this->consumerKey = $consumerKey;
    }

    /**
     * @return string consumerSecret the oauth_consumer_secret
     */
    public function getConsumerSecret()
    {
        return $this->consumerSecret;
    }

    /**
     * @param string consumerSecret the oauth_consumer_secret to set
     */
    public function setConsumerSecret($consumerSecret)
    {
        $this->consumerSecret = $consumerSecret;
    }

    /**
     * @return string token the oauth_token
     */
    public function getToken()
    {
        return $this->token;
    }

    /**
     * @param string token the oauth_token to set
     */
    public function setToken($token)
    {
        $this->token = $token;
    }

    /**
     * @return string tokenSecret the oauth_token_secret
     */
    public function getTokenSecret()
    {
        return $this->tokenSecret;
    }

    /**
     * @param string tokenSecret the oauth_token_secret to set
     */
    public function setTokenSecret($tokenSecret)
    {
        $this->tokenSecret = $tokenSecret;
    }

    /**
     * @return string host the service host
     */
    private function getHost()
    {
        return $this->host;
    }

    /**
     * @param string host the service host to set
     */
    private function setHost($host)
    {
        $this->host = $host;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // methods
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Does some initial stuff
     */
    public function init()
    {
        $this->setServer(self::KEY_SERVER_PILOT);
        $this->curl = curl_init();
    }

    /**
     * Sets the server of the contact api.
     * Possible values are pilot, asp1, asp2 and asp3.
     * Setting the server the host is set automatically.
     *
     * @param string server the service serve to set
     */
    public function setServer($server)
    {
        switch ($server) {
            case self::KEY_SERVER_PILOT:
                $this->setHost(self::HOST_PILOT);
                break;
            case self::KEY_SERVER_ASP1:
                $this->setHost(self::HOST_ASP1);
                break;
            case self::KEY_SERVER_ASP2:
                $this->setHost(self::HOST_ASP2);
                break;
            case self::KEY_SERVER_ASP3:
                $this->setHost(self::HOST_ASP3);
                break;
            default:
                $this->setHost(self::HOST_PILOT);
        }
    }

    /**
     * Sends an inquiry of object.
     *
     * @param $params
     * @param $type
     * @return object response the response object of the inquiry of object
     */
    public function doInquiry($params, $type)
    {
        $oauth = [
            'oauth_consumer_key' => $this->getConsumerKey(),
            'oauth_token' => $this->getToken(),
            'oauth_signature_method' => 'HMAC-SHA1',
            'oauth_timestamp' => time(),
            'oauth_nonce' => time(),
            'oauth_version' => '1.0'
        ];
        $base_uri = ($type === 'object') ? $this->getUrlForInquiryOfObject() : $this->getUrlForInquiryOfPetition();
        $base_info = $this->buildBaseString($base_uri, self::HTTP_METHOD, $oauth);
        $composite_key = rawurlencode($this->getConsumerSecret()) . '&' . rawurlencode($this->getTokenSecret());
        $oauth_signature = base64_encode(hash_hmac('sha1', $base_info, $composite_key, true));
        $oauth['oauth_signature'] = $oauth_signature;

        $this->setCurlOptions($oauth, $params, $base_uri);

        $response = curl_exec($this->curl);
        $info = curl_getinfo($this->curl);
        if ($info['http_code'] == 401) {
            $returnString = '{"success": "false", "message": "Authorization failed!"}';
        } elseif (curl_errno($this->curl)) {
            $returnString = '{"success": "false", "message": "' . curl_error($this->curl) . '"}';
        } else {
            $returnString = $response;
        }

        curl_close($this->curl);

        return json_decode($returnString, false);
    }

    /**
     * Produces the URL for an inquiry of object
     *
     * @return string url the inquiry of object url
     */
    private function getUrlForInquiryOfObject()
    {
        return self::PROTOCOL . $this->getHost() . self::PATH_INQUIRY_OF_OBJECT;
    }

    /**
     * Produces the URL for an inquiry of petition
     *
     * @return string url the inquiry of petition url
     */
    private function getUrlForInquiryOfPetition()
    {
        return self::PROTOCOL . $this->getHost() . self::PATH_INQUIRY_OF_PETITION;
    }

    /**
     * Sets the options of the curl object for an inquiry of object
     *
     * @param $oauth
     * @param $params
     * @param $base_uri
     */
    private function setCurlOptions($oauth, $params, $base_uri)
    {
        $options = [
            CURLOPT_HTTPHEADER => $this->getHttpHeader($oauth),
            CURLOPT_HEADER => false,
            CURLOPT_URL => $base_uri,
            CURLOPT_POST => 1,
            CURLOPT_POSTFIELDS => json_encode($params),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_VERBOSE => 1
        ];
        curl_setopt_array($this->curl, $options);
    }

    /**
     * Sets the http header for the request.
     * Sets the content type and the oauth params as http header.
     *
     * @param array oauth the oauth parameters to set as array (oauth_consumer_key, oauth_token, oauth_signature_method, oauth_timestamp, oauth_nonce, oauth_version)
     * @return array httpHeader the http header for the request
     */
    private function getHttpHeader($oauth)
    {
        $httpHeader = [
            'Content-Type: application/json',
            'Accept: application/json',
            $this->buildAuthorizationHeader($oauth)
        ];

        return $httpHeader;
    }

    /**
     * Creates the base string for oauth authorization.
     *
     * @param string baseURI the URI for the request
     * @param string method the http method
     * @param array params the params, which should be included in base string
     * @return string baseString the base string for oauth authorization
     */
    private function buildBaseString($baseURI, $method, $params)
    {
        $r = [];

        ksort($params);

        foreach ($params as $key => $value) {
            $r[] = "$key=" . rawurlencode($value);
        }

        return $method . "&" . rawurlencode($baseURI) . '&' . rawurlencode(implode('&', $r));
    }

    /**
     * Produces the oauth authorization header
     *
     * @param array oauth the oauth parameters
     * @return string r the oauth authorization header
     */
    private function buildAuthorizationHeader($oauth)
    {
        $r = 'Authorization: OAuth ';
        $values = [];

        foreach ($oauth as $key => $value) {
            $values[] = '' . $key . '="' . rawurlencode($value) . '"';
        }
        $r .= implode(', ', $values);

        return $r;
    }

}