<?php
namespace SG\SgEstateCore\Util;

use Doctrine\DBAL\Exception;
use JsonException;
use SG\SgEstateCore\Domain\Repository\OrtRepository;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Configuration\Exception\ExtensionConfigurationExtensionNotConfiguredException;
use TYPO3\CMS\Core\Configuration\Exception\ExtensionConfigurationPathDoesNotExistException;
use TYPO3\CMS\Core\Configuration\ExtensionConfiguration;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Mail\FluidEmail;
use TYPO3\CMS\Core\Mail\MailerInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;

class Services
{
    /**
     * @var string
     */
    private string $standardCountry = 'Germany';

    public function __construct(
        private readonly ConnectionPool $connectionPool,
        private readonly FrontendInterface $sgEstateCoreCache,
        private readonly ExtensionConfiguration $extensionConfiguration,
        protected ConfigurationManager $configurationManager,
        protected OrtRepository $ortRepository,
    ){
    }

    /**
     * @throws Exception
     */
    public function deleteAllRealties(): void
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $result = $queryBuilder
            ->select('uid', 'kontaktperson')->from('tx_sgestatecore_domain_model_immobilie')->executeQuery();

        if ($result->rowCount() >= 1) {
            $this->deleteRealties($result->fetchAllAssociative());
        }
    }

    /**
     * @throws Exception
     */
    public function deleteAllUpdatableRealties($importNumber = null): void
    {
        $result = null;
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        if (is_null($importNumber)) {
            $result = $queryBuilder
                ->select('uid', 'kontaktperson')
                ->from('tx_sgestatecore_domain_model_immobilie')->where($queryBuilder->expr()->eq('updatable', $queryBuilder->createNamedParameter(1)))->executeQuery();
        } elseif (is_numeric($importNumber)) {
            $result = $queryBuilder
                ->select('uid', 'kontaktperson')
                ->from('tx_sgestatecore_domain_model_immobilie')
                ->where($queryBuilder->expr()->eq('updatable', $queryBuilder->createNamedParameter(1)))->andWhere($queryBuilder->expr()->eq('import_number', $queryBuilder->createNamedParameter($importNumber)))->executeQuery();
        }

        if ($result !== null && $result->rowCount() >= 1) {
            $this->deleteRealties($result->fetchAllAssociative());
        }
    }

    /**
     * @throws Exception
     */
    public function deleteRealties($realties): array
    {
        $log = [];
        $log['realtiesInputCount'] = count($realties);
        $result = array();
        foreach ($realties as $row) {
            $result[$row['uid']] = is_numeric($row['kontaktperson']) ? $row['kontaktperson'] : 0;
        }
        $log['realtiesInputData'] = $result;

        $log['realtiesTryToRemoveAllContactPersonsStart'] = 'OK';
        if(count(array_values($result)) >= 1) {
            $queryBuilder__tx_sgestatecore_domain_model_kontaktperson = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_kontaktperson');
            $queryBuilder__tx_sgestatecore_domain_model_kontaktperson
                ->delete('tx_sgestatecore_domain_model_kontaktperson')->where($queryBuilder__tx_sgestatecore_domain_model_kontaktperson->expr()->in('uid', array_values($result)))->executeStatement();
            $log['realtiesTryToRemoveAllContactPersonsStop'] = 'OK';

            $log['realtiesTryToRemoveAllManyToManyStart'] = 'OK';
            $queryBuilder__tx_sgestatecore_immobilie_ausbaustufe_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_ausbaustufe_mm');
            $queryBuilder__tx_sgestatecore_immobilie_ausbaustufe_mm->delete('tx_sgestatecore_immobilie_ausbaustufe_mm')->where($queryBuilder__tx_sgestatecore_immobilie_ausbaustufe_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_badausstattung_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_badausstattung_mm');
            $queryBuilder__tx_sgestatecore_immobilie_badausstattung_mm->delete('tx_sgestatecore_immobilie_badausstattung_mm')->where($queryBuilder__tx_sgestatecore_immobilie_badausstattung_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_bauweise_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_bauweise_mm');
            $queryBuilder__tx_sgestatecore_immobilie_bauweise_mm->delete('tx_sgestatecore_immobilie_bauweise_mm')->where($queryBuilder__tx_sgestatecore_immobilie_bauweise_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_bodenbelag_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_bodenbelag_mm');
            $queryBuilder__tx_sgestatecore_immobilie_bodenbelag_mm->delete('tx_sgestatecore_immobilie_bodenbelag_mm')->where($queryBuilder__tx_sgestatecore_immobilie_bodenbelag_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_dachform_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_dachform_mm');
            $queryBuilder__tx_sgestatecore_immobilie_dachform_mm->delete('tx_sgestatecore_immobilie_dachform_mm')->where($queryBuilder__tx_sgestatecore_immobilie_dachform_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_erschliessungdetails_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_erschliessungdetails_mm');
            $queryBuilder__tx_sgestatecore_immobilie_erschliessungdetails_mm->delete('tx_sgestatecore_immobilie_erschliessungdetails_mm')->where($queryBuilder__tx_sgestatecore_immobilie_erschliessungdetails_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_heizungsart_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_heizungsart_mm');
            $queryBuilder__tx_sgestatecore_immobilie_heizungsart_mm->delete('tx_sgestatecore_immobilie_heizungsart_mm')->where($queryBuilder__tx_sgestatecore_immobilie_heizungsart_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_befeuerungsart_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_befeuerungsart_mm');
            $queryBuilder__tx_sgestatecore_immobilie_befeuerungsart_mm->delete('tx_sgestatecore_immobilie_befeuerungsart_mm')->where($queryBuilder__tx_sgestatecore_immobilie_befeuerungsart_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_himmelrichtung_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_himmelrichtung_mm');
            $queryBuilder__tx_sgestatecore_immobilie_himmelrichtung_mm->delete('tx_sgestatecore_immobilie_himmelrichtung_mm')->where($queryBuilder__tx_sgestatecore_immobilie_himmelrichtung_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_kuechenausstattung_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_kuechenausstattung_mm');
            $queryBuilder__tx_sgestatecore_immobilie_kuechenausstattung_mm->delete('tx_sgestatecore_immobilie_kuechenausstattung_mm')->where($queryBuilder__tx_sgestatecore_immobilie_kuechenausstattung_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_nutzungsart_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_nutzungsart_mm');
            $queryBuilder__tx_sgestatecore_immobilie_nutzungsart_mm->delete('tx_sgestatecore_immobilie_nutzungsart_mm')->where($queryBuilder__tx_sgestatecore_immobilie_nutzungsart_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_vermarktungsart_mm');
            $queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm->delete('tx_sgestatecore_immobilie_vermarktungsart_mm')->where($queryBuilder__tx_sgestatecore_immobilie_vermarktungsart_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();

            $queryBuilder__tx_sgestatecore_immobilie_weitereadressen_kontaktperson_mm = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_immobilie_weitereadressen_kontaktperson_mm');
            $queryBuilder__tx_sgestatecore_immobilie_weitereadressen_kontaktperson_mm->delete('tx_sgestatecore_immobilie_weitereadressen_kontaktperson_mm')->where($queryBuilder__tx_sgestatecore_immobilie_weitereadressen_kontaktperson_mm->expr()->in('uid_local', array_keys($result)))->executeStatement();
            $log['realtiesTryToRemoveAllManyToManyStop'] = 'OK';

            $log['realtiesGetAllBinariesForAllRealtiesFromInputStart'] = 'OK';
            $queryBuilder__tx_sgestatecore_domain_model_anhang = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhang');
            $resultAttachments = $queryBuilder__tx_sgestatecore_domain_model_anhang
                ->select('datei')
                ->from('tx_sgestatecore_domain_model_anhang')->where($queryBuilder__tx_sgestatecore_domain_model_anhang->expr()->in('immobilie', array_keys($result)))->executeQuery();
            $rows = $resultAttachments->fetchAllAssociative();
            $log['realtiesGetAllBinariesForAllRealtiesFromInputStop'] = 'OK';

            $log['realtiesCollectAllBinariesIntoArrayIdPathStart'] = 'OK';
            $anhange = array();
            foreach ($rows as $attachment) {
                if (!empty($attachment['uid'])) {
                    $anhange[$attachment['uid']] = '"/uploads/tx_sgestatecore/media/' .  $attachment['datei'] . '"';
                }
            }
            $log['realtiesCollectAllBinariesIntoArrayIdPathStop'] = 'OK';
            if(is_array($anhange) && count($anhange) >= 1) {
                $log['realtiesGetAllSysRelationsForBinariesStart'] = 'OK';
                $queryBuilder__sys_file = $this->connectionPool->getQueryBuilderForTable('sys_file');
                $resultSysFile = $queryBuilder__sys_file
                    ->select('uid')
                    ->from('sys_file')->where($queryBuilder__sys_file->expr()->in('identifier', array_values($anhange)))->executeQuery();
                $rowsSysFile = $resultSysFile->fetchAllAssociative();
                $log['realtiesGetAllSysRelationsForBinariesStop'] = 'OK';

                $syss = array();
                foreach ($rowsSysFile as $sys) {
                    $syss[] = $sys['uid'];
                }
                $log['realtiesCollectAllSysRelationsIdsStart'] = 'OK';
                $log['realtiesGetAllProcessedDataStart'] = 'OK';
                $queryBuilder__sys_file_processedfile = $this->connectionPool->getQueryBuilderForTable('sys_file_processedfile');
                $resultProcessedFile = $queryBuilder__sys_file_processedfile
                    ->select('uid', 'identifier')
                    ->from('sys_file_processedfile')->where($queryBuilder__sys_file_processedfile->expr()->in('original', empty($syss) ? array(0) : $syss))->executeQuery();
                $rowsProcessedFile = $resultProcessedFile->fetchAllAssociative();
                $log['realtiesGetAllProcessedDataStop'] = 'OK';
                $log['realtiesDeleteAllProcessedFilesStart'] = 'OK';
                $procs = array();
                foreach ($rowsProcessedFile as $processedFile) {
                    $procs[] = $processedFile['uid'];
                    if ($processedFile['identifier'] !== '') {
                        @unlink(Environment::getPublicPath() . $processedFile['identifier']);
                    }
                }
                $log['realtiesDeleteAllProcessedFilesStop'] = 'OK';
                $log['realtiesDeleteAllSysFilesRelationsStart'] = 'OK';
                $queryBuilder__sys_file = $this->connectionPool->getQueryBuilderForTable('sys_file');
                $queryBuilder__sys_file
                    ->delete('sys_file')->where($queryBuilder__sys_file->expr()->in('uid', empty($syss) ? array(0) : $syss))->executeStatement();
                $log['realtiesDeleteAllSysFilesRelationsStop'] = 'OK';
                $log['realtiesDeleteAllProcessedFilesRelationsStart'] = 'OK';
                $queryBuilder__sys_file_processedfile = $this->connectionPool->getQueryBuilderForTable('sys_file_processedfile');
                $queryBuilder__sys_file_processedfile
                    ->delete('sys_file_processedfile')->where($queryBuilder__sys_file_processedfile->expr()->in('uid', empty($procs) ? array(0) : $procs))->executeStatement();
                $log['realtiesDeleteAllProcessedFilesRelationsStop'] = 'OK';
                $log['realtiesDeleteAllOriginalFilesStart'] = 'OK';
                foreach ($anhange as $anhang) {
                    if ($anhange !== '/uploads/tx_sgestatecore/media/') {
                        @unlink(Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/' . $anhang);
                    }
                }
                $log['realtiesDeleteAllOriginalFilesStop'] = 'OK';
                $log['realtiesRmoveAllBinariesDataStart'] = 'OK';
                $queryBuilder__tx_sgestatecore_domain_model_anhang = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_anhang');
                $queryBuilder__tx_sgestatecore_domain_model_anhang
                    ->delete('tx_sgestatecore_domain_model_anhang')->where($queryBuilder__tx_sgestatecore_domain_model_anhang->expr()->in('immobilie', array_keys($result)))->executeStatement();
                $log['realtiesRmoveAllBinariesDataStop'] = 'OK';
            }

            $log['realtiesRmoveAllDataStart'] = 'OK';
            $queryBuilder__tx_sgestatecore_domain_model_immobilie = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
            $queryBuilder__tx_sgestatecore_domain_model_immobilie
                ->delete('tx_sgestatecore_domain_model_immobilie')->where($queryBuilder__tx_sgestatecore_domain_model_immobilie->expr()->in('uid', array_keys($result)))->executeStatement();
            $log['realtiesRmoveAllDataStop'] = 'OK';
            $log['realtiesFlushCacheStart'] = 'OK';
            $this->sgEstateCoreCache->flushByTag('immobilie');
            $log['realtiesFlushCacheStop'] = 'OK';
        } else {
            $log['realtiesDelete'] = 'NOT OK';
            $log['realtiesTryToRemoveAllManyToManyStop'] = 'NOT OK';
        }
        return $log;
    }

    /**
     * @param bool $renewAll
     * @throws Exception
     * @throws ExtensionConfigurationExtensionNotConfiguredException
     * @throws ExtensionConfigurationPathDoesNotExistException
     * @throws JsonException
     */
    public function geocodeRealties(bool $renewAll = false): void
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('tx_sgestatecore_domain_model_immobilie.uid', 'tx_sgestatecore_domain_model_immobilie.objekt_plz as zip', 'tx_sgestatecore_domain_model_immobilie.objekt_strasse as street', 'tx_sgestatecore_domain_model_immobilie.objekt_hausnummer as housenumber', 'city.bezeichner as city', 'country.bezeichner as country', 'tx_sgestatecore_domain_model_immobilie.objektnr_extern as objektnr_extern')
            ->from('tx_sgestatecore_domain_model_immobilie')
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_ort', 'city', $queryBuilder->expr()->eq('city.uid', $queryBuilder->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objekt_ort')))
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_land', 'country', $queryBuilder->expr()->eq('country.uid', $queryBuilder->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objekt_land')));
        if ($renewAll === false) {
            $queryBuilder->orWhere($queryBuilder->expr()->eq('objekt_breitengrad', $queryBuilder->createNamedParameter(0)));
            $queryBuilder->orWhere($queryBuilder->expr()->eq('objekt_laengengrad', $queryBuilder->createNamedParameter(0)));
        }
        $result = $queryBuilder->executeQuery();

        $log = [];

        if ($result->rowCount() > 0) {
            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_immobilie');
            $rows = $result->fetchAllAssociative();
            foreach ($rows as $row) {
                $coordinates = $this->geocodeAddress($row['country'], $row['city'], $row['zip'], $row['street'], $row['housenumber'], $row['objektnr_extern'], $log);
                if ($coordinates !== null && $row['uid'] > 0 && $coordinates['longitude'] > 0 && $coordinates['latitude'] > 0) {
                    $updateArray = [
                        'objekt_laengengrad' => $coordinates['longitude'],
                        'objekt_breitengrad' => $coordinates['latitude']
                    ];
                    $connection->update(
                        'tx_sgestatecore_domain_model_immobilie',
                        $updateArray,
                        ['uid' => $row['uid']]
                    );
                }
            }
        }

        $logFilename = Environment::getVarPath() . '/log/sg_estate_core_geocode.log';
        // keep log file smaller than 10mb
        if (file_exists($logFilename) && filesize($logFilename) > 10485760) {
            $logFileHandle = fopen($logFilename, 'wb');
        } else {
            $logFileHandle = fopen($logFilename, 'ab');
        }
        fwrite($logFileHandle, implode("\n", $log));
        fclose($logFileHandle);
    }

    /**
     * @throws ExtensionConfigurationPathDoesNotExistException
     * @throws ExtensionConfigurationExtensionNotConfiguredException
     * @throws Exception
     * @throws JsonException
     */
    private function geocodeAddress($country, $city, $zip, $street, $housenumber, $objectNumber, &$log): ?array
    {
        $logDate = date('Y-m-d H:i:s') . ' ';
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $configuration = $configurationManager->getConfiguration(
            ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        );
        $backendConfiguration = $this->extensionConfiguration->get('sg_estate_core');
        $settings = $configuration['module.']['tx_sgestatecore.']['settings.'];
        if ($country === '') {
            $country = $this->standardCountry;
        }
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_geocode_cache');
        $result = $queryBuilder
            ->select('longitude', 'latitude')
            ->from('tx_sgestatecore_geocode_cache')
            ->where($queryBuilder->expr()->eq('zip', $queryBuilder->createNamedParameter($zip)))
            ->andWhere($queryBuilder->expr()->eq('street', $queryBuilder->createNamedParameter($street)))
            ->andWhere($queryBuilder->expr()->eq('housenumber', $queryBuilder->createNamedParameter($housenumber)))->andWhere($queryBuilder->expr()->eq('country', $queryBuilder->createNamedParameter($country)))->executeQuery();
        if ($result->rowCount() > 0) {
            $resultItem = $result->fetchAllAssociative();
            $coordinates['longitude'] = !empty($resultItem['longitude']) ? $resultItem['longitude'] : 0;
            $coordinates['latitude'] = !empty($resultItem['latitude']) ? $resultItem['latitude'] : 0;
            $log[] = $logDate . 'object "' . $objectNumber . '": found coordinates for ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country . ' in cache: ' . implode(', ', $coordinates);
            return $coordinates;
        }
        $coordinates    = null;
        $provider       = $settings['geocoding.']['provider'];
        if(empty($provider)) {
            $provider   = 'google';
        }
        $address =
            $this->convertMutation($country) .
            ', ' . $zip . ' ' .
            $this->convertMutation($city) .
            ', ' .
            $this->convertMutation($street) .
            (
                !empty($housenumber)
                    ? ' ' . $housenumber
                    : ''
            );
        if($provider === 'google') {
            $delay = 0;
            $base_url = 'https://maps.googleapis.com/maps/api/geocode/xml';
            $geocode_pending = true;
            $apikey = $backendConfiguration['geocodingApiKey'] !== '' ? $backendConfiguration['geocodingApiKey'] : $settings['keys.']['googleapi'];
            if(!empty($apikey)) {
                while ($geocode_pending) {
                    $request_url = $base_url . '?address=' . urlencode($address) . '&key=' . $apikey;
                    $xml = simplexml_load_string(file_get_contents($request_url)) or die('url not loading');
                    $status = $xml->status;
                    if (strcmp($status, 'OK') === 0) {
                        $geocode_pending = false;
                        $coordinates = [
                            'latitude' => (float)$xml->result->geometry->location->lat,
                            'longitude' => (float)$xml->result->geometry->location->lng
                        ];
                        $insertData = [
                            'zip' => $zip,
                            'street' => $street,
                            'housenumber' => $housenumber,
                            'country' => $country,
                            'longitude' => (float)$xml->result->geometry->location->lng,
                            'latitude' => (float)$xml->result->geometry->location->lat,
                        ];
                        $log[] = $logDate . 'object "' . $objectNumber . '": geocoded: ' . implode(', ', $insertData);
                        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_geocode_cache');
                        $connection->insert(
                            'tx_sgestatecore_geocode_cache',
                            $insertData
                        );
                    } elseif (strcmp($status, 'OVER_QUERY_LIMIT') === 0) {
                        $delay += 100000;
                        $log[] = $logDate . '[WARNING] received over query limit... (provider: google) wait for ' . $delay . ' microseconds...';
                    } else {
                        $geocode_pending = false;
                        $log[] = $logDate . '[ERROR] object "' . $objectNumber . '": could not geocode (provider: google): ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country;
                    }
                    usleep($delay);
                }
            } else {
                $log[] = $logDate . '[ERROR] no key for provider (google)';
            }
        }
        if($provider === 'apple') {
            $apikey = $settings['keys.']['appleapi'];
            if(!empty($apikey)) {
                $output = json_decode(shell_exec('curl -s -H "Authorization: Bearer ' . $apikey . '" "https://maps-api.apple.com/v1/token"'), false, 512, JSON_THROW_ON_ERROR);
                $auth = trim($output->accessToken);
                if(!empty($auth)) {
                    $output = json_decode(shell_exec('curl -s -H "Authorization: Bearer ' . $auth . '" "https://maps-api.apple.com/v1/geocode?q=' . urlencode($address) . '"'), false, 512, JSON_THROW_ON_ERROR);
                    if($output) {
                        $coordinates = [
                            'latitude' => (float)$output->results[0]->coordinate->latitude,
                            'longitude' => (float)$output->results[0]->coordinate->longitude
                        ];
                        if($coordinates['latitude'] > 0 && $coordinates['longitude'] > 0) {
                            $insertData = [
                                'zip' => $zip,
                                'street' => $street,
                                'housenumber' => $housenumber,
                                'country' => $country,
                                'longitude' => $coordinates['longitude'],
                                'latitude' => $coordinates['latitude'],
                            ];
                            $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_geocode_cache');
                            $connection->insert('tx_sgestatecore_geocode_cache', $insertData);
                        } else {
                            $log[] = $logDate . '[ERROR] object "' . $objectNumber . '": could not geocode by provider (apple), lat or lng is 0, query: ' . urlencode($address) . ', address: ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country;
                        }
                    } else {
                        $log[] = $logDate . '[ERROR] object "' . $objectNumber . '": could not geocode by provider (apple), query: ' . urlencode($address) . ', address: ' . $zip . ', ' . $street . ', ' . $housenumber . ', ' . $country;
                    }
                } else {
                    $log[] = $logDate . '[ERROR] unpossible to authorize for provider (apple)';
                }
            } else {
                $log[] = $logDate . '[ERROR] no key for provider (apple)';
            }
        }
        return $coordinates;
    }

    /**
     * @param array $recipients
     * @param $sender
     * @param $subject
     * @param $templateName
     * @param array $bcc
     * @param array $variables
     * @param array $attachments
     *  [
     *      0 => [
     *          'type' => 'fromPath',
     *          'path' => '/path/to/attachment'
     *          'filename' => 'filenameInEmail.txt'
     *          ],
     *      1 => [
     *          'type' => 'content',
     *          'content' => 'iamthecontent'
     *          'filename' => 'filenameInEmail.txt'
     *          ]
     *  ]
     *
     * @param string $format
     * @return bool
     * @throws TransportExceptionInterface
     * @noinspection PhpUnused
     */
    public function sendTemplateEmail(array $recipients, $sender, $subject, $templateName, array $bcc = [], array $variables = [], array $attachments = [], string $format = 'html'): bool
    {
        $mailer = GeneralUtility::makeInstance(MailerInterface::class);
        if (!($mailer instanceof MailerInterface)) {
            return false;
        }
        $message = new FluidEmail();

        if (GeneralUtility::validEmail($sender)) {
            $message->from($sender);
        } else {
            return false;
        }

        // Add Recipients
        $recipients = array_filter($recipients);
        if (count($recipients) > 0) {
            foreach ($recipients as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addTo($email);
                }
            }
        } else {
            return false;
        }

        if (count($bcc) > 0) {
            foreach ($bcc as $email) {
                if (GeneralUtility::validEmail($email)) {
                    $message->addBcc($email);
                }
            }
        }

        if ($subject !== '') {
            $message->subject($subject);
        } else {
            return false;
        }

        // text/plain and text/html is legacy format from
        // constant editor sg_estate_base pre 10.11.2
        // may be removed in 11.0.0
        if (in_array($format, ['text', 'text/plain'])) {
            $message->format('plain');
        } elseif (in_array($format, ['html', 'text/html'])) {
            $message->format('html');
        }

        $message->assignMultiple($variables);
        $message->setTemplate($templateName . '.' . $format);

        foreach ($attachments as $attachment) {
            switch ($attachment['type']) {
                case 'fromPath':
                    $message->attachFromPath($attachment['path'], $attachment['filename']);
                    break;
                case 'content':
                    $message->attach($attachment['content'], $attachment['filename']);
                    break;
            }
        }

        $mailer->send($message);

        return true;
    }

    /**
     * @throws Exception|JsonException
     */
    public function countRealties(): bool
    {
        // reset
        $connection = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_ort');
        $connection->update('tx_sgestatecore_domain_model_ort')->set('anzahl_immobilien', null)->executeStatement();

        $connection = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_stadtteil');
        $connection->update('tx_sgestatecore_domain_model_stadtteil')->set('anzahl_immobilien', null)->executeStatement();
        // eof reset

        $countsCities = [];
        $countsDistricts = [];

        $queryBuilderCities = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilderCities
            ->select(
                'tx_sgestatecore_domain_model_immobilie.uid',
                'city.uid as city',
                'objektart.kuerzel as objektartkuerzel'
            )
            ->from('tx_sgestatecore_domain_model_immobilie')
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_ort', 'city', $queryBuilderCities->expr()->eq('city.uid', $queryBuilderCities->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objekt_ort')))
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_objektart', 'objektart', $queryBuilderCities->expr()->eq('objektart.uid', $queryBuilderCities->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objektart')))
            ->where($queryBuilderCities->expr()->eq('tx_sgestatecore_domain_model_immobilie.deleted', $queryBuilderCities->createNamedParameter(0)))
            ->andWhere($queryBuilderCities->expr()->eq('tx_sgestatecore_domain_model_immobilie.hidden', $queryBuilderCities->createNamedParameter(0)))
        ;

        $resultCities = $queryBuilderCities->executeQuery();
        if ($resultCities->rowCount() > 0) {
            $rows = $resultCities->fetchAllAssociative();

            foreach ($rows as $row) {
                if (isset($countsCities[$row['city']][$row['objektartkuerzel']])) {
                    $countsCities[$row['city']][$row['objektartkuerzel']]++;
                } else {
                    $countsCities[$row['city']][$row['objektartkuerzel']] = 1;
                }

                if (isset($countsCities[$row['city']]['_GESAMT'])) {
                    $countsCities[$row['city']]['_GESAMT']++;
                } else {
                    $countsCities[$row['city']]['_GESAMT'] = 1;
                }
            }
        }

        $queryBuilderDistricts = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilderDistricts
            ->select(
                'tx_sgestatecore_domain_model_immobilie.uid',
                'district.uid as district',
                'objektart.kuerzel as objektartkuerzel'
            )
            ->from('tx_sgestatecore_domain_model_immobilie')
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_stadtteil', 'district', $queryBuilderDistricts->expr()->eq('district.uid', $queryBuilderDistricts->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objekt_stadtteil')))
            ->leftJoin('tx_sgestatecore_domain_model_immobilie', 'tx_sgestatecore_domain_model_objektart', 'objektart', $queryBuilderDistricts->expr()->eq('objektart.uid', $queryBuilderDistricts->quoteIdentifier('tx_sgestatecore_domain_model_immobilie.objektart')))
            ->where($queryBuilderDistricts->expr()->eq('tx_sgestatecore_domain_model_immobilie.deleted', $queryBuilderDistricts->createNamedParameter(0)))
            ->andWhere($queryBuilderDistricts->expr()->eq('tx_sgestatecore_domain_model_immobilie.hidden', $queryBuilderDistricts->createNamedParameter(0)))
        ;

        $resultDistricts = $queryBuilderDistricts->executeQuery();
        if ($resultDistricts->rowCount() > 0) {
            $rows = $resultDistricts->fetchAllAssociative();
            foreach ($rows as $row) {
                if (isset($countsDistricts[$row['district']][$row['objektartkuerzel']])) {
                    $countsDistricts[$row['district']][$row['objektartkuerzel']]++;
                } else {
                    $countsDistricts[$row['district']][$row['objektartkuerzel']] = 1;
                }
                if (isset($countsDistricts[$row['district']]['_GESAMT'])) {
                    $countsDistricts[$row['district']]['_GESAMT']++;
                } else {
                    $countsDistricts[$row['district']]['_GESAMT'] = 1;
                }
            }
        }

        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_ort');
        foreach ($countsCities as $cityUid => $countsCity) {
            $updateArray = [
                'anzahl_immobilien' => json_encode($countsCity, JSON_THROW_ON_ERROR)
            ];
            $connection->update(
                'tx_sgestatecore_domain_model_ort',
                $updateArray,
                ['uid' => $cityUid]
            );
        }

        $connection = $this->connectionPool->getConnectionForTable('tx_sgestatecore_domain_model_stadtteil');
        foreach ($countsDistricts as $districtUid => $countsDistrict) {
            $updateArray = [
                'anzahl_immobilien' => json_encode($countsDistrict, JSON_THROW_ON_ERROR)
            ];
            $connection->update(
                'tx_sgestatecore_domain_model_stadtteil',
                $updateArray,
                ['uid' => $districtUid]
            );
        }
        return true;
    }

    /**
     * @param $value
     * @return string|array
     */
    private function convertMutation($value): string|array
    {
        return str_replace(
            array('ü', 'Ü', 'ö', 'Ö', 'ä', 'Ä', 'ß'),
            array('ue', 'Ue', 'oe', 'Oe', 'ae', 'Ae', 'ss'),
            $value
        );
    }
}
