<?php
namespace SG\SgEstateCore\Domain\Repository;

use SG\SgEstateCore\Domain\Model\Badausstattung;
use SG\SgEstateCore\Domain\Model\Immobilie;
use SG\SgEstateCore\Domain\Model\Kuechenausstattung;
use SG\SgEstateCore\Domain\Model\Ort;
use SG\SgEstateCore\Domain\Model\Region;
use SG\SgEstateCore\Enum\SortingForFilterEnum;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use SG\SgEstateCore\Enum\NullableBooleanEnum;

/**
 * The repository for Immobilien
 */
class ImmobilieRepository extends AbstractEstateRepository
{

    /**
     * @var ConnectionPool
     */
    protected $connectionPool;

    public function __construct(
        \TYPO3\CMS\Core\Database\ConnectionPool $connectionPool,
        \SG\SgEstateCore\Domain\Repository\KuechenausstattungRepository $kuechenausstattungRepository,
        \SG\SgEstateCore\Domain\Repository\BadausstattungRepository $badausstattungRepository,
        \TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper $dataMapper
    ){
        parent::__construct();
        $this->connectionPool = $connectionPool;
        $this->kuechenausstattungRepository = $kuechenausstattungRepository;
        $this->badausstattungRepository = $badausstattungRepository;
        $this->dataMapper = $dataMapper;
    }

    /**
     * @var KuechenausstattungRepository
     */
    protected $kuechenausstattungRepository;

    /**
     * @var BadausstattungRepository
     */
    protected $badausstattungRepository;

    /**
     * @var DataMapper
     */
    protected $dataMapper;

    public function initializeObject(): void
    {
        $query = $this->createQuery();
        $querySettings = $query->getQuerySettings();
        $querySettings->setRespectStoragePage(false);
        $querySettings->setRespectSysLanguage(false);
        $this->setDefaultQuerySettings($querySettings);
    }

    protected function getSettings() {
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $configuration = $configurationManager->getConfiguration(
            \TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        );
        return $configuration['plugin.']['tx_sgestatecore.']['settings.'];
    }

    public function findAllWithFilterData(){
        $sql = "immobilie.wohnflaeche AS size, 
                            immobilie.anzahl_zimmer AS rooms,
                            immobilie.nettokaltmiete AS rent,
                            immobilie.objekt_stadtteil AS district_id,
                            immobilie.objekt_ort AS city_id,
                            ort.bezeichner AS ort_bezeichner,
                            stadtteil.bezeichner AS stadtteil_bezeichner
                        FROM tx_sgestatecore_domain_model_immobilie as immobilie
                        LEFT JOIN tx_sgestatecore_domain_model_ort as ort ON immobilie.objekt_ort = ort.uid
                        LEFT JOIN tx_sgestatecore_domain_model_stadtteil as stadtteil ON immobilie.objekt_stadtteil = stadtteil.uid
                        WHERE immobilie.deleted = '0' AND immobilie.hidden = '0' ORDER BY stadtteil.bezeichner ASC";
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)->getConnectionByName('Default');
        $allRealties = $connection->createQueryBuilder()->addSelectLiteral($sql)->executeQuery()->fetchAllAssociative();
        $realtiesReturn = [];
        $citiesReturn = [];
        $districtsReturn = [];
        foreach ($allRealties as $realty) {
            $realtiesReturn[] = [
                'vermietbareFlaeche' => $realty['size'],
                'anzahlZimmer' => $realty['rooms'],
                'kaltmiete' => $realty['rent'],
                'objektOrt' => 'city-' . $realty['city_id'],
                'objektStadtteil' => 'district-' . $realty['district_id'],
            ];
            if (!isset($districtsReturn[$realty['district_id']])){
                if (isset($districtsReturn[$realty['city_id']])){
                    $citiesReturn[$realty['city_id']] = [
                        'uid' => $realty['city_id'],
                        'id' => 'city-'.$realty['city_id'],
                        'name' => $realty['ort_bezeichner'],
                        'type' => 'city',
                        'selector' => '',
                    ];
                }
                //Stadtteil
                $districtsReturn[$realty['district_id']] = [
                    'uid' => $realty['district_id'],
                    'id' => 'district-'.$realty['district_id'],
                    'name' => $realty['ort_bezeichner'].' - '.$realty['stadtteil_bezeichner'],
                    'type' => 'district',
                    'selector' => '',
                ];
            }
        }
        return ['realties'=>$realtiesReturn, 'cities'=>array_merge($citiesReturn, $districtsReturn)];
    }

    public function findAllWithFilter(\SG\SgEstateCore\Domain\Model\Filter $filter, $limit = null, $excludeUids = null)
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');

        if (is_array($filter->getUids()) && count($filter->getUids()) > 0) {
            $queryBuilder->where($queryBuilder->expr()->in('immobilie.uid', $filter->getUids()));
        } else {
            if ($filter->getRaeumeAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_zimmer', $filter->getRaeumeAb()));
            }
            if ($filter->getRaeumeBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('anzahl_zimmer', $filter->getRaeumeBis()));
            }

            if ($filter->getFlaecheAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('wohnflaeche', $filter->getFlaecheAb()));
            }
            if ($filter->getFlaecheBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('wohnflaeche', $filter->getFlaecheBis()));
            }

            if ($filter->getKaltmieteAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('kaltmiete', $filter->getKaltmieteAb()));
            }
            if ($filter->getKaltmieteBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('kaltmiete', $filter->getKaltmieteBis()));
            }

            if ($filter->getRegionen()->count() > 0) {
                $arrayOfOrtUids = [];
                foreach ($filter->getRegionen() as $region) {
                    /**
                     * @var $region Region
                     */
                    foreach ($region->getOrte() as $ort) {
                        /**
                         * @var $ort Ort
                         */
                        if (!in_array($ort->getUid(), $arrayOfOrtUids)) {
                            $arrayOfOrtUids[] = $ort->getUid();
                        }
                    }
                }
                if (count($arrayOfOrtUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_ort', $arrayOfOrtUids));
                }
            }

            if ($filter->getOrte()->count() > 0) {
                $arrayOfOrtUids = [];
                foreach ($filter->getOrte() as $ort) {
                    $arrayOfOrtUids[] = $ort->getUid();
                }
                if (count($arrayOfOrtUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_ort', $arrayOfOrtUids));
                }
            }

            if ($filter->getStadtteile()->count() > 0) {
                $arrayOfStadtteilUids = [];
                foreach ($filter->getStadtteile() as $stadtteil) {
                    $arrayOfStadtteilUids[] = $stadtteil->getUid();
                }
                if (count($arrayOfStadtteilUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_stadtteil', $arrayOfStadtteilUids));
                }
            }

            if ($filter->getObjektarten()->count() > 0) {
                $arrayOfObjektartUids = [];
                foreach ($filter->getObjektarten() as $objektart) {
                    $arrayOfObjektartUids[] = $objektart->getUid();
                }
                if (count($arrayOfObjektartUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objektart', $arrayOfObjektartUids));
                }
            }

            if ($filter->getObjektarttypen()->count() > 0) {
                $arrayOfObjektarttypUids = [];
                foreach ($filter->getObjektarttypen() as $objektarttyp) {
                    $arrayOfObjektarttypUids[] = $objektarttyp->getUid();
                }
                if (count($arrayOfObjektarttypUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objektarttyp', $arrayOfObjektarttypUids));
                }
            }

            if ($filter->getBalkon() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_balkone', 1));
            } elseif ($filter->getBalkon() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('anzahl_balkone', 0));
            }

            if ($filter->getGarten() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('gartennutzung', 1));
            } elseif ($filter->getGarten() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('gartennutzung', -1));
            }

            if ($filter->getKeller() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('unterkellert', 1));
            } elseif ($filter->getKeller() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('unterkellert', -1));
            }

            if ($filter->getGaestewc() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_separate_wc', 1));
            } elseif ($filter->getGaestewc() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('anzahl_separate_wc', 0));
            }

            if ($filter->getHaustiere() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('haustiere', 1));
            } elseif ($filter->getHaustiere() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('haustiere', -1));
            }

            if ($filter->getWbs() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('wbs_erforderlich', 1));
            } elseif ($filter->getWbs() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('wbs_erforderlich', -1));
            }

            if ($filter->getBarrierefrei() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('barrierefrei', 1));
            } elseif ($filter->getBarrierefrei() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('barrierefrei', -1));
            }

            if ($filter->getLeichterZugang() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('rollstuhlgerecht', 1));
            } elseif ($filter->getLeichterZugang() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('rollstuhlgerecht', -1));
            }

            if ($filter->getBadModernisiert() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('bad_modernisiert', 1));
            } elseif ($filter->getLeichterZugang() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('bad_modernisiert', -1));
            }

            if ($filter->getAufzug() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('aufzug_personen', 1));
            } elseif ($filter->getAufzug() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('aufzug_personen', -1));
            }

            // Restrict to Import Ids if set
            if (is_array($filter->getImportIds()) && count($filter->getImportIds()) >= 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->in('import_number', $filter->getImportIds()));
            }

            if ($filter->getHeizungsarten()->count() > 0) {
                $arrayOfHeizungsartUids = [];
                foreach ($filter->getHeizungsarten() as $heizungsart) {
                    $arrayOfHeizungsartUids[] = $heizungsart->getUid();
                }
                if (count($arrayOfHeizungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_heizungsart_mm',
                        'immobilie_heizungsart_mm',
                        'immobilie.uid = immobilie_heizungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_heizungsart_mm.uid_foreign', $arrayOfHeizungsartUids));
                }
            }

            if ($filter->getBefeuerungsarten()->count() > 0) {
                $arrayOfBefeuerungsartUids = [];
                foreach ($filter->getBefeuerungsarten() as $befeuerungsart) {
                    $arrayOfBefeuerungsartUids[] = $befeuerungsart->getUid();
                }
                if (count($arrayOfBefeuerungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_befeuerungsart_mm',
                        'immobilie_befeuerungsart_mm',
                        'immobilie.uid = immobilie_befeuerungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_befeuerungsart_mm.uid_foreign', $arrayOfBefeuerungsartUids));
                }
            }

            if ($filter->getBodenbelaege()->count() > 0) {
                $arrayOfBodenbelagUids = [];
                foreach ($filter->getBodenbelaege() as $bodenbelag) {
                    $arrayOfBodenbelagUids[] = $bodenbelag->getUid();
                }
                if (count($arrayOfBodenbelagUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_bodenbelag_mm',
                        'immobilie_bodenbelag_mm',
                        'immobilie.uid = immobilie_bodenbelag_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_bodenbelag_mm.uid_foreign', $arrayOfBodenbelagUids));
                }
            }

            if ($filter->getVermarktungsarten()->count() > 0) {
                $arrayOfVermarktungsartUids = [];
                foreach ($filter->getVermarktungsarten() as $vermarktungsart) {
                    $arrayOfVermarktungsartUids[] = $vermarktungsart->getUid();
                }
                if (count($arrayOfVermarktungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_vermarktungsart_mm',
                        'immobilie_vermarktungsart_mm',
                        'immobilie.uid = immobilie_vermarktungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_vermarktungsart_mm.uid_foreign', $arrayOfVermarktungsartUids));
                }
            }

            if ($filter->getKampagnen()->count() > 0) {
                $arrayOfKampagneUids = [];
                foreach ($filter->getKampagnen() as $kampagne) {
                    $arrayOfKampagneUids[] = $kampagne->getUid();
                }
                if (count($arrayOfKampagneUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kampagne_mm',
                        'immobilie_kampagne_mm',
                        'immobilie.uid = immobilie_kampagne_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_kampagne_mm.uid_foreign', $arrayOfKampagneUids));
                }
            }

            if ($filter->getEinbaukueche() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $kuechenausstattungEBK = $this->kuechenausstattungRepository->findOneBy(['kuerzel' => 'EBK']);
                if ($kuechenausstattungEBK instanceof Kuechenausstattung) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kuechenausstattung_mm',
                        'immobilie_kuechenausstattung_mm',
                        'immobilie.uid = immobilie_kuechenausstattung_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_kuechenausstattung_mm.uid_foreign', $kuechenausstattungEBK->getUid()));
                }
            } elseif ($filter->getEinbaukueche() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $kuechenausstattungEBK = $this->kuechenausstattungRepository->findOneBy(['kuerzel' => 'EBK']);
                if ($kuechenausstattungEBK instanceof Kuechenausstattung) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kuechenausstattung_mm',
                        'immobilie_kuechenausstattung_mm',
                        'immobilie.uid = immobilie_kuechenausstattung_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_kuechenausstattung_mm.uid_foreign', $kuechenausstattungEBK->getUid()));
                }
            }

            if ((
                $filter->getBadMitWanne() === NullableBooleanEnum::POSITIVE_AUSWAHL)
                || ($filter->getBadMitWanne() === NullableBooleanEnum::NEGATIVE_AUSWAHL)
                || ($filter->getBadMitDusche() === NullableBooleanEnum::POSITIVE_AUSWAHL)
                || ($filter->getBadMitDusche() === NullableBooleanEnum::NEGATIVE_AUSWAHL)
                || ($filter->getBadMitFenster() === NullableBooleanEnum::POSITIVE_AUSWAHL)
                || ($filter->getBadMitFenster() === NullableBooleanEnum::NEGATIVE_AUSWAHL)
            ) {
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_immobilie_badausstattung_mm',
                    'immobilie_badausstattung_mm',
                    'immobilie.uid = immobilie_badausstattung_mm.uid_local'
                );
            }

            if ($filter->getBadMitWanne() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $badausstattungWanne = $this->badausstattungRepository->findOneBy(['kuerzel' => 'WANNE']);
                if ($badausstattungWanne instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungWanne->getUid()));
                }
            } elseif ($filter->getBadMitWanne() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $badausstattungWanne = $this->badausstattungRepository->findOneBy(['kuerzel' => 'WANNE']);
                if ($badausstattungWanne instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungWanne->getUid()));
                }
            }

            if ($filter->getBadMitDusche() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $badausstattungDusche = $this->badausstattungRepository->findOneBy(['kuerzel' => 'DUSCHE']);
                if ($badausstattungDusche instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungDusche->getUid()));
                }
            } elseif ($filter->getBadMitDusche() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $badausstattungDusche = $this->badausstattungRepository->findOneBy(['kuerzel' => 'DUSCHE']);
                if ($badausstattungDusche instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungDusche->getUid()));
                }
            }

            if ($filter->getBadMitFenster() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $badausstattungFenster = $this->badausstattungRepository->findOneBy(['kuerzel' => 'FENSTER']);
                if ($badausstattungFenster instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungFenster->getUid()));
                }
            } elseif ($filter->getBadMitFenster() === NullableBooleanEnum::NEGATIVE_AUSWAHL) {
                $badausstattungFenster = $this->badausstattungRepository->findOneBy(['kuerzel' => 'FENSTER']);
                if ($badausstattungFenster instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungFenster->getUid()));
                }
            }

            if ($filter->getLongitude() > 0 && $filter->getLatitude() > 0 && $filter->getRadius() > 0) {
                $queryBuilder->addSelectLiteral('6371 * ACOS(COS(RADIANS(`immobilie`.`objekt_breitengrad`)) * COS(RADIANS(' . $filter->getLatitude() . ')) * COS(RADIANS(' . $filter->getLongitude() . ') - RADIANS(`immobilie`.`objekt_laengengrad`)) + SIN(RADIANS(`immobilie`.`objekt_breitengrad`)) * SIN(RADIANS(' . $filter->getLatitude() . '))) AS distance');
                $queryBuilder->having($queryBuilder->expr()->lte('distance', $filter->getRadius()));
            }

            if ($filter->getPersonsFrom() > 0 && $filter->getPersonsTo() > 0) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('minimale_personenanzahl', $filter->getPersonsFrom()));
                $queryBuilder->andWhere($queryBuilder->expr()->lte('maximale_personenanzahl', $filter->getPersonsTo()));
            } elseif ($filter->getPersonsFrom() > 0) {
                $queryBuilder->andWhere($filter->getPersonsFrom() . ' BETWEEN `immobilie`.`minimale_personenanzahl` AND `immobilie`.`maximale_personenanzahl`');
            } elseif ($filter->getPersonsTo() > 0) {
                $queryBuilder->andWhere($filter->getPersonsTo() . ' BETWEEN `immobilie`.`minimale_personenanzahl` AND `immobilie`.`maximale_personenanzahl`');
            }
        }

        if ($limit != null) {
            $queryBuilder->setMaxResults($limit);
        }

        $settings = $this->getSettings();
        if($settings['defaultSort']) {
            $filter->setSortierung($settings['defaultSort']);
        }

        switch ($filter->getSortierung()) {
            case SortingForFilterEnum::KALTMIETE_ASC:
                // Kaltmiete (niedriegste zuerst)
                $queryBuilder->orderBy('kaltmiete', 'ASC');
                break;
            case SortingForFilterEnum::KALTMIETE_DESC:
                // Kaltmiete (höchste zuerst)
                $queryBuilder->orderBy('kaltmiete', 'DESC');
                break;
            case SortingForFilterEnum::FLAECHE_ASC:
                // Fläche (niedriegste zuerst)
                $queryBuilder->orderBy('wohnflaeche', 'ASC');
                break;
            case SortingForFilterEnum::FLAECHE_DESC:
                // Fläche (höchste zuerst)
                $queryBuilder->orderBy('wohnflaeche', 'DESC');
                break;
            case SortingForFilterEnum::ZIMMER_ASC:
                // Zimmer (aufsteigend)
                $queryBuilder->orderBy('anzahl_zimmer', 'ASC');
                break;
            case SortingForFilterEnum::ZIMMER_DESC:
                // Zimmer (absteigend)
                $queryBuilder->orderBy('anzahl_zimmer', 'DESC');
                break;
            case SortingForFilterEnum::STADTTEIL:
                // Stadtteil alphabetisch
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_domain_model_stadtteil',
                    'stadtteil',
                    'immobilie.objekt_stadtteil = stadtteil.uid'
                );
                $queryBuilder->orderBy('stadtteil.bezeichner');
                break;
            case SortingForFilterEnum::ETAGE_ASC:
                // Etage aufsteigend
                $queryBuilder->orderBy('objekt_etage', 'ASC');
                break;
            case SortingForFilterEnum::ETAGE_DESC:
                // Etage absteigend
                $queryBuilder->orderBy('objekt_etage', 'DESC');
                break;
            case SortingForFilterEnum::STADT:
                // Stadt
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_domain_model_ort',
                    'ort',
                    'immobilie.objekt_ort = ort.uid'
                );
                $queryBuilder->orderBy('ort.bezeichner');
                break;
            case SortingForFilterEnum::TSTAMP_ASC:
                $queryBuilder->orderBy('tstamp', 'ASC');
                break;
            case SortingForFilterEnum::TSTAMP_DESC:
                $queryBuilder->orderBy('tstamp', 'DESC');
                break;
            case SortingForFilterEnum::DISTANCE:
                // sort by distance
                $queryBuilder->orderBy('distance', 'ASC');
                break;
        }



        $objects = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->executeQuery()->fetchAllAssociative()
        );


        return $objects;
    }

    /**
     * @param $urlIdentifier
     *
     * @return Immobilie|null
     */
    public function findOneByUrlIdentifier($urlIdentifier)
    {
        $realty = $this->findOneByExternalNumber($urlIdentifier);
        if ($realty instanceof Immobilie) {
            return $realty;
        }
        //return $this->findOneByExternalNumber(preg_replace(['/-/'], '/', $urlIdentifier));
        return $this->findOneByExternalNumber(str_replace('///', '/-/', preg_replace(['/-/'], '/', $urlIdentifier)));
    }

    /**
     * @param $externalNumber
     *
     * @return Immobilie|null
     */
    public function findOneByExternalNumber($externalNumber)
    {
        // Get the raw table row
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');
        $queryBuilder->where(
            $queryBuilder->expr()->eq('objektnr_extern', $queryBuilder->createNamedParameter($externalNumber))
        );

        // now map the result set to the immobilie model
        $result = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->executeQuery()->fetchAllAssociative()
        );

        // check whether we have an immobilie object and return first
        if (count($result) > 0 && $result[0] instanceof Immobilie) {
            return $result[0];
        }
        return null;
    }

    /**
     * @param int $uid
     *
     * @return Immobilie|null
     */
    public function findByUid($uid)
    {
        // Get the raw table row
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');
        $queryBuilder->where(
            $queryBuilder->expr()->eq('uid', (int) $uid )
        );

        // now map the result set to the immobilie model
        $result = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->executeQuery()->fetchAllAssociative()
        );

        // check whether we have an immobilie object and return first
        if (count($result) > 0 && $result[0] instanceof Immobilie) {
            return $result[0];
        }
        return null;
    }


    /**
     * Finds all Realty matching a like
     *
     * @param string $objektnrExtern
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllByObjektnrExtern($objektnrExtern)
    {
        /**
         * @var QueryInterface $query
         */
        $query = $this->createQuery();

        $query->matching(
            $query->like('objektnrExtern', '%' . $objektnrExtern . '%')
        );

        return $query->execute();
    }

    /**
     * @param array $arrayOfImportIds
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllForImportIds(array $arrayOfImportIds)
    {
        /**
         * @var QueryInterface $query
         */
        $query = $this->createQuery();

        $query->matching(
            $query->in('importNumber', $arrayOfImportIds)
        );

        return $query->execute();
    }
}
