<?php
namespace SG\SgEstateCore\ViewHelpers;

use SG\SgEstateCore\Domain\Model\Ort;
use SG\SgEstateCore\Domain\Model\Region;
use SG\SgEstateCore\Domain\Model\Stadtteil;
use TYPO3Fluid\Fluid\Core\ViewHelper\AbstractViewHelper;

class PlacesListViewHelper extends AbstractViewHelper {
    public function initializeArguments()
    {
        parent::initializeArguments();
        $this->registerArgument('regions', 'array', 'Regions to include', false, []);
        $this->registerArgument('cities', 'array', 'Cities to include', false, []);
        $this->registerArgument('classPrefix', 'string', 'Optional class prefix override', false, 'js-sge-place');
        $this->registerArgument('jsonEncoded', 'bool', 'Return value json encoded?', false, false);
    }

    /**
     * Returns a combined array or place information to use e.g.
     * in a megaselect or any other tool that uses this data structure
     */
    public function render(){
        $r = [];
        foreach( $this->arguments['regions'] as $region ) {
            if( $region instanceof Region ) {
                $r[] = [
                    'uid' => $region->getUid(),
                    'id' => 'region-'.$region->getUid(),
                    'name' => $region->getBezeichner(),
                    'type' => 'region',
                    'selector' => $this->getClassListForModel($region->getOrte(), $this->arguments[ 'classPrefix' ], 'city'),
                ];
            }
        }
        foreach( $this->arguments['cities'] as $key => $city ) {
            if( $city instanceof Ort ) {
                $r[] = [
                    'uid' => $city->getUid(),
                    'id' => 'city-'.$city->getUid(),
                    'name' => $city->getBezeichner(),
                    'type' => 'city',
                    'selector' => $this->getClassListForModel([$city], $this->arguments[ 'classPrefix' ], 'city'),
                ];
                foreach( $city->getStadtteile() as $district ) {
                    if( $district instanceof Stadtteil ) {
                        $r[] = [
                            'uid' => $district->getUid(),
                            'id' => 'district-'.$district->getUid(),
                            'name' => $city->getBezeichner().' - '.$district->getBezeichner(),
                            'type' => 'district',
                            'selector' => $this->getClassListForModel([$district], $this->arguments[ 'classPrefix' ], 'district'),
                        ];
                    }
                }
            }
            // we have a different structure for nested cities in landingpage,
            // so we should adapt to that
            else if ( array_key_exists('bezeichner', $city ) ) {
                $r[] = [
                    'uid' => $key,
                    'id' => 'city-'.$key,
                    'name' => $city['bezeichner'],
                    'type' => 'city',
                    'selector' => $this->getClassListForArray([$key => $city], $this->arguments[ 'classPrefix' ], 'city'),
                ];
                if( array_key_exists('districts', $city ) ) {
                    foreach( $city['districts'] as $district ) {
                        if( array_key_exists('bezeichner', $district ) ) {
                            $r[] = [
                                'uid' => $district['uid'],
                                'id' => 'district-'.$district['uid'],
                                'name' => $city['bezeichner'].' - '.$district['bezeichner'],
                                'type' => 'district',
                                'selector' => $this->getClassListForArray([$district], $this->arguments[ 'classPrefix' ], 'district'),
                            ];
                        }
                    }
                }
            }
        }
        if($this->arguments['jsonEncoded'] === true)
            return json_encode($r);
        $tm = $ar = [];
        foreach($r as $k=>$v) if($v['type'] == 'district') $ar[] = $r[$k]; else $tmp[] = $r[$k];
        usort($ar, function ($a, $b) {return strcmp($a["name"], $b["name"]);});
        return array_merge($tm, $ar);
    }

    /**
     * Returns a class selector list for all given items with the given class prefix and type
     *
     * @param        $items
     * @param        $classPrefix
     * @param string $type
     *
     * @return string
     */
    private function getClassListForModel( $items, $classPrefix, $type = 'city' ): string
    {
        $r = '';
        $firstIteration = true;
        foreach( $items as $key => $item ) {
            if( !$firstIteration ) {
                $r .= ', ';
            }
            $r .= '.'.$classPrefix.'-'.$type.'-'.$item->getUid();
            $firstIteration = false;
        }
        return $r;
    }

    /**
     * Returns a class selector list for all given items with class prefix and type by predefined array structure
     *
     * @param        $items
     * @param        $classPrefix
     * @param string $type
     *
     * @return string
     */
    private function getClassListForArray( $items, $classPrefix, $type = 'city' ): string
    {
        $r = '';
        $firstIteration = true;
        foreach( $items as $key => $item ) {
            // in some cases we have an uid array field and in some we use the array key for
            // the uid field. so we try to figure out which type we have and use the correct field then.
            $uid = 0;
            if( array_key_exists('uid', $item)) {
                $uid = (int) $item['uid'];
            }
            else {
                $uid = $key;
            }

            if( $uid > 0 ) {
                if( !$firstIteration ) {
                    $r .= ', ';
                }

                $r .= '.'.$classPrefix.'-'.$type.'-'.$uid;
                $firstIteration = false;
            }
        }
        return $r;
    }

}
