<?php
namespace SG\SgEstateCore\Domain\Repository;

use SG\SgEstateCore\Domain\Model\Badausstattung;
use SG\SgEstateCore\Domain\Model\Immobilie;
use SG\SgEstateCore\Domain\Model\Kuechenausstattung;
use SG\SgEstateCore\Domain\Model\Ort;
use SG\SgEstateCore\Domain\Model\Region;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Extbase\Persistence\Generic\Mapper\DataMapper;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;

/**
 * The repository for Immobilien
 */
class ImmobilieRepository extends \TYPO3\CMS\Extbase\Persistence\Repository
{

    /**
     * @var ConnectionPool
     */
    private $connectionPool;

    public function injectConnectionPool(ConnectionPool $connectionPool)
    {
        $this->connectionPool = $connectionPool;
    }

    /**
     * @var KuechenausstattungRepository
     */
    private $kuechenausstattungRepository;

    /**
     * @param KuechenausstattungRepository $kuechenausstattungRepository
     */
    public function injectKuechenausstattungRepository(KuechenausstattungRepository $kuechenausstattungRepository)
    {
        $this->kuechenausstattungRepository = $kuechenausstattungRepository;
    }

    /**
     * @var BadausstattungRepository
     */
    private $badausstattungRepository;

    /**
     * @param BadausstattungRepository $badausstattungRepository
     */
    public function injectBadausstattungRepository(BadausstattungRepository $badausstattungRepository)
    {
        $this->badausstattungRepository = $badausstattungRepository;
    }

    /**
     * @var DataMapper
     */
    private $dataMapper;

    /**
     * @param DataMapper $dataMapper
     */
    public function injectDataMapper(DataMapper $dataMapper)
    {
        $this->dataMapper = $dataMapper;
    }

    public function initializeObject()
    {
        /** @var $querySettings \TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings */
        $querySettings = $this->objectManager->get('TYPO3\\CMS\\Extbase\\Persistence\\Generic\\Typo3QuerySettings');
        $querySettings->setRespectStoragePage(false);
        $querySettings->setRespectSysLanguage(false);
        $this->setDefaultQuerySettings($querySettings);
    }

    protected function getSettings() {
        $configurationManager = GeneralUtility::makeInstance(ConfigurationManager::class);
        $configuration = $configurationManager->getConfiguration(
            \TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT
        );
        return $configuration['plugin.']['tx_sgestatecore.']['settings.'];
    }

    public function findAllWithFilter(\SG\SgEstateCore\Domain\Model\Filter $filter, $limit = null, $excludeUids = null)
    {
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');

        if (is_array($filter->getUids())) {
            $queryBuilder->where($queryBuilder->expr()->in('uid', $filter->getUids()));
        } else {
            if ($filter->getRaeumeAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_zimmer', $filter->getRaeumeAb()));
            }
            if ($filter->getRaeumeBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('anzahl_zimmer', $filter->getRaeumeBis()));
            }

            if ($filter->getFlaecheAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('wohnflaeche', $filter->getFlaecheAb()));
            }
            if ($filter->getFlaecheBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('wohnflaeche', $filter->getFlaecheBis()));
            }

            if ($filter->getKaltmieteAb() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('kaltmiete', $filter->getKaltmieteAb()));
            }
            if ($filter->getKaltmieteBis() !== null) {
                $queryBuilder->andWhere($queryBuilder->expr()->lte('kaltmiete', $filter->getKaltmieteBis()));
            }

            if ($filter->getRegionen()->count() > 0) {
                $arrayOfOrtUids = [];
                foreach ($filter->getRegionen() as $region) {
                    /**
                     * @var $region Region
                     */
                    foreach ($region->getOrte() as $ort) {
                        /**
                         * @var $ort Ort
                         */
                        if (!in_array($ort->getUid(), $arrayOfOrtUids)) {
                            array_push($arrayOfOrtUids, $ort->getUid());
                        }
                    }
                }
                if (count($arrayOfOrtUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_ort', $arrayOfOrtUids));
                }
            }

            if ($filter->getOrte()->count() > 0) {
                $arrayOfOrtUids = [];
                foreach ($filter->getOrte() as $ort) {
                    array_push($arrayOfOrtUids, $ort->getUid());
                }
                if (count($arrayOfOrtUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_ort', $arrayOfOrtUids));
                }
            }

            if ($filter->getStadtteile()->count() > 0) {
                $arrayOfStadtteilUids = [];
                foreach ($filter->getStadtteile() as $stadtteil) {
                    array_push($arrayOfStadtteilUids, $stadtteil->getUid());
                }
                if (count($arrayOfStadtteilUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objekt_stadtteil', $arrayOfStadtteilUids));
                }
            }

            if ($filter->getObjektarten()->count() > 0) {
                $arrayOfObjektartUids = [];
                foreach ($filter->getObjektarten() as $objektart) {
                    array_push($arrayOfObjektartUids, $objektart->getUid());
                }
                if (count($arrayOfObjektartUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objektart', $arrayOfObjektartUids));
                }
            }

            if ($filter->getObjektarttypen()->count() > 0) {
                $arrayOfObjektarttypUids = [];
                foreach ($filter->getObjektarttypen() as $objektarttyp) {
                    array_push($arrayOfObjektarttypUids, $objektarttyp->getUid());
                }
                if (count($arrayOfObjektarttypUids) > 0) {
                    $queryBuilder->andWhere($queryBuilder->expr()->in('objektarttyp', $arrayOfObjektarttypUids));
                }
            }

            if ($filter->getBalkon() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_balkone', 1));
            } elseif ($filter->getBalkon() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('anzahl_balkone', 0));
            }

            if ($filter->getGarten() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('gartennutzung', 1));
            } elseif ($filter->getGarten() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('gartennutzung', -1));
            }

            if ($filter->getKeller() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('unterkellert', 1));
            } elseif ($filter->getKeller() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('unterkellert', -1));
            }

            if ($filter->getGaestewc() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('anzahl_separate_wc', 1));
            } elseif ($filter->getGaestewc() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('anzahl_separate_wc', 0));
            }

            if ($filter->getHaustiere() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('haustiere', 1));
            } elseif ($filter->getHaustiere() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('haustiere', -1));
            }

            if ($filter->getWbs() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('wbs_erforderlich', 1));
            } elseif ($filter->getWbs() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('wbs_erforderlich', -1));
            }

            if ($filter->getBarrierefrei() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('barrierefrei', 1));
            } elseif ($filter->getBarrierefrei() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('barrierefrei', -1));
            }

            if ($filter->getLeichterZugang() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('rollstuhlgerecht', 1));
            } elseif ($filter->getLeichterZugang() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('rollstuhlgerecht', -1));
            }

            if ($filter->getBadModernisiert() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('bad_modernisiert', 1));
            } elseif ($filter->getLeichterZugang() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('bad_modernisiert', -1));
            }

            if ($filter->getAufzug() == 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('aufzug_personen', 1));
            } elseif ($filter->getAufzug() == -1) {
                $queryBuilder->andWhere($queryBuilder->expr()->eq('aufzug_personen', -1));
            }

            // Restrict to Import Ids if set
            if (is_array($filter->getImportIds()) && count($filter->getImportIds()) >= 1) {
                $queryBuilder->andWhere($queryBuilder->expr()->in('import_number', $filter->getImportIds()));
            }

            if ($filter->getHeizungsarten()->count() > 0) {
                $arrayOfHeizungsartUids = [];
                foreach ($filter->getHeizungsarten() as $heizungsart) {
                    array_push($arrayOfHeizungsartUids, $heizungsart->getUid());
                }
                if (count($arrayOfHeizungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_heizungsart_mm',
                        'immobilie_heizungsart_mm',
                        'immobilie.uid = immobilie_heizungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_heizungsart_mm.uid_foreign', $arrayOfHeizungsartUids));
                }
            }

            if ($filter->getBefeuerungsarten()->count() > 0) {
                $arrayOfBefeuerungsartUids = [];
                foreach ($filter->getBefeuerungsarten() as $befeuerungsart) {
                    array_push($arrayOfBefeuerungsartUids, $befeuerungsart->getUid());
                }
                if (count($arrayOfBefeuerungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_befeuerungsart_mm',
                        'immobilie_befeuerungsart_mm',
                        'immobilie.uid = immobilie_befeuerungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_befeuerungsart_mm.uid_foreign', $arrayOfBefeuerungsartUids));
                }
            }

            if ($filter->getBodenbelaege()->count() > 0) {
                $arrayOfBodenbelagUids = [];
                foreach ($filter->getBodenbelaege() as $bodenbelag) {
                    array_push($arrayOfBodenbelagUids, $bodenbelag->getUid());
                }
                if (count($arrayOfBodenbelagUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_bodenbelag_mm',
                        'immobilie_bodenbelag_mm',
                        'immobilie.uid = immobilie_bodenbelag_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_bodenbelag_mm.uid_foreign', $arrayOfBodenbelagUids));
                }
            }

            if ($filter->getVermarktungsarten()->count() > 0) {
                $arrayOfVermarktungsartUids = [];
                foreach ($filter->getVermarktungsarten() as $vermarktungsart) {
                    array_push($arrayOfVermarktungsartUids, $vermarktungsart->getUid());
                }
                if (count($arrayOfVermarktungsartUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_vermarktungsart_mm',
                        'immobilie_vermarktungsart_mm',
                        'immobilie.uid = immobilie_vermarktungsart_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_vermarktungsart_mm.uid_foreign', $arrayOfVermarktungsartUids));
                }
            }

            if ($filter->getKampagnen()->count() > 0) {
                $arrayOfKampagneUids = [];
                foreach ($filter->getKampagnen() as $kampagne) {
                    array_push($arrayOfKampagneUids, $kampagne->getUid());
                }
                if (count($arrayOfKampagneUids) > 0) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kampagne_mm',
                        'immobilie_kampagne_mm',
                        'immobilie.uid = immobilie_kampagne_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->in('immobilie_kampagne_mm.uid_foreign', $arrayOfKampagneUids));
                }
            }

            if ($filter->getEinbaukueche() == 1) {
                $kuechenausstattungEBK = $this->kuechenausstattungRepository->findOneByKuerzel('EBK');
                if ($kuechenausstattungEBK instanceof Kuechenausstattung) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kuechenausstattung_mm',
                        'immobilie_kuechenausstattung_mm',
                        'immobilie.uid = immobilie_kuechenausstattung_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_kuechenausstattung_mm.uid_foreign', $kuechenausstattungEBK->getUid()));
                }
            } elseif ($filter->getEinbaukueche() == -1) {
                $kuechenausstattungEBK = $this->kuechenausstattungRepository->findOneByKuerzel('EBK');
                if ($kuechenausstattungEBK instanceof Kuechenausstattung) {
                    $queryBuilder->leftJoin(
                        'immobilie',
                        'tx_sgestatecore_immobilie_kuechenausstattung_mm',
                        'immobilie_kuechenausstattung_mm',
                        'immobilie.uid = immobilie_kuechenausstattung_mm.uid_local'
                    );
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_kuechenausstattung_mm.uid_foreign', $kuechenausstattungEBK->getUid()));
                }
            }

            if (($filter->getBadMitWanne() == 1) || ($filter->getBadMitWanne() == -1) || ($filter->getBadMitDusche() == 1) || ($filter->getBadMitDusche() == -1) || ($filter->getBadMitFenster() == 1) || ($filter->getBadMitFenster() == -1)) {
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_immobilie_badausstattung_mm',
                    'immobilie_badausstattung_mm',
                    'immobilie.uid = immobilie_badausstattung_mm.uid_local'
                );
            }

            if ($filter->getBadMitWanne() == 1) {
                $badausstattungWanne = $this->badausstattungRepository->findOneByKuerzel('WANNE');
                if ($badausstattungWanne instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungWanne->getUid()));
                }
            } elseif ($filter->getBadMitWanne() == -1) {
                $badausstattungWanne = $this->badausstattungRepository->findOneByKuerzel('WANNE');
                if ($badausstattungWanne instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungWanne->getUid()));
                }
            }

            if ($filter->getBadMitDusche() == 1) {
                $badausstattungDusche = $this->badausstattungRepository->findOneByKuerzel('DUSCHE');
                if ($badausstattungDusche instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungDusche->getUid()));
                }
            } elseif ($filter->getBadMitDusche() == -1) {
                $badausstattungDusche = $this->badausstattungRepository->findOneByKuerzel('DUSCHE');
                if ($badausstattungDusche instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungDusche->getUid()));
                }
            }

            if ($filter->getBadMitFenster() == 1) {
                $badausstattungFenster = $this->badausstattungRepository->findOneByKuerzel('FENSTER');
                if ($badausstattungFenster instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->eq('immobilie_badausstattung_mm.uid_foreign', $badausstattungFenster->getUid()));
                }
            } elseif ($filter->getBadMitFenster() == -1) {
                $badausstattungFenster = $this->badausstattungRepository->findOneByKuerzel('FENSTER');
                if ($badausstattungFenster instanceof Badausstattung) {
                    $queryBuilder->andWhere($queryBuilder->expr()->neq('immobilie_badausstattung_mm.uid_foreign', $badausstattungFenster->getUid()));
                }
            }

            if ($filter->getLongitude() > 0 && $filter->getLatitude() > 0 && $filter->getRadius() > 0) {
                $queryBuilder->addSelectLiteral('6371 * ACOS(COS(RADIANS(`immobilie`.`objekt_breitengrad`)) * COS(RADIANS(' . $filter->getLatitude() . ')) * COS(RADIANS(' . $filter->getLongitude() . ') - RADIANS(`immobilie`.`objekt_laengengrad`)) + SIN(RADIANS(`immobilie`.`objekt_breitengrad`)) * SIN(RADIANS(' . $filter->getLatitude() . '))) AS distance');
                $queryBuilder->having($queryBuilder->expr()->lte('distance', $filter->getRadius()));
            }

            if ($filter->getPersonsFrom() > 0 && $filter->getPersonsTo() > 0) {
                $queryBuilder->andWhere($queryBuilder->expr()->gte('minimale_personenanzahl', $filter->getPersonsFrom()));
                $queryBuilder->andWhere($queryBuilder->expr()->lte('maximale_personenanzahl', $filter->getPersonsTo()));
            } elseif ($filter->getPersonsFrom() > 0) {
                $queryBuilder->andWhere($filter->getPersonsFrom() . ' BETWEEN `immobilie`.`minimale_personenanzahl` AND `immobilie`.`maximale_personenanzahl`');
            } elseif ($filter->getPersonsTo() > 0) {
                $queryBuilder->andWhere($filter->getPersonsTo() . ' BETWEEN `immobilie`.`minimale_personenanzahl` AND `immobilie`.`maximale_personenanzahl`');
            }
        }

        if ($limit != null) {
            $queryBuilder->setMaxResults($limit);
        }

        $settings = $this->getSettings();
        if($settings['defaultSort']) {
            $filter->setSortierung($settings['defaultSort']);
        }

        switch ($filter->getSortierung()) {
            case 0:
                // Kaltmiete (niedriegste zuerst)
                $queryBuilder->orderBy('kaltmiete', 'ASC');
                break;
            case 1:
                // Kaltmiete (höchste zuerst)
                $queryBuilder->orderBy('kaltmiete', 'DESC');
                break;
            case 2:
                // Fläche (niedriegste zuerst)
                $queryBuilder->orderBy('wohnflaeche', 'ASC');
                break;
            case 3:
                // Fläche (höchste zuerst)
                $queryBuilder->orderBy('wohnflaeche', 'DESC');
                break;
            case 4:
                // Zimmer (aufsteigend)
                $queryBuilder->orderBy('anzahl_zimmer', 'ASC');
                break;
            case 5:
                // Zimmer (absteigend)
                $queryBuilder->orderBy('anzahl_zimmer', 'DESC');
                break;
            case 6:
                // Stadtteil alphabetisch
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_domain_model_stadtteil',
                    'stadtteil',
                    'immobilie.objekt_stadtteil = stadtteil.uid'
                );
                $queryBuilder->orderBy('stadtteil.bezeichner');
                break;
            case 7:
                // Etage aufsteigend
                $queryBuilder->orderBy('objekt_etage', 'ASC');
                break;
            case 8:
                // Etage absteigend
                $queryBuilder->orderBy('objekt_etage', 'DESC');
                break;
            case 9:
                // Stadt
                $queryBuilder->leftJoin(
                    'immobilie',
                    'tx_sgestatecore_domain_model_ort',
                    'ort',
                    'immobilie.objekt_ort = ort.uid'
                );
                $queryBuilder->orderBy('ort.bezeichner');
                break;
            case 10:
                $queryBuilder->orderBy('tstamp', 'ASC');
                break;
            case 11:
                $queryBuilder->orderBy('tstamp', 'DESC');
                break;
        }

        $objects = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->execute()->fetchAll()
        );

        return $objects;
    }

    /**
     * @param $urlIdentifier
     *
     * @return Immobilie|null
     */
    public function findOneByUrlIdentifier($urlIdentifier)
    {
        $realty = $this->findOneByExternalNumber($urlIdentifier);
        if ($realty instanceof Immobilie) {
            return $realty;
        }
        return $this->findOneByExternalNumber(preg_replace(['/-/'], '/', $urlIdentifier));
    }

    /**
     * @param $externalNumber
     *
     * @return Immobilie|null
     */
    public function findOneByExternalNumber($externalNumber)
    {
        // Get the raw table row
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');
        $queryBuilder->where(
            $queryBuilder->expr()->eq('objektnr_extern', $queryBuilder->createNamedParameter($externalNumber))
        );

        // now map the result set to the immobilie model
        $result = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->execute()->fetchAll()
        );

        // check whether we have an immobilie object and return first
        if (count($result) > 0 && $result[0] instanceof Immobilie) {
            return $result[0];
        }
        return null;
    }

    /**
     * @param int $uid
     *
     * @return Immobilie|null
     */
    public function findByUid($uid)
    {
        // Get the raw table row
        $queryBuilder = $this->connectionPool->getQueryBuilderForTable('tx_sgestatecore_domain_model_immobilie');
        $queryBuilder
            ->select('immobilie.*')
            ->from('tx_sgestatecore_domain_model_immobilie', 'immobilie');
        $queryBuilder->where(
            $queryBuilder->expr()->eq('uid', (int) $uid )
        );

        // now map the result set to the immobilie model
        $result = $this->dataMapper->map(
            Immobilie::class,
            $queryBuilder->execute()->fetchAll()
        );

        // check whether we have an immobilie object and return first
        if (count($result) > 0 && $result[0] instanceof Immobilie) {
            return $result[0];
        }
        return null;
    }


    /**
     * Finds all Realty matching a like
     *
     * @param string $objektnrExtern
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllByObjektnrExtern($objektnrExtern)
    {
        /**
         * @var QueryInterface $query
         */
        $query = $this->createQuery();

        $query->matching(
            $query->like('objektnrExtern', '%' . $objektnrExtern . '%')
        );

        return $query->execute();
    }

    /**
     * @param array $arrayOfImportIds
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllForImportIds(array $arrayOfImportIds)
    {
        /**
         * @var QueryInterface $query
         */
        $query = $this->createQuery();

        $query->matching(
            $query->in('importNumber', $arrayOfImportIds)
        );

        return $query->execute();
    }
}
