<?php
namespace SG\SgEstateBase\Controller;

use Doctrine\DBAL\Exception;
use JsonException;
use Psr\Http\Message\ResponseInterface;
use SG\SgEstateCore\Domain\Model\Badausstattung;
use SG\SgEstateCore\Domain\Model\Filter;
use SG\SgEstateCore\Domain\Model\Heizungsart;
use SG\SgEstateCore\Domain\Model\Immobilie;
use SG\SgEstateCore\Domain\Model\Kontaktperson;
use SG\SgEstateCore\Domain\Model\Objektart;
use SG\SgEstateCore\Domain\Model\Objektarttyp;
use SG\SgEstateCore\Domain\Model\Ort;
use SG\SgEstateCore\Domain\Model\Region;
use SG\SgEstateCore\Domain\Model\Stadtteil;
use SG\SgEstateCore\Domain\Repository\HeizungsartRepository;
use SG\SgEstateCore\Domain\Repository\ImmobilieRepository;
use SG\SgEstateCore\Domain\Repository\ObjektartRepository;
use SG\SgEstateCore\Domain\Repository\ObjektarttypRepository;
use SG\SgEstateCore\Domain\Repository\OrtRepository;
use SG\SgEstateCore\Domain\Repository\RegionRepository;
use SG\SgEstateCore\Domain\Repository\StadtteilRepository;
use SG\SgEstateCore\Enum\NullableBooleanEnum;
use SG\SgEstateCore\Util\Services;
use SG\SgEstateCore\Util\SessionHandler;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Context\Exception\AspectNotFoundException;
use TYPO3\CMS\Core\Core\Environment;
use TYPO3\CMS\Core\Database\ConnectionPool;
use TYPO3\CMS\Core\Session\Backend\Exception\SessionNotCreatedException;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Mvc\Exception\NoSuchArgumentException;
use TYPO3\CMS\Extbase\Mvc\Web\Routing\UriBuilder;
use TYPO3\CMS\Extbase\Persistence\ObjectStorage;
use TYPO3\CMS\Extbase\Service\ImageService;
use TYPO3\CMS\Extbase\Utility\LocalizationUtility;
use TYPO3\CMS\Fluid\View\StandaloneView;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;

class BaseController extends ActionController
{
    protected Filter $filter;

    protected bool $noSearchParameters = false;

    public function __construct(
        protected FrontendInterface      $sgEstateCoreCache,
        protected RegionRepository       $regionRepository,
        protected OrtRepository          $ortRepository,
        protected StadtteilRepository    $stadtteilRepository,
        protected ImmobilieRepository    $immobilieRepository,
        protected ObjektartRepository    $objektartRepository,
        protected ObjektarttypRepository $objektarttypRepository,
        protected HeizungsartRepository  $heizungsartRepository,
        protected SessionHandler         $sessionHandler,
        protected Services               $sgEstateCoreServices,
        protected ImageService           $imageService,
        protected UriBuilder             $uriBuilder
    ) {}

    /**
     * Initializing is called before every Action
     * @throws SessionNotCreatedException
     */
    public function initializeAction(): void
    {
        $this->setFilter();
    }

    /********************************************
     * Actions for MVC
     *********************************************/

    /**
     * Sets Property Filter from Request Arguments
     * @throws SessionNotCreatedException
     */
    protected function setFilter(): void
    {
        if ($this->request->hasArgument('filter')) {
            /**
             * @var $filterValues array
             */
            $this->filter = new Filter();

            $filterValues = $this->request->getArgument('filter');

            // extract city and districts from groupedCitiesAndDistricts if set
            // and override filter value city and add filter values district
            if (isset($filterValues['groupedCitiesAndDistricts']) && $filterValues['groupedCitiesAndDistricts'] !== '') {
                $split = explode('_', $filterValues['groupedCitiesAndDistricts']);
                if (isset($split[0]) && is_numeric($split[0])) {
                    $filterValues['city'] = $split[0];
                }
                if (isset($split[1]) && is_numeric($split[1])) {
                    $filterValues['district'][] = $split[1];
                }
            }

            if (isset($filterValues['regions']) && is_array($filterValues['regions']) && count($filterValues['regions']) > 0) {
                foreach ($filterValues['regions'] as $regionUid) {
                    $region = $this->regionRepository->findByUid($regionUid);
                    if ($region instanceof Region) {
                        $this->filter->addRegion($region);
                    }
                }
            }
            if (isset($filterValues['region']) && is_numeric($filterValues['region'])) {
                $region = $this->regionRepository->findByUid($filterValues['region']);
                if ($region instanceof Region) {
                    $this->filter->setRegion($region);
                }
            }

            if (is_numeric($filterValues['city'])) {
                $ort = $this->ortRepository->findByUid($filterValues['city']);
                if ($ort instanceof Ort) {
                    $this->filter->setOrt($ort);
                }
            }
            if (isset($filterValues['cities']) && is_array($filterValues['cities']) && count($filterValues['cities']) > 0) {
                foreach ($filterValues['cities'] as $cityUid) {
                    $city = $this->ortRepository->findByUid($cityUid);
                    if ($city instanceof Ort) {
                        $this->filter->addOrt($city);
                    }
                }
            }

            if (is_numeric($filterValues['district'])) {
                $stadtteil = $this->stadtteilRepository->findByUid($filterValues['district']);
                if ($stadtteil instanceof Stadtteil) {
                    $this->filter->addStadtteil($stadtteil);
                }
            } elseif (is_array($filterValues['district'])) {
                foreach ($filterValues['district'] as $stadtteilUid) {
                    $stadtteil = $this->stadtteilRepository->findByUid($stadtteilUid);
                    if ($stadtteil instanceof Stadtteil) {
                        $this->filter->addStadtteil($stadtteil);
                    }
                }
            }

            $this->filter->setRaeumeAb(is_numeric($filterValues['roomMin']) ? (float)$filterValues['roomMin'] : null);
            $this->filter->setRaeumeBis(is_numeric($filterValues['roomMax']) ? (float)$filterValues['roomMax'] : null);

            $this->filter->setFlaecheAb(is_numeric($filterValues['sizeMin']) ? (float)$filterValues['sizeMin'] : null);
            $this->filter->setFlaecheBis(is_numeric($filterValues['sizeMax']) ? (float)$filterValues['sizeMax'] : null);

            $this->filter->setKaltmieteAb(is_numeric($filterValues['rentMin']) ? (float)$filterValues['rentMin'] : null);
            $this->filter->setKaltmieteBis(is_numeric($filterValues['rentMax']) ? (float)$filterValues['rentMax'] : null);

            if (isset($filterValues['kindOfObjects'])) {
                $objektarten = new ObjectStorage();
                foreach ($filterValues['kindOfObjects'] as $uid) {
                    $objektart = $this->objektartRepository->findByUid($uid);
                    if ($objektart instanceof Objektart) {
                        $objektarten->attach($objektart);
                    }
                }
                if ($objektarten->count() > 0) {
                    $this->filter->setObjektarten($objektarten);
                }
            }

            if (isset($filterValues['objektartTypen'])&&is_array($filterValues['objektartTypen'])) {
                $objektartTypen = new ObjectStorage();
                foreach ($filterValues['objektartTypen'] as $item) {
                    $objektartTyp = $this->objektarttypRepository->findOneByKuerzel($item);
                    if ($objektartTyp instanceof Objektarttyp) {
                        $objektartTypen->attach($objektartTyp);
                    }
                }
                if ($objektartTypen->count() > 0) {
                    $this->filter->setObjektarttypen($objektartTypen);
                }
            }

            if (isset($filterValues['heizungsarten'])&&is_array($filterValues['heizungsarten'])) {
                $heizungsarten = new ObjectStorage();
                foreach ($filterValues['heizungsarten'] as $item) {
                    $heizungsart = $this->heizungsartRepository->findOneByKuerzel($item);
                    if ($heizungsart instanceof Heizungsart) {
                        $heizungsarten->attach($heizungsart);
                    }
                }
                if ($heizungsarten->count() > 0) {
                    $this->filter->setHeizungsarten($heizungsarten);
                }
            }
            $this->filter->setAufzug(is_numeric($filterValues['aufzug']) ? (int)$filterValues['aufzug'] : 0);
            $this->filter->setBarrierefrei(is_numeric($filterValues['barrierefrei']) ? (int)$filterValues['barrierefrei'] : 0);

            if (isset($filterValues['latitude'], $filterValues['longitude']) && is_numeric($filterValues['latitude']) && is_numeric($filterValues['longitude']) && $filterValues['latitude'] !== 0 && $filterValues['longitude'] !== 0) {
                $this->filter->setLatitude($filterValues['latitude']);
                $this->filter->setLongitude($filterValues['longitude']);
                if (isset($filterValues['radius']) && is_numeric($filterValues['radius']) && $filterValues['radius'] > 0) {
                    $this->filter->setRadius($filterValues['radius']);
                } else {
                    $this->filter->setRadius($this->settings['preset']['search']['radius']);
                }
            }
            if (isset($filterValues['addressString']) && $filterValues['addressString'] !== '') {
                $this->filter->setAddressString($filterValues['addressString']);
            } else {
                $this->filter->setAddressString('');
            }

            if (isset($filterValues['personsFrom']) && is_numeric($filterValues['personsFrom']) && $filterValues['personsFrom'] > 0) {
                $this->filter->setPersonsFrom($filterValues['personsFrom']);
            } else if($this->settings['preset']['search']['persons'] > 0){
                $this->filter->setPersonsFrom($this->settings['preset']['search']['persons']);
            }else{
                $this->filter->setPersonsFrom(0);
            }
            if (isset($filterValues['personsTo']) && is_numeric($filterValues['personsTo']) && $filterValues['personsTo'] > 0) {
                $this->filter->setPersonsTo($filterValues['personsTo']);
            } else {
                $this->filter->setPersonsTo(0);
            }
            if (isset($filterValues['persons']) && $filterValues['persons'] !== '') {
                $this->filter->setPersonsString($filterValues['persons']);

                $tempPersonsArray = explode('-', $filterValues['persons']);
                if (isset($tempPersonsArray[0]) && is_numeric($tempPersonsArray[0]) && $tempPersonsArray[0] > 0) {
                    $this->filter->setPersonsFrom($tempPersonsArray[0]);
                } else {
                    $this->filter->setPersonsFrom(0);
                }
                if (isset($tempPersonsArray[1]) && is_numeric($tempPersonsArray[1]) && $tempPersonsArray[1] > 0) {
                    $this->filter->setPersonsTo($tempPersonsArray[1]);
                } else {
                    $this->filter->setPersonsTo(0);
                }
            }else if($this->settings['preset']['search']['persons'] > 0){
                $this->filter->setPersonsString($this->settings['preset']['search']['persons']);
            }
        } else {
            // No Data for Filter in Request
            $filter = $this->sessionHandler->restoreFromSession('filter');
            if ($filter instanceof Filter) {
                $this->filter = $filter;
            } else {
                $this->filter = new Filter();
            }
            $this->noSearchParameters = true;
        }

        $this->sessionHandler->writeToSession($this->filter, 'filter');
    }

    /**
     * Action for Standard Search
     * @return ResponseInterface
     * @throws AspectNotFoundException
     * @throws Exception
     * @throws JsonException
     */
    public function searchAction(): ResponseInterface
    {
        $shows = '0';
        if($this->settings['show']) {
            if (is_array($this->settings['show'])) {
                $shows = implode('', $this->settings['show']);
            }
            if (is_string($this->settings['show'])) {
                $shows = $this->settings['show'];
            }
            if (is_int($this->settings['show'])) {
                $shows = (string)$this->settings['show'];
            }
        }
        $cacheIdentifier = 'search_' . $this->settings['search']['layout'] . '_' . $shows . '_' . $this->settings['search']['layout'] . '_L' . $this->getLanguageUid();

        if (($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
            if ($this->settings['preset']['search']['filterData'] !== 0) {
                $kindOfObjectUids = explode(',', $this->settings['search']['kindOfObjects']);
                foreach ($kindOfObjectUids as $kindOfObjectUid) {
                    $kindOfObject = $this->objektartRepository->findByUid($kindOfObjectUid);
                    if ($kindOfObject instanceof Objektart) {
                        $this->filter->addObjektart($kindOfObject);
                    }
                }

                if (isset($this->settings['importids']) && $this->settings['importids'] !== '') {
                    $arrayOfImportIds = explode(',', $this->settings['importids']);
                    if (is_array($arrayOfImportIds) && count($arrayOfImportIds) >= 1) {
                        $this->filter->setImportIds($arrayOfImportIds);
                    } else {
                        $this->filter->setImportIds([]);
                    }
                } else {
                    $arrayOfImportIds = null;
                }

                if ((int)$this->settings['preset']['search']['useSlider'] === 1) {
                    // BEGIN: Assign Values for Slider-Configuration to View
                    $this->view->assign('sizeSliderData', $this->getSizeSliderData());
                    $this->view->assign('roomSliderData', $this->getRoomSliderData());
                    $this->view->assign('amountSliderData', $this->getAmountSliderData());
                    // END: Assign Values for Slider-Configuration to View
                }

                $this->view->assign('regionOptions', $this->regionRepository->getAllForFilter($arrayOfImportIds));
                $this->view->assign('districtOptions', $this->stadtteilRepository->getAllForFilter($this->settings['preset']['search']['districtFormat'], $arrayOfImportIds));
                $this->view->assign('cityOptions', $this->ortRepository->getAllForFilter($arrayOfImportIds));
                $this->view->assign('groupedCitiesAndDistricts', $this->ortRepository->getGroupedCitiesAndDistrictsForFilter($arrayOfImportIds));
                $this->view->assign('filter', $this->filter);
                $this->view->assign('cities', $this->ortRepository->findAll());


                $this->view->assign('aggregatedFilterData', json_encode($this->getAggregatedFilterData(), JSON_THROW_ON_ERROR));
                $this->view->assign('citiesAndDistrictsForFilter', json_encode($this->generateCityAndDistrictStructureForFilter(), JSON_THROW_ON_ERROR));

                if ((int)$this->settings['preset']['search']['simpleRealtyData'] === 1){
                    $returnArray = $this->immobilieRepository->findAllWithFilterData();
                    $realtiesModifited = $returnArray['realties'];
                    $this->view->assign('places', $returnArray['cities']);
                }
                else {
                    $realties = $this->immobilieRepository->findAllWithFilter($this->filter);
                    $this->view->assign('presets', $this->generateFilterPresetsFromResult($realties));

                    $this->view->assign('realties', $realties);
                    $realtiesModifited = array();
                    foreach ($realties as $k => $v) {
                        $realtiesModifited[$k]['anzahlZimmer'] = $v->getAnzahlZimmer();
                        $realtiesModifited[$k]['vermietbareFlaeche'] = $v->getVermietbareFlaeche();
                        $realtiesModifited[$k]['kaltmiete'] = $v->getKaltmiete();
                        $realtiesModifited[$k]['objektOrt'] = $v->getObjektOrt() === null ? '' : 'region-' . $v->getObjektOrt()->getUid();
                        $realtiesModifited[$k]['objektStadtteil'] = $v->getObjektStadtteil() === null ? '' : 'district-' . $v->getObjektStadtteil()->getUid();
                    }
                }
                $this->view->assign('realtiesModifited', $realtiesModifited);
            }

            $entry = $this->view->render();

            $tags = ['search'];
            $lifetime = $this->settings['defaultCacheLifetime'];

            // Save value in cache
            $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
        }
        return $this->htmlResponse($entry);
    }

    /**
     * Get the current language
     * @throws AspectNotFoundException
     */
    protected function getLanguageUid(): int
    {
        $context = GeneralUtility::makeInstance(Context::class);
        $sysLanguageUid = $context->getPropertyFromAspect('language', 'id');
        if (is_int($sysLanguageUid)) {
            return $sysLanguageUid;
        }
        return 0;
    }

    /**
     * Generate Array with Slider Data for the Size Slider
     * @todo: Read from Config
     */
    protected function getSizeSliderData(): array
    {
        return [
            'range' => $this->settings['preset']['slider']['size']['range'],
            'min'   => $this->settings['preset']['slider']['size']['min'],
            'max'   => $this->settings['preset']['slider']['size']['max'],
            'value' => $this->settings['preset']['slider']['size']['value']
        ];
    }

    /**
     * Generate Array with Slider Data for the Room Slider
     * @todo: Read from Config
     */
    protected function getRoomSliderData(): array
    {
        return [
            'range' => $this->settings['preset']['slider']['room']['range'],
            'min'   => $this->settings['preset']['slider']['room']['min'],
            'max'   => $this->settings['preset']['slider']['room']['max'],
            'value' => $this->settings['preset']['slider']['room']['value']
        ];
    }

    /**
     * Generate Array with Slider Data for the Amount Slider
     * @todo: Read from Config
     */
    protected function getAmountSliderData(): array
    {
        return [
            'range' => $this->settings['preset']['slider']['amount']['range'],
            'min'   => $this->settings['preset']['slider']['amount']['min'],
            'max'   => $this->settings['preset']['slider']['amount']['max'],
            'value' => $this->settings['preset']['slider']['amount']['value']
        ];
    }

    protected function getAggregatedFilterData(): array
    {
        return [
            'size' => !$this->filter->getFlaecheAb() ? $this->getSizeSliderData()['value'] : $this->filter->getFlaecheAb(),
            'rooms' => !$this->filter->getRaeumeAb() ? $this->getRoomSliderData()['value'] : $this->filter->getRaeumeAb(),
            'rent' => !$this->filter->getKaltmieteBis() ? $this->getAmountSliderData()['value'] : $this->filter->getKaltmieteBis()
        ];
    }

    /**
     * @throws Exception|JsonException
     */
    protected function generateCityAndDistrictStructureForFilter(): array
    {
        return $this->generateCityAndDistrictUniversal();
    }

    /**
     * @throws Exception|JsonException
     */
    protected function generateCityAndDistrictUniversal($type='array'): array
    {
        $cityAndDistrictStructure = [];
        $sql = "a.`uid`, 
        a.`bezeichner`, 
        a.`anzahl_immobilien`, 
        a.`fuer_suche_verwenden`, 
        a.`pid`,
        GROUP_CONCAT(CONCAT('{\"uid\":|', b.`uid`, '|, \"pid\":|', b.`pid`, '|, \"bezeichner\":|', b.`bezeichner`, '|, \"fuer_suche_verwenden\":|', b.`fuer_suche_verwenden`, '|}')) AS `stadtteile` 
        FROM `tx_sgestatecore_domain_model_stadtteil` AS b 
        INNER JOIN `tx_sgestatecore_domain_model_ort` AS a 
        ON b.`ort` = a.`uid` 
        GROUP BY b.`ort`";
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)?->getConnectionByName('Default');
        $cities = $connection->createQueryBuilder()->addSelectLiteral(str_replace('|', '"', $sql))->executeQuery()->fetchAllAssociative();
        foreach ($cities as $city) {
            $city_qtys  = is_null($city['anzahl_immobilien']) ? array() : json_decode($city['anzahl_immobilien'], true, 512, JSON_THROW_ON_ERROR);
            $city_qty   = array_key_exists('WOHNUNG', $city_qtys) ? $city_qtys['WOHNUNG'] : 0;
            $districts  = !empty($city['stadtteile']) ? json_decode('['.$city['stadtteile'].']', true, 512, JSON_THROW_ON_ERROR) : array();
            if($type !== 'array') {
                $newCity = GeneralUtility::makeInstance(Ort::class);
                $newCity->_setProperty('uid', $city['uid']);
                $newCity->_setProperty('pid', $city['pid']);
                $newCity->_setProperty('bezeichner', $city['bezeichner']);
                $newCity->_setProperty('fuer_suche_verwenden', $city['fuer_suche_verwenden']);
                $newCity->_setProperty('anzahlImmobilien', $city['anzahl_immobilien']);
                if (is_array($districts) && count($districts) > 0) {
                    $newDistricts = [];
                    foreach ($districts as $district) {
                        $newDistrict = GeneralUtility::makeInstance(Stadtteil::class);
                        $newDistrict->_setProperty('uid', $district['uid']);
                        $newDistrict->_setProperty('pid', $district['pid']);
                        $newDistrict->_setProperty('bezeichner', $district['bezeichner']);
                        $newDistrict->_setProperty('fuer_suche_verwenden', $district['fuer_suche_verwenden']);
                        $newDistricts[] = $newDistrict;
                    }
                    $newCity->_setProperty('stadtteile', $newDistricts);
                }
            } else {

                $newCity = [
                    'caption' => $city['bezeichner'],
                    'uid' => $city['uid'],
                    'qty' => $city_qty
                ];

                if (is_array($districts) && count($districts) > 0) {
                    $newDistricts = [];
                    foreach ($districts as $district) {
                        $newDistrict = [
                            'caption' => $district['bezeichner'],
                            'uid' => $district['uid'],
                            'qty' => $city_qty
                        ];
                        $newDistricts[] = $newDistrict;
                    }

                    $newCity['districts'] = $newDistricts;
                }
            }
            $cityAndDistrictStructure[] = $newCity;
        }

        return $cityAndDistrictStructure;
    }

    /**
     * @param $result
     * @return array
     */
    protected function generateFilterPresetsFromResult($result): array
    {
        $presets = array();

        // Initialize
        $cities = array();
        $allCities = array();
        $balcony = false;
        $lift = false;
        $wbs = false;
        $bathShower = false;
        $bathTub = false;
        $bathWindow = false;

        foreach ($result as $realty){
            /**
             * Identifiy those cities and districts that lead to valid results (take all realties locations)
             * @var $realty Immobilie
             */
            $ort = $realty->getObjektOrt();
            $cities[$ort->getUid()]['bezeichner'] = $ort->getBezeichner();
            $objektStadtteil = $realty->getObjektStadtteil();
            if ($objektStadtteil !== null){
                $cities[$ort->getUid()]['districts'][$objektStadtteil->getUid()] = array(
                    'uid' => $objektStadtteil->getUid(),
                    'bezeichner' => $objektStadtteil->getBezeichner()
                );
            }

            /**
             * Identify all cities and districts and create hierarchical tree
             */

            $tempAllCities = $this->ortRepository->findAll();
            $presets['tempAllCities'] = $tempAllCities;
            foreach( $tempAllCities as $tempCity )
            {
                if(($tempCity instanceof Ort) && $tempCity->getFuerSucheVerwenden() === true) {
                    $tempDistricts = $tempCity->getStadtteile();
                    $tempDistrictsCount = $tempDistricts?->count();
                    if( $tempDistrictsCount > 0 )
                    {
                        // now inspect all disricts and identify those we want to add to the presets
                        $districtsArray = array();
                        foreach( $tempDistricts as $tempDistrict )
                        {
                            if(($tempDistrict instanceof Stadtteil) && $tempDistrict->getFuerSucheVerwenden() === true) {
                                // choose this districts
                                $districtsArray[] = array(
                                    'uid' => $tempDistrict->getUid(),
                                    'bezeichner' => $tempDistrict->getBezeichner()
                                );
                            }
                        }
                        // valid city and districts combination found, add to "all cities array"
                        //// sort districts, pass reference
                        $this->sortDistrictsPresetArray( $districtsArray );
                        // append array
                        $allCities[ $tempCity->getUid() ] = array(
                            'bezeichner' => $tempCity->getBezeichner(),
                            'districts' => $districtsArray
                        );
                    }
                    else {
                        $allCities[ $tempCity->getUid() ] = array(
                            'bezeichner' => $tempCity->getBezeichner(),
                            'districts' => []
                        );
                    }
                }
            }

            // Balcony
            if ($realty->getAnzahlBalkone() > 0){$balcony = true;}

            // Lift
            if ($realty->getAufzugPersonen() === NullableBooleanEnum::POSITIVE_AUSWAHL){$lift = true;}

            // WBS
            if ($realty->getWbsErforderlich() === NullableBooleanEnum::POSITIVE_AUSWAHL){$wbs = true;}

            // Bath
            $bathShower = $realty->getHasBathShower();
            $bathTub = $realty->getHasBathTub();
            $bathWindow = $realty->getHasBathWindow();
        }

        // Build Return Variable
        $presets['cities'] = $cities;
        $presets['allCities'] = $allCities;
        $presets['allRegions'] = $this->regionRepository->findAll();
        $presets['balcony'] = $balcony;
        $presets['lift'] = $lift;
        $presets['wbs'] = $wbs;
        $presets['bathShower'] = $bathShower;
        $presets['bathTub'] = $bathTub;
        $presets['bathWindow'] = $bathWindow;

        return $presets;
    }

    /**
     * Sort a given district array that was optimized for preset use in template
     * in ascending order (bezeichner field)
     * @param array &$districts Array to sort
     */
    protected function sortDistrictsPresetArray(array &$districts ): void
    {
        $sorted = false;
        $districtsCount = count( $districts )-1;
        while( $sorted === false )
        {
            $changes = 0;
            for( $i = 0; $i < $districtsCount; $i++ )
            {
                if( $districts[$i]['bezeichner'] > $districts[$i+1]['bezeichner'] )
                {
                    // change position and increment counter
                    $h = $districts[$i];
                    $districts[$i] = $districts[$i+1];
                    $districts[$i+1] = $h;
                    $changes++;
                }
            }
            if( $changes === 0 )
            {
                $sorted = true;
            }
        }
    }

    /**
     * Action for Standard Searchbox
     */
    public function searchboxAction(): ResponseInterface
    {
        $kindOfObjectUids = explode(',', $this->settings['search']['kindOfObjects']);
        foreach ($kindOfObjectUids as $kindOfObjectUid) {
            $kindOfObject = $this->objektartRepository->findByUid($kindOfObjectUid);
            if ($kindOfObject instanceof Objektart) {
                $this->filter->addObjektart($kindOfObject);
            }
        }
        if (isset($this->settings['importids']) && $this->settings['importids'] !== '') {
            $arrayOfImportIds = explode(',', $this->settings['importids']);
            if (is_array($arrayOfImportIds) && count($arrayOfImportIds) >= 1) {
                $this->filter->setImportIds($arrayOfImportIds);
            } else {
                $this->filter->setImportIds([]);
            }
        } else {
            $arrayOfImportIds = null;
        }
        $this->view->assign('regionOptions', $this->regionRepository->getAllForFilter($arrayOfImportIds));
        $this->view->assign('districtOptions', $this->stadtteilRepository->getAllForFilter($this->settings['preset']['search']['districtFormat'], $arrayOfImportIds));
        $this->view->assign('cityOptions', $this->ortRepository->getAllForFilter($arrayOfImportIds));
        $this->view->assign('groupedCitiesAndDistricts', $this->ortRepository->getGroupedCitiesAndDistrictsForFilter($arrayOfImportIds));
        $this->view->assign('filter', $this->filter);
        $this->view->assign('cities', $this->ortRepository->findAll());
        return $this->htmlResponse($this->view->render());
    }

    /**
     * Action for Standard List View
     * @throws AspectNotFoundException
     * @throws JsonException
     * @throws Exception
     */
    public function listAction(): ResponseInterface
    {
        $target = '';
        if(!empty($_REQUEST) && !empty($_REQUEST['target'])) {
            $target = trim(strip_tags($_REQUEST['target']));
        }

        if($target === 'react_parking') {
            $fromPost = [];
            if (!empty($_POST)) {
                if (!empty($_POST['zip'])) {
                    $fromPost['zip'] = trim(strip_tags($_POST['zip']));
                }
                if (!empty($_POST['city'])) {
                    $fromPost['city'] = trim(strip_tags($_POST['city']));
                }
                if (!empty($_POST['street'])) {
                    $fromPost['street'] = trim(strip_tags($_POST['street']));
                }
                if (!empty($_POST['nr'])) {
                    $fromPost['nr'] = trim(strip_tags($_POST['nr']));
                }
                if (!empty($_POST['type'])) {
                    $fromPost['type'] = trim(strip_tags($_POST['type']));
                }
            }
            $this->view->assign('fromPost', $fromPost);
            return $this->htmlResponse($this->view->render());

        }

        /**
         * @var $filter Filter
         */
        if (isset($this->settings['importids']) && $this->settings['importids'] !== '') {
            $arrayOfImportIds = explode(',', $this->settings['importids']);
            if (is_array($arrayOfImportIds) && count($arrayOfImportIds) >= 1) {
                $this->filter->setImportIds($arrayOfImportIds);
            } else {
                $this->filter->setImportIds([]);
            }
        } else {
            $arrayOfImportIds = null;
        }

        if (isset($this->settings['list']['layout']) && $this->settings['list']['layout'] === 'LiveSearch') {
            $filter = new Filter();

            // add city to this filte
            $ort = $this->filter->getOrt();
            if ($ort instanceof Ort) {
                $filter->addOrt($ort);
            }
            if (isset($this->settings['importids']) && $this->settings['importids'] !== '') {
                $filter->setImportIds($this->filter->getImportIds());
            }

            $filterToUse = $filter;
            $cacheIdentifier = 'list_' . $this->filter->getCacheIdentifier() . '_L' . $this->getLanguageUid();
        } else {
            $filterToUse = $this->filter;
            $cacheIdentifier = 'list_' . $filterToUse->getCacheIdentifier() . '_L' . $this->getLanguageUid();
        }

        if ($this->noSearchParameters && $this->settings['list']['noParametersNoResults'] === '1') {
            if (($entry = $this->sgEstateCoreCache->get('sg_estate_base_empty_result' . '_L' . $this->getLanguageUid())) === false) {
                $this->view->assign('realties', []);
                $this->view->assign('aggregatedFilterData', json_encode([], JSON_THROW_ON_ERROR));
                $this->view->assign('citiesAndDistrictsForFilter', json_encode($this->generateCityAndDistrictStructureForFilter(), JSON_THROW_ON_ERROR));

                $this->view->assign('presets', $this->generateFilterPresetsFromResult([]));

                $entry = $this->view->render();

                $tags = ['listing', 'searchresult'];
                $lifetime = $this->settings['defaultCacheLifetime'];

                // Save value in cache
                $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
            }
            return $this->htmlResponse($entry);
        }

        if (($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
            $realties = $this->immobilieRepository->findAllWithFilter($filterToUse);

            // Sorting data by current city
            if ($this->settings['preset']['search']['sortByCurrentCity'] === 1) {
                $cityName = explode(',', $filterToUse->getAddressString());
                $sortingData1 = [];
                $sortingData2 = [];
                foreach ($realties as $key => $realty) {
                    if ($cityName[0] === $realty->getObjektOrt()->getBezeichner()) {
                        $sortingData1[$key] = $realty;
                    } else {
                        $sortingData2[$key] = $realty;
                    }
                }
                $realties = array_merge($sortingData1, $sortingData2);
            }

            if ($this->settings['preset']['search']['useSlider'] === 1) {
                // BEGIN: Assign Values for Slider-Configuration to View
                $this->view->assign('sizeSliderData', $this->getSizeSliderData());
                $this->view->assign('roomSliderData', $this->getRoomSliderData());
                $this->view->assign('amountSliderData', $this->getAmountSliderData());
                // END: Assign Values for Slider-Configuration to View
            }
            $this->view->assign('realties', $realties);
            $citiesAndDistricts = $this->generateCityAndDistrictUniversal();
            $this->view->assign('instantFilter', $this->generateInstantFilterPresetsFromResult($realties));
            $this->view->assign('districtOptions', $this->stadtteilRepository->getAllForFilter($this->settings['preset']['search']['districtFormat'], $arrayOfImportIds));
            $this->view->assign('cityOptions', $this->getCitiesInRealty($citiesAndDistricts));
            $this->view->assign('groupedCitiesAndDistricts', $this->getStrangeFormatedCitiesAndDistricts($citiesAndDistricts));
            $this->view->assign('filter', $this->filter);
            $this->view->assign('aggregatedFilterData', json_encode($this->getAggregatedFilterData(), JSON_THROW_ON_ERROR));
            $this->view->assign('citiesAndDistrictsForFilter', json_encode($this->generateCityAndDistrictStructureForFilter(), JSON_THROW_ON_ERROR));
            $this->view->assign('presets', $this->generateFilterPresetsFromResult($realties));

            $entry = $this->view->render();

            $tags = ['listing', 'searchresult'];
            $lifetime = $this->settings['defaultCacheLifetime'];

            // Save value in cache
            $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
        }
        return $this->htmlResponse($entry);
    }

    /**
     * @param $result
     * @return array
     */
    protected function generateInstantFilterPresetsFromResult($result): array
    {
        $presets = [];

        // Initialize
        $cities = [];
        $balcony = false;
        $lift = false;
        $wbs = false;
        $bathShower = false;
        $bathTub = false;
        $bathWindow = false;

        foreach ($result as $realty) {
            /**
             * @var $realty Immobilie
             */
            // City & Distric
            $ort = $realty->getObjektOrt();
            $cities[$ort->getUid()]['bezeichner'] = $ort->getBezeichner();
            $objektStadtteil = $realty->getObjektStadtteil();
            if ($objektStadtteil !== null) {
                $cities[$ort->getUid()]['districts'][$objektStadtteil->getUid()] = [
                    'uid' => $objektStadtteil->getUid(),
                    'bezeichner' => $objektStadtteil->getBezeichner()
                ];
            }

            // Balcony
            if ($realty->getAnzahlBalkone() > 0) {
                $balcony = true;
            }

            // Lift
            if ($realty->getAufzugPersonen() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $lift = true;
            }

            // WBS
            if ($realty->getWbsErforderlich() === NullableBooleanEnum::POSITIVE_AUSWAHL) {
                $wbs = true;
            }

            // Bath
            /**
             * @var $badAusstattung Badausstattung
             */
            foreach ($realty->getAusstattungBad() as $badAusstattung) {
                if ($badAusstattung->getKuerzel() === 'WANNE') {
                    $bathTub = true;
                }
                if ($badAusstattung->getKuerzel() === 'DUSCHE') {
                    $bathShower = true;
                }
                if ($badAusstattung->getKuerzel() === 'FENSTER') {
                    $bathWindow = true;
                }
            }
        }

        $regions = [];

        foreach ($cities as $key => $value) {
            $regionsForCity = $this->regionRepository->findRegionsForCity($this->ortRepository->findByUid($key));
            foreach ($regionsForCity as $region) {
                /**
                 * @var Region $region
                 */
                $bezeichner = $region->getBezeichner();
                $regions[$bezeichner] = [
                    'id' => $region->getUid(),
                    'name' => $bezeichner
                ];
            }
        }

        ksort($regions);
        $regions = array_values($regions);

        // Build Return Variable
        $presets['regions'] = $regions;
        $presets['cities'] = $cities;
        $presets['balcony'] = $balcony;
        $presets['lift'] = $lift;
        $presets['wbs'] = $wbs;
        $presets['bathShower'] = $bathShower;
        $presets['bathTub'] = $bathTub;
        $presets['bathWindow'] = $bathWindow;

        return $presets;
    }

    protected function getCitiesInRealty($cities): array
    {
        $returner = array();
        foreach($cities as $city) {
            $returner[$city['uid']] = $city['caption'];
        }
        return $returner;
    }

    protected function getStrangeFormatedCitiesAndDistricts($cities): array
    {
        $returner = array();
        foreach($cities as $city) {
            $returner[] = array($city['uid'] => $city['caption']);
            if (is_array($city['districts']) && count($city['districts']) > 0) {
                foreach ($city['districts'] as $district) {
                    $returner[] = array($city['uid'] . '_' . $district['uid'] => $district['caption']);
                }
            }
        }
        return $returner;
    }

    /**
     * @param string $realty Without any route enhancer the urlIdentifier (externalId) will be passed.
     *                       With route enhancer the uid of the estate item will be passed.
     *                       You can also always pass the uid of the element.
     *
     * @throws AspectNotFoundException
     */
    public function detailAction(string $realty=''): ResponseInterface
    {
        $noCache    = $this->settings['detail']['cache'] ?? 0;
        if(!empty($realty)) {
            if($noCache !== 1) {
                $cacheIdentifier = 'detail_' . sha1($realty) . '_L' . $this->getLanguageUid();
                if(($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
                    $entry = $this->getEntry($realty);
                    $tags = ['detail'];
                    $lifetime = $this->settings['defaultCacheLifetime'];
                    $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
                }
            } else {
                $entry = $this->getEntry($realty);
            }

            return $this->htmlResponse($entry);
        }

        return $this->redirectToUri('/');
    }

    /**
     * Action for Standard Contact Form & Process
     *
     * @throws NoSuchArgumentException|TransportExceptionInterface
     */
    public function contactAction(): ResponseInterface
    {
        $this->pregenerateContactForm();
        if ($this->request->hasArgument('realty')) {
            $realty = $this->immobilieRepository->findOneByUrlIdentifier($this->request->getArgument('realty'));
            $this->view->assign('realty', $realty);
        } elseif ($this->request->hasArgument('formValues')) {
            return $this->processContactForm();
        } else {
            return $this->redirectToUri('/');
        }

        return $this->htmlResponse();
    }

    protected function pregenerateContactForm(): void
    {
        ksort($this->settings['contact']['fields']);
        $this->view->assign('settings', $this->settings);
        $this->view->assign('salutationOptions', $this->getSalutationOptions());
    }

    /**
     * Get Salutation for Contact Form
     *
     * @return array
     */
    protected function getSalutationOptions(): array
    {
        return [
            'mr' => LocalizationUtility::translate('template.contact.salutation.mr', 'SgEstateBase'),
            'mrs' => LocalizationUtility::translate('template.contact.salutation.mrs', 'SgEstateBase'),
        ];
    }

    /**
     * @throws TransportExceptionInterface
     */
    protected function processContactForm(): ResponseInterface
    {
        /**
         * @var $formValues array
         * @var $realty Immobilie
         */
        $formValues = $this->request->getArgument('formValues');

        $processedFormValues = [];
        $errors = [];
        foreach ($this->settings['contact']['fields'] as $fieldConfiguration) {
            if ($fieldConfiguration['required'] === '1' && (!isset($formValues[$fieldConfiguration['name']]) || $formValues[$fieldConfiguration['name']] === '')) {
                $errors[$fieldConfiguration['name']][] = 'required';
            } else {
                $processedFormValues[$fieldConfiguration['name']] = $formValues[$fieldConfiguration['name']] ?? '';
            }
        }

        $realty = $this->immobilieRepository->findByUid($formValues['realty']);

        if (count($errors) === 0) {
            $valid = true;
        } else {
            $valid = false;
        }

        if ($valid) {
            // Generate Receivers
            $kontaktperson = $realty?->getKontaktperson();
            if (($this->settings['contact']['useRealtyContactAsReceiver'] === 1) && $kontaktperson instanceof Kontaktperson) {
                $emailDirekt = $kontaktperson->getEmailDirekt();
                $emailZentrale = $kontaktperson->getEmailZentrale();
                $emailSonstige = $kontaktperson->getEmailSonstige();
                if (GeneralUtility::validEmail($emailDirekt)) {
                    $receivers = [$emailDirekt];
                } elseif (GeneralUtility::validEmail($emailZentrale)) {
                    $receivers = [$emailZentrale];
                } elseif (GeneralUtility::validEmail($emailSonstige)) {
                    $receivers = [$emailSonstige];
                } else {
                    $receivers = explode(',', $this->settings['contact']['receivers']);
                }
            } else {
                $receivers = explode(',', $this->settings['contact']['receivers']);
            }

            if ($this->settings['contact']['addFeedbackAttachment']) {
                $attachmentContent = $this->generateFeedbackAttachment('Email/FeedbackAttachment', ['processedFormValues' => $processedFormValues, 'settings' => $this->settings, 'realty' => $realty]);
                $attachment = [
                    'type' => 'content',
                    'content' => $attachmentContent,
                    'filename' => 'feedback.xml'
                ];
                $attachments = [$attachment];
            } else {
                $attachments = [];
            }

            $success = $this->sgEstateCoreServices->sendTemplateEmail($receivers, $this->settings['contact']['sender'], $this->settings['contact']['subject'], 'SgEstateBaseContact', [], ['realty' => $realty, 'formValues' => $processedFormValues, 'settings' => $this->settings], $attachments, $this->settings['contact']['mailformat']);

            if ($success && (int)$this->settings['contact']['redirectPidSuccess'] >0) {
                $this->uriBuilder->reset();
                $this->uriBuilder->setTargetPageUid((int)$this->settings['contact']['redirectPidSuccess']);
                $uri = $this->uriBuilder->build();
                return $this->redirectToUri($uri);
            }

            if (!$success && (int)$this->settings['contact']['redirectPidFailure'] > 0) {
                $this->uriBuilder->reset();
                $this->uriBuilder->setTargetPageUid((int)$this->settings['contact']['redirectPidFailure']);
                $uri = $this->uriBuilder->build();
                return $this->redirectToUri($uri);
            }

            if ($success && (int)$this->settings['contact']['redirectPidSuccess'] === 0) {
                $this->view->assign('showResult', true);
                $this->view->assign('success', true);
            } elseif (!$success && (int)$this->settings['contact']['redirectPidFailure'] === 0) {
                $this->view->assign('showResult', true);
                $this->view->assign('success', false);
            }
        } else {
            $this->view->assign('formValues', $formValues);
            if (is_array($formValues['channel'])) {
                $this->view->assign('channelValues', array_fill_keys(array_keys(array_flip($formValues['channel'])), 1));
            }
            if (is_array($formValues['wish'])) {
                $this->view->assign('wishValues', array_fill_keys(array_keys(array_flip($formValues['wish'])), 1));
            }
            $this->view->assign('errors', $errors);
        }
        $this->view->assign('realty', $realty);

        return $this->htmlResponse();
    }

    protected function generateFeedbackAttachment($templateName, array $variables = [], $format = 'xml')
    {
        /**
         * @var $standaloneView StandaloneView
         */
        $objectManager = GeneralUtility::makeInstance('TYPO3\CMS\Extbase\Object\ObjectManager');
        $extbaseFrameworkConfiguration = $this->configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FRAMEWORK);

        $standaloneView = $objectManager->get(StandaloneView::class);
        $standaloneView->setTemplateRootPaths($extbaseFrameworkConfiguration['view']['templateRootPaths']);
        $standaloneView->setPartialRootPaths($extbaseFrameworkConfiguration['view']['partialRootPaths']);
        $standaloneView->setLayoutRootPaths($extbaseFrameworkConfiguration['view']['layoutRootPaths']);
        $standaloneView->setTemplate($templateName);

        $standaloneView->setFormat($format);
        $standaloneView->assignMultiple($variables);

        return $standaloneView->render();
    }

    /**
     * Action for Mapinfo Called via AJAX
     */
    public function mapinfoAction(): ResponseInterface
    {
        if (isset($_REQUEST['uid'])&&is_numeric($_REQUEST['uid'])) {
            $cacheIdentifier = 'mapinfo_' . sha1($_REQUEST['uid']);

            if (($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
                $immobilie = $this->immobilieRepository->findByUid($_REQUEST['uid']);
                if ($immobilie instanceof Immobilie) {
                    $this->view->assign('realty', $immobilie);
                    $entry = $this->view->render();

                    $tags = ['mapinfo'];
                    $lifetime = $this->settings['defaultCacheLifetime'];

                    // Save value in cache
                    $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
                } else {
                    $entry = '';
                }
            }
            return $this->htmlResponse($entry);
        }
        return $this->htmlResponse('');
    }

    /**
     * Action used for several AJAX Calls
     *
     * ViewHelper for URI Function "allRealties" <f:uri.action action='ajax' controller='Base' arguments='{function:"allRealties"}' additionalParams='{type: 1403693888}' noCacheHash='true' />
     *
     * @throws NoSuchArgumentException
     * @throws TransportExceptionInterface
     * @throws JsonException|Exception
     */
    public function ajaxAction(): ResponseInterface
    {
        if ($this->request->hasArgument('function')) {
            $function = $this->request->getArgument('function');
            $this->view->assign('function', ucfirst($function));
            switch ($function) {
                case 'realtiesForReactApplication':
                    $json = $this->getDataForReact();
                    $this->view->assign('jsonReturn', $json);
                    $entry = $this->view->render();
                    return $this->htmlResponse($entry);
                case 'allRealties':
                    $cacheIdentifier = 'ajax_allRealties';

                    if (($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
                        $allRealties = $this->immobilieRepository->findAll();
                        $arrayReturn = [
                            'realties' => [],
                            'cities' => [],
                            'size' => ['min' => 0, 'max' => 0 ],
                            'rooms' => ['min' => 0, 'max' => 0 ],
                            'rent' => ['min' => 0, 'max' => 0 ]
                        ];

                        // make list of cities and districts
                        // and find min and max values for slider fields
                        $cities = [];
                        foreach ($allRealties as $realty) {
                            // see whether this realty has higher or lower limes values
                            // size
                            if ($realty->getGesamtflaeche() > $arrayReturn['size']['max']) {
                                $arrayReturn['size']['max'] = $realty->getGesamtflaeche();
                            }
                            if ($arrayReturn['size']['min'] === 0 || $realty->getGesamtflaeche() < $arrayReturn['size']['min']) {
                                $arrayReturn['size']['min'] = $realty->getGesamtflaeche();
                            }
                            // rooms
                            if ($realty->getAnzahlZimmer() > $arrayReturn['rooms']['max']) {
                                $arrayReturn['rooms']['max'] = $realty->getAnzahlZimmer();
                            }
                            if ($arrayReturn['rooms']['min'] === 0 || $realty->getAnzahlZimmer() < $arrayReturn['rooms']['min']) {
                                $arrayReturn['rooms']['min'] = $realty->getAnzahlZimmer();
                            }
                            // rent
                            if ($realty->getNettokaltmiete() > $arrayReturn['rent']['max']) {
                                $arrayReturn['rent']['max'] = $realty->getNettokaltmiete();
                            }
                            if ($arrayReturn['rent']['min'] === 0 || $realty->getNettokaltmiete() < $arrayReturn['rent']['min']) {
                                $arrayReturn['rent']['min'] = $realty->getNettokaltmiete();
                            }

                            // find city in cities
                            $cityMatched = false;
                            $citiesCount = count($cities);
                            for ($i = 0; $i < $citiesCount; $i++) {
                                if ($cities[$i]['id'] === $realty->getObjektOrt()->getUid()) {
                                    // found match. we have this city
                                    $cityMatched = true;
                                    // take all districts and walk through all districts to find out
                                    // whether we also have this district
                                    $districtMatched = false;
                                    $countDistricts = count($cities[$i]['districts']);
                                    if ($realty->getObjektStadtteil() instanceof Stadtteil) {
                                        for ($k = 0; $k < $countDistricts; $k++) {
                                            if ($cities[$i]['districts'][$k]['id'] === $realty->getObjektStadtteil()->getUid()) {
                                                // this district exists
                                                $districtMatched = true;
                                            }
                                        }
                                        if ($districtMatched === false) {
                                            // add district to this city
                                            $thisDistrict = [
                                                'id' => $realty->getObjektStadtteil()->getUid(),
                                                'name' => $realty->getObjektStadtteil()->getBezeichner()
                                            ];
                                            $cities[$i]['districts'][] = $thisDistrict;
                                        }
                                    }
                                }
                            }
                            if ($cityMatched === false) {
                                // add city to city stack
                                // add district to new city stack
                                $districts = [];
                                if ($realty->getObjektStadtteil() instanceof Stadtteil) {
                                    $districts = [
                                        [
                                            'id' => $realty->getObjektStadtteil()->getUid(),
                                            'name' => $realty->getObjektStadtteil()->getBezeichner()
                                        ]
                                    ];
                                }
                                $thisCity = [
                                    'id' => $realty->getObjektOrt()->getUid(),
                                    'name' => $realty->getObjektOrt()->getBezeichner(),
                                    'districts' => $districts
                                ];
                                $cities[] = $thisCity;
                            }
                        }
                        $arrayReturn['cities'] = $cities;

                        $tempRegions = [];

                        foreach ($cities as $city) {
                            $regionsForCity = $this->regionRepository->findRegionsForCity($this->ortRepository->findByUid($city['id']));
                            foreach ($regionsForCity as $region) {
                                /**
                                 * @var Region $region
                                 */
                                $bezeichner = $region->getBezeichner();
                                $tempRegions[$bezeichner] = [
                                    'id' => $region->getUid(),
                                    'name' => $bezeichner
                                ];
                            }
                        }
                        ksort($tempRegions);
                        $tempRegions = array_values($tempRegions);

                        $arrayReturn['regions'] = $tempRegions;

                        foreach ($allRealties as $realty) {
                            /**
                             * @var $realty Immobilie
                             */
                            $district = [];
                            if ($realty->getObjektStadtteil() instanceof Stadtteil) {
                                $district = [
                                    'id' => $realty->getObjektStadtteil()->getUid(),
                                    'name' => $realty->getObjektStadtteil()->getBezeichner()
                                ];
                            }
                            $arrayReturn['realties'][] = [
                                'uid' => $realty->getUid(),
                                'city' => [
                                    'id' => $realty->getObjektOrt()->getUid(),
                                    'name' => $realty->getObjektOrt()->getBezeichner()
                                ],
                                'district' => $district,
                                'rooms' => $realty->getAnzahlZimmer(),
                                'size' => $realty->getWohnflaeche(),
                                'rent' => $realty->getNettokaltmiete(),
                            ];
                        }

                        $this->view->assign('jsonReturn', json_encode($arrayReturn, JSON_THROW_ON_ERROR));

                        $entry = $this->view->render();

                        $tags = ['ajax_allRealties'];
                        $lifetime = $this->settings['defaultCacheLifetime'];

                        // Save value in cache
                        $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
                    }
                    return $this->htmlResponse($entry);
                case 'contactForm':
                    $this->pregenerateContactForm();
                    if ($this->request->hasArgument('realtyUid')) {
                        $this->view->assign('realty', ['uid' => $this->request->getArgument('realtyUid')]);
                    }
                    if ($this->request->hasArgument('formValues')) {
                        $this->processContactForm();
                    }
                    break;
                case 'getDistance':
                    if ($this->request->hasArgument('realtyUid')) {
                        $realty = $this->immobilieRepository->findOneBy(['uid' => $this->request->getArgument('realtyUid')]);
                        if ($realty instanceof Immobilie) {
                            $lat = $this->filter->getLatitude() > 0
                                ? $this->filter->getLatitude()
                                : ($this->request->hasArgument('lat')
                                    ? $this->request->getArgument('lat')
                                    : 0);
                            $lon = $this->filter->getLongitude() > 0
                                ? $this->filter->getLongitude()
                                : ($this->request->hasArgument('lon')
                                    ? $this->request->getArgument('lon')
                                    : 0);
                            $this->view->assign('distance', $this->calcDistance($realty->getObjektBreitengrad(), $realty->getObjektLaengengrad(), $lat, $lon));
                        }
                    }
                    break;
                case 'getCityAndDistricts':
                    $cacheIdentifier = 'ajax_getCityAndDistricts';
                    if (($entry = $this->sgEstateCoreCache->get($cacheIdentifier)) === false) {
                        $cityAndDistrictStructure = $this->generateCityAndDistrictStructureForFilter();

                        $this->view->assign('jsonReturn', $cityAndDistrictStructure);

                        $entry = $this->view->render();

                        $tags = ['ajax_getCityAndDistricts'];
                        $lifetime = $this->settings['defaultCacheLifetime'];

                        // Save value in cache
                        $this->sgEstateCoreCache->set($cacheIdentifier, $entry, $tags, $lifetime);
                    }
                    return $this->htmlResponse($entry);
                case 'getRealtiesByUids':
                    $uidData = json_decode($this->request->getParsedBody()['uids'] ?? null, false, 512, JSON_THROW_ON_ERROR);
                    if (is_array($uidData)) {
                        $arrayReturn = [];
                        $processingInstructions = [
                            'width' => '400c',
                            'height' => '300c',
                        ];
                        foreach ($uidData as $uid) {
                            $realty = $this->immobilieRepository->findOneByUrlIdentifier($uid);
                            if ($realty instanceof Immobilie) {
                                /**
                                 * @var $realty Immobilie
                                 */
                                $district = [];
                                if ($realty->getObjektStadtteil() instanceof Stadtteil) {
                                    $district = [
                                        'id' => $realty->getObjektStadtteil()->getUid(),
                                        'name' => $realty->getObjektStadtteil()->getBezeichner()
                                    ];
                                }
                                $image = $this->imageService->getImage(Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/' . $realty->getTitelbild()?->getDatei(), null, false);
                                $processedImage = $this->imageService->applyProcessingInstructions($image, $processingInstructions);
                                $arrayReturn[] = [
                                    'id' => $realty->getUid(),
                                    'data' => [
                                        'objectNumberInternal' => $realty->getObjektnrIntern(),
                                        'objectNumberExternal' => $realty->getObjektnrExtern(),
                                        'openimmoObjid' => $realty->getOpenimmoObjid(),
                                        'title' => $realty->getObjekttitel(),
                                        'street' => $realty->getObjektStrasse(),
                                        'houseNumber' => $realty->getObjektHausnummer(),
                                        'zip' => $realty->getObjektPlz(),
                                        'size' => $realty->getWohnflaeche(),
                                        'rooms' => $realty->getAnzahlZimmer(),
                                        'rent' => $realty->getNettokaltmiete(),
                                        'balcony' => $realty->getAnzahlBalkone() > 0,
                                        'city' => [
                                            'id' => $realty->getObjektOrt()->getUid(),
                                            'name' => $realty->getObjektOrt()->getBezeichner()
                                        ],
                                        'district' => $district,
                                        'url' => $this->uriBuilder->reset()->setTargetPageUid($this->settings['pid']['detail'])->uriFor('detail', ['realty' => $realty->getUrlIdentifier()], 'Base', 'sgestatebase', 'pi1')
                                    ],
                                    'media' => [
                                        'preview' => $this->imageService->getImageUri($processedImage)
                                    ],
                                    'geometry' => [
                                        'location' => [
                                            'lat' => $realty->getObjektBreitengrad(),
                                            'lng' => $realty->getObjektLaengengrad()
                                        ]
                                    ],
                                ];
                            }
                        }
                        $this->view->assign('jsonReturn', json_encode($arrayReturn, JSON_THROW_ON_ERROR));
                    } else {
                        $this->view->assign('jsonReturn', json_encode([], JSON_THROW_ON_ERROR));
                    }
                    $entry = $this->view->render();
                    return $this->htmlResponse($entry);
                default:
                    break;
            }
        } else {
            die('');
        }
        return $this->htmlResponse();
    }

    /**
     * @throws Exception
     * @throws JsonException
     */
    protected function getDataForReact(): false|string
    {
        // $file = dirname(__FILE__) . '../../Resources/Public/Data/react.json';
        $file = Environment::getPublicPath() . '/fileadmin/React/Data/react.json';
        if(file_exists($file) && filesize($file)) {
            $json = file_get_contents($file);
        }
        else {
            $json = $this->getReactJson();
        }
        return json_encode(json_decode($json, false, 512, JSON_THROW_ON_ERROR), JSON_THROW_ON_ERROR);
    }

    /**
     * @throws Exception
     * @throws JsonException
     */
    public function getReactJson(): false|string
    {
        $cw = '320';
        $ch = '240';
        $sql = "a.uid AS id,
            a.objektnr_intern AS objectNumberInternal,
            a.objektnr_extern AS objectNumberExternal,
            a.openimmo_objid AS openimmoObjId,
            a.objekttitel AS title,
            a.objekt_strasse AS street,
            a.objekt_hausnummer AS houseNumber,
            a.objekt_plz AS zip,
            a.wohnflaeche AS size,
            a.anzahl_zimmer AS rooms,
            a.nettokaltmiete AS rent,
            IF(a.anzahl_balkone > 0, 'true', 'false') as balcony,
            a.objekt_breitengrad AS lat,
            a.objekt_laengengrad AS lng,
            a.objekt_stadtteil AS district_id,
            IFNULL(b.bezeichner, '') AS district_name,
            c.kuerzel AS object_art,
            a.objekt_ort AS city_id,
            d.bezeichner AS city_name,
            e.datei AS image,
            '' as identifier, 
            t.bezeichner AS object_arttyp 
            FROM tx_sgestatecore_domain_model_immobilie AS a
            LEFT JOIN tx_sgestatecore_domain_model_stadtteil AS b ON a.objekt_stadtteil = b.uid
            LEFT JOIN tx_sgestatecore_domain_model_objektart AS c ON a.objektart = c.uid
            LEFT JOIN tx_sgestatecore_domain_model_ort AS d ON a.objekt_ort = d.uid 
            LEFT JOIN (SELECT immobilie, datei, typ FROM tx_sgestatecore_domain_model_anhang WHERE typ IN (1, 7) GROUP BY immobilie) 
                AS e ON a.uid = e.immobilie 
            LEFT JOIN tx_sgestatecore_domain_model_objektarttyp AS t ON a.objektarttyp = t.uid 
            WHERE a.deleted = 0
            AND a.hidden = 0;";
        $connection = GeneralUtility::makeInstance(ConnectionPool::class)?->getConnectionByName('Default');
        $allRealties = $connection->createQueryBuilder()->addSelectLiteral($sql)->executeQuery()->fetchAllAssociative();
        $arrayReturn = ['realties' => []];
        $_files = glob(Environment::getPublicPath() . '/fileadmin/React/Data/Previews/*');
        foreach($_files as $_file) {
            if (is_file($_file)) {
                @unlink($_file);
            }
        }
        foreach ($allRealties as $realty) {
            $src = Environment::getPublicPath() . '/uploads/tx_sgestatecore/media/' . $realty['image'];
            $dst = Environment::getPublicPath() . '/fileadmin/React/Data/Previews/' . $realty['image'];
            $imageUrl = '/fileadmin/React/Data/Previews/' . $realty['image'];
            if (file_exists($src)) {
                $this->image_resize($src, $dst, $cw, $ch);
            }
            $objectManager = GeneralUtility::makeInstance('TYPO3\\CMS\Extbase\\Object\\ObjectManager');
            $configurationManager = $objectManager->get(ConfigurationManager::class);
            $extbaseFrameworkConfiguration = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FULL_TYPOSCRIPT);
            $detailPid = $extbaseFrameworkConfiguration['plugin.']['tx_sgestatebase.']['widgets.']['searchbox.']['settings.']['pid.']['detail'];
            $itemUrl = $this->uriBuilder->reset()->setTargetPageUid($detailPid)->uriFor('detail', ['realty' => preg_replace(array('/\//','/\\\/'),'-',$realty['objectNumberExternal'])], 'Base', 'sgestatebase', 'pi1');
            if(empty($itemUrl) || str_contains($itemUrl, 'typo3')) {
                $itemUrl = '/immobilien/detail/' . preg_replace(array('/\//', '/\\\/'), '-', $realty['objectNumberExternal']);
            }
//            $is = rand(0, 1);
//            $can = $is == 0 ? rand(0, 1) : 0;
//            $lock = rand(0, 1);
            $is = 0;
            $can = 0;
            $lock = 0;
            $arrayReturn['realties'][] = [
                'show' => false,
                'highlighted' => false,
                'id' => $realty['id'],
                'type' => $realty['object_art'],
                'arttype' => $realty['object_arttyp'],
                'data' => [
                    'objectNumberInternal' => $realty['objectNumberInternal'],
                    'objectNumberExternal' => $realty['objectNumberExternal'],
                    'openimmoObjid' => $realty['openimmoObjId'],
                    'title' => $realty['title'],
                    'street' => $realty['street'],
                    'nr' => $realty['houseNumber'],
                    'zip' => $realty['zip'],
                    'size' => $realty['size'],
                    'rooms' => $realty['rooms'],
                    'rent' => $realty['rent'],
                    'balcony' => $realty['balcony'],
                    'city' => [
                        'id' => $realty['city_id'],
                        'name' => $realty['city_name']
                    ],
                    'district' => [
                        'id' => $realty['district_id'],
                        'name' => $realty['district_name']
                    ],
                    'url' => $itemUrl
                ],
                'media' => [
                    'preview' => $imageUrl
                ],
                'geometry' => [
                    'location' => [
                        'lat' => $realty['lat'],
                        'lng' => $realty['lng']
                    ]
                ],
                'parking' => [
                    'wallbox' => [
                        'is' => $is,
                        'can' => $can
                    ],
                    'lock' => $lock
                ],
            ];
        }
        return json_encode($arrayReturn, JSON_THROW_ON_ERROR);
    }

    protected function image_resize($src, $dst, $resize_width, $resize_height): int
    {
        [$width, $height] = getimagesize($src);
        $ratio = $width / $height;
        if ($width > $height) {
            $width = ceil($width-($width*abs($ratio - ($resize_width / $resize_height))));
        } else {
            $height = ceil($height-($height*abs($ratio - ($resize_width / $resize_height))));
        }
        $new_width = $resize_width;
        $new_height = $resize_height;
        $resized_image = imagecreatetruecolor($new_width, $new_height);
        $info = pathinfo($src);
        $type = $info['extension'];
        switch ($type) {
            case 'png':
                $image = imagecreatefrompng($src);
                imagecopyresampled($resized_image, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
                imagepng($resized_image, $dst);
                break;
            case 'gif':
                $image = imagecreatefromgif($src);
                imagecopyresampled($resized_image, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
                imagegif($resized_image, $dst);
                break;
            default:
                $image = imagecreatefromjpeg($src);
                imagecopyresampled($resized_image, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
                imagejpeg($resized_image, $dst);
        }
        return 0;
    }

    /********************************************
     * Internal & Helper Functions
     *********************************************/

    protected function calcDistance($latitudePoint1, $longitudePoint1, $latitudePoint2, $longitudePoint2): float
    {
        $pi80 = M_PI / 180;
        $latitudePoint1 *= $pi80;
        $longitudePoint1 *= $pi80;
        $latitudePoint2 *= $pi80;
        $longitudePoint2 *= $pi80;
        $r = 6372.797; // mean radius of Earth in km
        $dlat = $latitudePoint2 - $latitudePoint1;
        $dlon = $longitudePoint2 - $longitudePoint1;
        $a = sin($dlat / 2) * sin($dlat / 2) + cos($latitudePoint1) * cos($latitudePoint2) * sin($dlon / 2) * sin($dlon / 2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        return $r * $c;
    }

    public function presetsAction(): ResponseInterface {
        // $this->view->assign('cities', $this->ortRepository->findAll());
        $this->view->assign('cities', $this->generateCityAndDistrictUniversal('object'));
        $entry = $this->view->render();
        return $this->htmlResponse($entry);
    }

    protected function transformFieldConfiguration(): array
    {
        $transformedConfiguration = [];
        foreach ($this->settings['contact']['fields'] as $fieldConfiguration) {
            $transformedConfiguration[$fieldConfiguration['name']] = $fieldConfiguration;
        }
        return $transformedConfiguration;
    }

    private function getEntry(string $realty): mixed
    {
        if ((int)$realty > 0) {
            $estateItem = $this->immobilieRepository->findByUid((int)$realty);
        }
        if (!$estateItem instanceof Immobilie) {
            $estateItem = $this->immobilieRepository->findOneByUrlIdentifier($realty);
        }
        $this->view->assign('realty', $estateItem);
        $this->view->assign('filter', $this->filter);
        return $this->view->render();
    }

}
