<?php
namespace SG\SgContact\Controller;

use SG\SgContact\Domain\Model\Category;
use SG\SgContact\Domain\Model\Person;
use SG\SgContact\Domain\Repository\CategoryRepository;
use SG\SgContact\Domain\Repository\DepartmentRepository;
use SG\SgContact\Domain\Repository\PersonRepository;
use TYPO3\CMS\Core\Cache\CacheManager;
use TYPO3\CMS\Core\Cache\Exception\NoSuchCacheException;
use TYPO3\CMS\Core\Cache\Frontend\FrontendInterface;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Mvc\Exception\NoSuchArgumentException;

/***************************************************************
   *
   *  Copyright notice
   *
   *  (c) 2020 Daniel Hoffmann <hoffmann@smply.gd>, smply.gd GmbH
   *           Marc Saleiko <saleiko@smply.gd>, smply.gd GmbH
   *
   *  All rights reserved
   *
   *  This script is part of the TYPO3 project. The TYPO3 project is
   *  free software; you can redistribute it and/or modify
   *  it under the terms of the GNU General Public License as published by
   *  the Free Software Foundation; either version 3 of the License, or
   *  (at your option) any later version.
   *
   *  The GNU General Public License can be found at
   *  http://www.gnu.org/copyleft/gpl.html.
   *
   *  This script is distributed in the hope that it will be useful,
   *  but WITHOUT ANY WARRANTY; without even the implied warranty of
   *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   *  GNU General Public License for more details.
   *
   *  This copyright notice MUST APPEAR in all copies of the script!
   ***************************************************************/

/**
 * ContactController
 */
class ContactController extends ActionController
{

    //region Class-Member
    /**
     * categoryRepository
     *
     * @var CategoryRepository
     */
    private $categoryRepository = NULL;

    /**
     * departmentRepository
     *
     * @var DepartmentRepository
     */
    private $departmentRepository = NULL;

    /**
     * personRepository
     *
     * @var PersonRepository
     */
    private $personRepository = NULL;

    /**
     * @var FrontendInterface
     */
    private $cache;
    //endregion



    //region Frontend-Functions
    /**
     * @throws NoSuchCacheException
     */
    public function initializeAction()
    {
        parent::initializeAction();
        $this->cache = GeneralUtility::makeInstance(CacheManager::class)->getCache('sgcontact_cache');
    }

    /**
     * Action for display a single Contact
     *
     * @return string
     */
    public function singleAction(){

        $cacheIdentifier = 'person_'.$this->settings['single']['person'].'_'.implode('', $this->settings['show']).'_'.$this->settings['single']['layout'].'_L'.$this->getLanguageUid();

        if (($content = $this->cache->get($cacheIdentifier)) === FALSE) {
            if (is_numeric($this->settings['single']['person'])){
                $person = $this->personRepository->findByUid($this->settings['single']['person']);
                if ($person instanceof Person){
                    $contentObjectData = $this->configurationManager->getContentObject()->data;

                    $this->view->assign('contentObjectData', $contentObjectData);
                    $this->view->assign('person', $person);

                    $content = $this->view->render();

                    $tags = array('persons');
                    $lifetime = $this->settings['defaultCacheLifetime'];
                    $this->cache->set($cacheIdentifier, $content, $tags, $lifetime);
                }
            }
        }
        return $content;
    }

    public function categoryAction(){
        $cacheIdentifier = 'category_'.$this->settings['category']['item'].'_'.implode('', $this->settings['show']).'_'.$this->settings['category']['layout'].'_L'.$this->getLanguageUid();
        if (($content = $this->cache->get($cacheIdentifier)) === FALSE) {
            if (is_numeric($this->settings['category']['item'])){
                $category = $this->categoryRepository->findByUid($this->settings['category']['item']);
                if ($category instanceof Category){
                    $this->view->assign('category', $category);
                    $persons = $this->personRepository->findByCategory($category);
                    $this->view->assign('persons', $persons);
                    $content = $this->view->render();

                    $tags = array('category');
                    $lifetime = $this->settings['defaultCacheLifetime'];
                    $this->cache->set($cacheIdentifier, $content, $tags, $lifetime);
                }
            }
        }
        return $content;
    }

    public function multiAction(){
        $cacheIdentifier = 'multi_'.$this->settings['multi']['layout'].str_replace(',','-',$this->settings['multi']['persons']).'_'.implode('', $this->settings['show']).'_L'.$this->getLanguageUid();
        if (($content = $this->cache->get($cacheIdentifier)) === FALSE) {
            $persons = array();
            if ($this->settings['multi']['persons'] != ''){
                $personsUids = explode(',', $this->settings['multi']['persons']);
                foreach ($personsUids as $personUid){
                    if (is_numeric($personUid)){
                        $person = $this->personRepository->findByUid($personUid);
                        if ($person instanceof \SG\SgContact\Domain\Model\Person){
                            array_push($persons, $person);
                        }
                    }
                }
                $this->view->assign('persons', $persons);
                $content = $this->view->render();

                $tags = array('multi');
                $lifetime = $this->settings['defaultCacheLifetime'];
                $this->cache->set($cacheIdentifier, $content, $tags, $lifetime);
            }
        }
        return $content;
    }
    //endregion

    //region Injector-Functions
    /**
     * @param CategoryRepository $categoryRepository
     */
    public function injectCategoryRepository(CategoryRepository $categoryRepository){
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @param DepartmentRepository $departmentRepository
     */
    public function injectDepartmentRepository(DepartmentRepository $departmentRepository){
        $this->departmentRepository = $departmentRepository;
    }

    /**
     * @param PersonRepository $personRepository
     */
    public function injectPersonRepository(PersonRepository $personRepository){
        $this->personRepository = $personRepository;
    }
    //endregion

    /**
     * Get the current language
     */
    private function getLanguageUid() {
        $context = GeneralUtility::makeInstance(Context::class);
        $sysLanguageUid = $context->getPropertyFromAspect('language', 'id');
        if (is_numeric($sysLanguageUid)){
            return $sysLanguageUid;
        }else{
            return 0;
        }
    }
}