<?php


namespace SG\BlogSgEnhancements\Controller;


use SG\BlogSgEnhancements\Domain\Repository\PostRepository;
use T3G\AgencyPack\Blog\Domain\Model\Category;
use T3G\AgencyPack\Blog\Domain\Model\Post;
use T3G\AgencyPack\Blog\Domain\Repository\CategoryRepository;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

class BlogPostController extends ActionController
{

    /** @var CategoryRepository */
    protected $categoryRepository;

    /** @var PostRepository */
    protected $postRepository;

    /**
     * @param CategoryRepository $categoryRepository
     */
    public function injectCategoryRepository(CategoryRepository $categoryRepository): void
    {
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @param PostRepository $postRepository
     */
    public function injectPostRepository(PostRepository $postRepository): void
    {
        $this->postRepository = $postRepository;
    }

    /**
     * Selects the newest post (of a given category)
     *
     * @throws InvalidQueryException
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     */
    public function featuredPostAction(): void
    {
        $category = $this->getFirstSelectedCategory();

        if( $category instanceof Category ) {
            $posts = $this->postRepository->findAllByCategoryWithLimit($category, 1);
        }
        else {
            $posts = $this->postRepository->findAllWithLimit(1);
        }

        $post = null;
        if($posts->count() > 0) {
            $post = $posts[0];
        }

        $this->view->assignMultiple([
            'posts' => $posts,
            'post' => $post,
        ]);
    }

    /**
     * Selects posts (by category) and uses a limit and offset setting
     *
     * @throws InvalidQueryException
     * @throws \TYPO3\CMS\Core\Context\Exception\AspectNotFoundException
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     */
    public function listPostsAction(): void
    {
        $limit = (int) ($this->settings['limit'] ?? 2 );
        $offset = (int) ($this->settings['offset'] ?? 0);

        /** @var Category|null $category */
        $category = $this->getFirstSelectedCategory();

        if( $category instanceof Category) {
            $posts = $this->postRepository->findAllByCategoryWithLimitAndOffset($category, $limit, $offset);
        }
        else {
            $posts = $this->postRepository->findAllWithLimitAndOffset($limit, $offset);
        }

        $this->view->assign('posts', $posts);
    }

    /**
     * Selects a curated list of posts provided by flexform settings
     */
    public function curatedPostsAction(): void
    {
        $posts = [];
        if( isset( $this->settings['postUids'] ) && $this->settings['postUids'] !== '' )
        {
            $uids = [];
            if( strpos($uids, ',') !== false ) {
                $uids = explode(',', $this->settings['postUids']);
            }
            else {
                $uids = [$this->settings['postUids']];
            }

            foreach($uids as $uid) {
                $post = $this->postRepository->findByUid($uid);
                if( $post instanceof Post ) {
                    $posts[] = $post;
                }
            }
        }
        $this->view->assign('posts', $posts);
    }

    /**
     * Selects a list of categories provided by the records categories selection
     *
     * @throws InvalidQueryException
     */
    public function listCategoriesAction(): void
    {
        $categories = $this->categoryRepository->getByReference(
            'tt_content',
            $this->configurationManager->getContentObject()->data['uid']
        );
        $this->view->assign('categories', $categories);
    }

    /**
     * @return Category|null
     */
    private function getFirstSelectedCategory(): ?Category
    {
        $category = null;
        try {
            $categories = $this->categoryRepository->getByReference(
                'tt_content',
                $this->configurationManager->getContentObject()->data[ 'uid' ]
            );

            if ( ! empty($categories)) {
                /** @noinspection CallableParameterUseCaseInTypeContextInspection */
                $category = $categories->getFirst();
            }
        } catch (InvalidQueryException $e) {}

        return $category;
    }

}