<?php


namespace SG\BlogSgEnhancements\Controller;

use SG\BlogSgEnhancements\Domain\Repository\PostRepository;
use SG\BlogSgEnhancements\Domain\Repository\CategoryRepository;
use T3G\AgencyPack\Blog\Domain\Model\Category;
use T3G\AgencyPack\Blog\Domain\Model\Post;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;
use TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException;

/**
 * Class BlogPostController
 *
 * @author Marc Saleiko <saleiko@smply.gd>
 * @package SG\BlogSgEnhancements\Controller
 */
class BlogPostController extends ActionController
{

    /** @var CategoryRepository */
    protected $categoryRepository;

    /** @var PostRepository */
    protected $postRepository;

    /**
     * @param CategoryRepository $categoryRepository
     */
    public function injectCategoryRepository(CategoryRepository $categoryRepository): void
    {
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * @param PostRepository $postRepository
     */
    public function injectPostRepository(PostRepository $postRepository): void
    {
        $this->postRepository = $postRepository;
    }

    /**
     * Selects the newest post (of a given category)
     *
     * @throws InvalidQueryException
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     */
    public function featuredPostAction(): void
    {
        $category = $this->getFirstSelectedCategory();

        if ($category instanceof Category) {
            $posts = $this->postRepository->findAllByCategoryWithLimit($category, 1);
        } else {
            $posts = $this->postRepository->findAllWithLimit(1);
        }

        $post = null;
        if ($posts->count() > 0) {
            $post = $posts[ 0 ];
        }

        $this->view->assign('post', $post);
    }

    /**
     * Selects posts (by category) and uses a limit and offset setting
     *
     * @throws InvalidQueryException
     * @throws \TYPO3\CMS\Core\Context\Exception\AspectNotFoundException
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     */
    public function listPostsAction(): void
    {
        $limit = (int)($this->settings[ 'limit' ] ?? 2);
        $offset = (int)($this->settings[ 'offset' ] ?? 0);

        if ($this->request->hasArgument('page')){
            $page = $this->request->getArgument('page');
        }
        else {
            $page = 1;
        }

        /** @var Category|null $category */
        $category = $this->getFirstSelectedCategory();

        if ($category instanceof Category) {
            $posts = $this->postRepository->findAllByCategoryWithLimitAndOffset($category, $limit * $page, $offset);
        } else {
            $posts = $this->postRepository->findAllWithLimitAndOffset($limit, $offset);
        }

        if ($this->settings['list']['includeNews'] == 1){

            $results_per_page = $this->settings['limit'];

            $newsRepository = GeneralUtility::makeInstance(\GeorgRinger\News\Domain\Repository\NewsRepository::class);
            /* @var $demand \GeorgRinger\News\Domain\Model\Dto\NewsDemand */

            $demand = $this->objectManager->get(\GeorgRinger\News\Domain\Model\Dto\NewsDemand::class, []);
            $demand->setLimit($this->settings['limit'] * $page);
            $demand->setOrderByAllowed('datetime');
            $demand->setOrder('datetime desc');
            $demand->setTimeRestriction(7776000);
            $demand->setDateField('crdate');
            $newsRecords = $newsRepository->findDemanded($demand);

            foreach ($newsRecords->toArray() as $result) {
                $posts->offsetSet(($posts->count()), $result);
            }

            $posts = $posts->toArray();
            $posts_count = count($posts);
            usort($posts, array('SG\BlogSgEnhancements\Controller\BlogPostController' ,"orderPosts"));
            $posts = array_splice($posts, ($page - 1) * $results_per_page, $this->settings['limit']);

            $pages = ceil(($posts_count / $results_per_page));

            $this->view->assign('page', $page);
            $this->view->assign('post_count', $posts_count);
            $this->view->assign('pages', $pages);
            $this->view->assign('results_per_page', $this->settings['limit']);
        }

        $this->view->assign('posts', $posts);
    }

    function orderPosts($post1, $post2){
        $a = $this->getTimestamp($post1);
        $b = $this->getTimestamp($post2);

        if ($a == $b) {
            return 0;
        }
        return ($a > $b) ? -1 : 1;
    }

    private function getTimestamp($object){
        $timestamp = 0;
        if (get_class($object) == 'GeorgRinger\News\Domain\Model\NewsDefault'){
            $timestamp = $object->getDatetime()->getTimestamp();
        }
        else if ($object instanceof Post){
            $timestamp = $object->getPublishDate();
        }

        return $timestamp;
    }

    /**
     * Selects a curated list of posts provided by flexform settings
     */
    public function curatedPostsAction(): void
    {
        $posts = [];
        if (isset($this->settings[ 'postUids' ]) && $this->settings[ 'postUids' ] !== '') {
            if (strpos($this->settings[ 'postUids' ], ',') !== false) {
                $uids = explode(',', $this->settings[ 'postUids' ]);
            } else {
                $uids = [$this->settings[ 'postUids' ]];
            }

            foreach ($uids as $uid) {
                $post = $this->postRepository->findByUid($uid);
                if ($post instanceof Post) {
                    $posts[] = $post;
                }
            }
        }
        $this->view->assign('posts', $posts);
    }

    /**
     * Selects a list of categories provided by the records categories selection
     *
     * @throws InvalidQueryException
     */
    public function listCategoriesAction(): void
    {
        $categories = $this->categoryRepository->getByReference(
            'tt_content',
            $this->configurationManager->getContentObject()->data[ 'uid' ]
        );

        $this->view->assign('categories', $categories);
    }

    /**
     * Get categories of plugin content element and return the first category
     *
     * @return Category|null
     */
    private function getFirstSelectedCategory(): ?Category
    {
        $category = null;
        try {
            $categories = $this->categoryRepository->getByReference(
                'tt_content',
                $this->configurationManager->getContentObject()->data[ 'uid' ]
            );

            if (! empty($categories)) {
                /**
                 * @noinspection CallableParameterUseCaseInTypeContextInspection
                 * @var $category Category
                 */
                $category = $categories->getFirst();
            }
        } catch (InvalidQueryException $e) {
            // Do not throw anything here when getByReference throws an exception.
            // We just return null instead.
        }

        return $category;
    }

    /**
     * Get header data
     */
    public function blogHeaderAction() : void
    {

        $categories = $this->categoryRepository->findAll();
        for ($i=0; $i < count($categories); $i++){
            if ($categories[$i]->getDescription() == $GLOBALS['TSFE']->id){
                $this->view->assign('active_category', $categories[$i]);
                break;
            }
        }

        $this->view->assign('categories', $categories);
    }
}
