<?php
declare(strict_types = 1);


namespace SG\BlogSgEnhancements\Domain\Repository;


use T3G\AgencyPack\Blog\Constants;
use T3G\AgencyPack\Blog\Domain\Model\Category;
use T3G\AgencyPack\Blog\Domain\Model\Post;
use TYPO3\CMS\Core\Context\Context;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManager;
use TYPO3\CMS\Extbase\Configuration\ConfigurationManagerInterface;
use TYPO3\CMS\Extbase\Persistence\Generic\Qom\ComparisonInterface;
use TYPO3\CMS\Extbase\Persistence\Generic\Typo3QuerySettings;
use TYPO3\CMS\Extbase\Persistence\QueryInterface;
use TYPO3\CMS\Extbase\Utility\DebuggerUtility;

class PostRepository extends \T3G\AgencyPack\Blog\Domain\Repository\PostRepository
{

    /**
     * This method is overridden because we need to set the objectType
     * to the original Post model
     *
     * @inheritDoc
     * @throws \Exception
     */
    public function initializeObject(): void
    {
        // Set the object type to the original Post model
        // because createQuery would otherwise try to find
        // a Post model in this extension
        $this->objectType = Post::class;

        // now call parent method
        parent::initializeObject();
    }

    /**
     * @param Category $category
     * @param int      $limit
     *
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllByCategoryWithLimit(Category $category, int $limit = 1)
    {
        return $this->findAllByCategoryWithLimitAndOffset($category, $limit, 0);
    }

    /**
     * @param Category $category
     * @param int      $limit
     * @param int      $offset
     *
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllByCategoryWithLimitAndOffset(Category $category, int $limit = 0, int $offset = 0)
    {
        $query = $this->createQuery();
        $constraints = $this->defaultConstraints;
        $constraints[] = $query->contains('categories', $category);
        $storagePidConstraint = $this->getStoragePidConstraint();
        if ($storagePidConstraint instanceof ComparisonInterface) {
            $constraints[] = $storagePidConstraint;
        }
        if( $limit > 0 ) {
            $query->setLimit($limit);
        }
        if( $offset > 0 ) {
            $query->setOffset($offset);
        }

        return $query->matching($query->logicalAnd($constraints))->execute();
    }

    /**
     * @param int $limit
     * @param int $offset
     *
     * @return array|\TYPO3\CMS\Extbase\Persistence\QueryResultInterface
     * @throws \TYPO3\CMS\Core\Context\Exception\AspectNotFoundException
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     * @throws \TYPO3\CMS\Extbase\Persistence\Exception\InvalidQueryException
     */
    public function findAllWithLimitAndOffset(int $limit, int $offset)
    {
        $query = $this->getFindAllQuery();
        $query->setLimit($limit);
        $query->setOffset($offset);

        return $query->execute();
    }

    /**
     * Overrides method from PostRepository because we need
     * to get the
     * @return array
     * @throws \TYPO3\CMS\Extbase\Configuration\Exception\InvalidConfigurationTypeException
     */
    protected function getStoragePidsFromTypoScript(): array
    {
        $configurationManager = $this->objectManager->get(ConfigurationManager::class);
        $settings = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_FRAMEWORK);

        if( $settings['persistence']['storagePid'] === '' || $settings['persistence']['storagePid'] === 0 ) {
            $typoscriptSettings = $configurationManager->getConfiguration(ConfigurationManagerInterface::CONFIGURATION_TYPE_SETTINGS);
            if( $typoscriptSettings['storagePid'] !== '' ) {
                return [(int)$typoscriptSettings['storagePid']];
            }
        }
        return GeneralUtility::intExplode(',', $settings['persistence']['storagePid']);
    }

}
